<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Components;

use Core\Model\Event\ControlEvent;
use Core\Model\Parameters;
use Core\Model\UI\BaseControl;
use Core\Model\UI\FrontPresenter;
use Currency\Model\Exchange;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Dao\Category;
use EshopCatalog\FrontModule\Model\Dao\FilterGroup;
use EshopCatalog\FrontModule\Model\Event\FilterEvent;
use EshopCatalog\FrontModule\Model\Event\FilterLinkEvent;
use EshopCatalog\FrontModule\Model\FeatureProducts;
use EshopCatalog\FrontModule\Model\FilterService;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Helpers\ProductsHelper;
use EshopCatalog\Model\Helpers\VariantsHelper;
use Exception;
use Nette\Application\Attributes\Persistent;
use Nette\Application\UI\InvalidLinkException;
use Nette\Http\Request;

class ProductsFilter extends BaseControl
{
	/** @var int[] */
	public array $productIds = [];

	/** @var int[] */
	protected array $productIdsAfterFilter = [];

	protected ?array $resultAfterFilter = null;

	/** @var int[] */
	public array $productIdsNoPriceFilter = [];

	public array   $categories  = [];
	public ?string $searchQuery = null;

	/** @var callable[] */
	public array $onFilter = [];

	protected array     $sortValues      = [];
	public ?array       $allowedFeatures = null;
	protected ?array    $cFilters        = null;
	protected ?array    $cMinMax         = null;
	protected ?array    $filterValues    = null;
	protected ?Category $firstCategory   = null;
	protected bool      $canLoadFilters  = true;

	#[Persistent]
	public ?array $filter = [];

	#[Persistent]
	public array $fr = [];

	#[Persistent]
	public int|string|null $min = null;

	#[Persistent]
	public int|string|null $max = null;

	#[Persistent]
	public ?string $sort = null;

	public function __construct(
		protected FeatureProducts $featureProducts,
		protected Products        $productsService,
		protected FilterService   $filterService,
		protected Categories      $categoriesService,
		protected ProductsFacade  $productsFacade,
		protected Exchange        $exchange,
		protected VariantsHelper  $variantsHelper,
		protected Request         $httpRequest
	)
	{
		$this->sortValues = $this->getSortValues();

		$this->monitor(FrontPresenter::class, function(FrontPresenter $presenter): void {
			$do = $presenter->getParameter('do');

			if (!$do || (!\str_contains($do, 'cartAddForm') && !\str_contains($do, 'productPreview'))) {
				if ($presenter->isAjax()) {
					if ($do && \str_contains($do, 'formSubmit')) {
						$this->formSubmit();
					}
					$this->eventDispatcher->dispatch(new FilterEvent($this->getFilterValues()), ProductsFilter::class . '::onFilter');
					$this->redrawControl('filters');
				}

				$this->eventDispatcher->addListener(ProductsFilter::class . '::onFilter', function(): void {
					$this->redrawControl('filters');
				});
			}
		});
	}

	public function getSortValues(): array
	{
		return [
			'recommended',
			'p.price+ASC',
			'p.price+DESC',
			'p.created+DESC',
		];
	}

	public function loadState(array $params): void
	{
		$sort = $params['sort'] ?? null;

		if ($sort) {
			$sort = str_replace('%2B', '+', $sort);

			if (!in_array($sort, $this->getSortValues(), true)) {
				unset($params['sort']);
				$this->presenter->redirect('this');
			}
		}

		parent::loadState($params);
	}

	public function render(): void
	{
		$this->template->canLoadFilters = $this->canLoadFilters;

		if ($this->canLoadFilters) {
			$filters      = $this->applyFilters();
			$minMax       = $this->getMinMax();
			$filterActive = false;
			foreach ($filters as $group) {
				if ($group->hasActive()) {
					$filterActive = true;
					break;
				}
			}

			if (Config::load('productPreview.basePriceWithoutVat', false)) {
				if (isset($minMax['min'])) {
					$minMax['min'] = round($minMax['min'] / (1 + ((float) $minMax['minVat']) / 100), 2);
				}
				if (isset($minMax['max'])) {
					$minMax['max'] = round($minMax['max'] / (1 + ((float) $minMax['maxVat']) / 100), 2);
				}
			}

			$this->template->filters       = $filters;
			$this->template->minMax        = $minMax;
			$this->template->currentMinMax = [
				'min' => $this->min ?: '',
				'max' => $this->max ?: '',
			];
			$this->template->currentSort   = $this->sort;
			$this->template->sortValues    = $this->getSortValues();

			$this->template->filterActive = $filterActive || $this->min || $this->max || $this->fr !== [];
			$this->template->resetLink    = $this->getResetLink();

			try {
				$tmp = $this->presenter->template->breadcrumb;
				if ($this->searchQuery) {
					$this->template->resetLink = $this->link('this', ['filter' => []]);
				} else if ($tmp) {
					$this->template->resetLink = end($tmp)->link;
				} else {
					$this->template->resetLink = null;
				}
			} catch (Exception) {
				$presenter = $this->presenter;
				$args      = $presenter->getParameters();
				unset($args['activeNavigation']);
				unset($args['action']);
				unset($args['do']);
				$this->template->resetLink = $presenter->link(':' . $presenter->getName() . ':' . $presenter->getAction(), $args);
			}
		} else {
			$this->template->filters         = [];
			$this->template->loadFiltersLink = $this->presenter->link('this', ['loadFilters' => true]);
		}

		$this->template->render($this->getTemplateFile());
	}

	public function getResetLink(): ?string
	{
		try {
			$tmp = $this->getPresenter()->template->breadcrumb;
			if ($this->searchQuery) {
				return $this->link('this', ['filter' => []]);
			} else if ($tmp) {
				return end($tmp)->link;
			}

			return null;
		} catch (Exception $e) {
			$presenter = $this->getPresenter();
			$args      = $presenter->getParameters();
			unset($args['activeNavigation']);
			unset($args['action']);
			unset($args['do']);

			return $presenter->link(':' . $presenter->getName() . ':' . $presenter->getAction(), $args);
		}
	}

	public function getActiveFiltersCount(): int
	{
		$return = 0;

		if ($this->min || $this->max) {
			$return += 1;
		}

		$return += count($this->fr);

		$hasFilter = false;
		foreach ($this->applyFilters() as $v) {
			if ($v->hasActive()) {
				$return += 1;
			}
		}

		return $return;
	}

	public function filtersExist(): bool
	{
		$minMax = $this->getMinMax();

		$hasFilter = false;
		foreach ($this->applyFilters() as $v) {
			if (count($v->items) > 0) {
				$hasFilter = true;
				break;
			}
		}

		return isset($minMax['min'], $minMax['max']) && $hasFilter;
	}

	public function getHtmlFilters(): string
	{
		ob_start();
		$this->render();

		return (string) ob_get_clean();
	}

	public function createLink(string $type, string $key, string $value): string
	{
		$result = null;
		if (Parameters::load('eshopAdvancedFeature.allowVirtualCategories')) {
			$event = new FilterLinkEvent($this->filterValues ?? [], $this->categories, $type, $key, $value);
			$this->eventDispatcher->dispatch($event, self::class . '::createFilterLink');

			if ($event->result) {
				$result = $event->result;
			}
		}

		return $result ?: $this->link('set!', func_get_args());
	}

	public function createSortLink(?string $sort): string
	{
		$result = null;
		if (Parameters::load('eshopAdvancedFeature.allowVirtualCategories')) {
			$event = new FilterLinkEvent($this->filterValues ?? [], $this->categories, 'sort', 'sort', $sort);
			$this->eventDispatcher->dispatch($event, self::class . '::createSortLink');

			if ($event->result) {
				$result = $event->result;
			}
		}

		return $result ?: $this->link('sort!', func_get_args());
	}

	/*******************************************************************************************************************
	 * ============================== Handle
	 */

	/**
	 * @param int|string $t
	 * @param int|string $k
	 * @param int|string $v
	 *
	 * @crossOrigin
	 *
	 * @throws InvalidLinkException
	 */
	public function handleSet($t, $k, $v): void
	{
		if ($t) {
			$this->filter['f' . $t][$k][] = $v;
		}

		if ($this->presenter->isAjax()) {
			$this->eventDispatcher->dispatch(new FilterEvent($this->getFilterValues()), ProductsFilter::class . '::onFilter');
			$this->presenter->payload->url = $this->link('this');
		}
	}

	/**
	 * @param int|string $min
	 * @param int|string $max
	 *
	 * @throws InvalidLinkException
	 */
	public function handlePrice($min, $max): void
	{
		$this->min = $this->getMinMax()['min'] === $min ? null : $min;
		$this->max = $this->getMinMax()['max'] === $max ? null : $max;

		if ($this->presenter->isAjax()) {
			$this->eventDispatcher->dispatch(new FilterEvent($this->getFilterValues()), ProductsFilter::class . '::onFilter');
			$this->presenter->payload->url = $this->link('this');
		}
	}

	public function handleSort(?string $sort): void
	{
		if ($sort === 'recommended') {
			$sort = null;
		}

		$this->sort = $sort;

		if ($this->presenter->isAjax()) {
			$this->eventDispatcher->dispatch(new FilterEvent($this->getFilterValues()), ProductsFilter::class . '::onFilter');
			$this->presenter->payload->url = $this->link('this');
		}
	}

	public function handleFormSubmit(): void
	{
		$this->formSubmit();
		if ($this->presenter->isAjax()) {
			$this->eventDispatcher->dispatch(new FilterEvent($this->getFilterValues()), ProductsFilter::class . '::onFilter');

			if (!$this->presenter->payload->url) {
				$this->presenter->payload->url = $this->link('this');
			}
		}
	}

	protected function formSubmit(): void
	{
		$presenter   = $this->presenter;
		$httpRequest = $presenter->getHttpRequest();
		$this->resetFilter();

		$catIds = $presenter->getParameter('id');
		if ($catIds && is_array($catIds)) {
			$this->filter['c'] = implode('|', $catIds);
		}

		$features = $httpRequest->getPost('feature');
		if ($features) {
			foreach ($features as $g => $vals) {
				if (is_string($vals) && $vals === 'on') {
					continue;
				}
				$this->filter['ff'][$g] = implode('|', $vals);
			}
		}

		foreach ($httpRequest->getPost('range') ?: [] as $k => $v) {
			if ($v['min'] || $v['max']) {
				$this->fr[$k] = implode('|', $v);
			} else {
				unset($this->fr[$k]);
			}
		}

		$manufacturers = $httpRequest->getPost('manu');
		if ($manufacturers) {
			foreach ($manufacturers as $g => $vals) {
				if (is_string($vals) && $vals === 'on') {
					continue;
				}
				$this->filter['fm'][$g] = implode('|', $vals);
			}
		}

		$availabilities = $httpRequest->getPost('av');
		if ($availabilities) {
			foreach ($availabilities as $g => $vals) {
				if (is_string($vals) && $vals === 'on') {
					continue;
				}
				$this->filter['fa'][$g] = implode('|', $vals);
			}
		}

		$tags = $httpRequest->getPost('tags') ?: $httpRequest->getPost('tag');
		if ($tags) {
			foreach ($tags as $g => $vals) {
				if (is_string($vals) && $vals === 'on') {
					continue;
				}
				$this->filter['ft'][$g] = implode('|', $vals);
			}
		}

		$priceRange = $httpRequest->getPost('priceRange');
		if ($priceRange) {
			if (isset($priceRange['min']) && $this->getMinMax()['min'] != $priceRange['min']) {
				$this->min = $priceRange['min'];
			}
			if (isset($priceRange['max']) && $this->getMinMax()['max'] != $priceRange['max']) {
				$this->max = $priceRange['max'];
			}
		}


		if (Parameters::load('eshopAdvancedFeature.allowVirtualCategories')) {
			$event = new FilterLinkEvent($this->getFilterValues() ?? [], $this->categories);
			$this->eventDispatcher->dispatch($event, self::class . '::createFilterLink');

			if ($event->result) {
				$this->presenter->payload->url = $event->result;
			}
		}

		$sort = $httpRequest->getPost('sort');
		if (!$sort) {
			$sort = $httpRequest->getQuery('productsFilter-sort');
		}

		if ($sort) {
			$this->sort = $sort;
		}
	}

	/*******************************************************************************************************************
	 * ============================== Get / Set
	 */

	public function setCategories(array $categories): self
	{
		$this->categories                     = array_map(static fn($c) => $c ? $c->getId() : null, $categories);
		$this->categories                     = array_filter($this->categories, static fn($value): bool => strlen((string) $value) > 0);
		FilterService::$prepareDataCategories = $this->categories;
		$this->productIds                     = $this->productsService->getProductsIdInCategory($this->categories, 0, null, $this->sort);

		if (count($this->categories) === 1) {
			$this->firstCategory = $this->categoriesService->get(array_values($this->categories)[0]);
		}

		$this->loadFilters();

		return $this;
	}

	public function setSearchQuery(string $search): self
	{
		$this->searchQuery = $search;
		$this->productIds  = $this->productsService->getProductsIdBySearch($search, 0, null, $this->sort);
		$this->loadFilters();

		return $this;
	}

	protected function filterPriceCheck(array $activeFilter, array $productData): bool
	{
		$pr    = $activeFilter['priceRange'] ?? null;
		$price = $productData['price'];

		return !$pr
			|| ((!isset($pr['min']) || $price >= $pr['min']) && (!isset($pr['max']) || $price <= $pr['max']));
	}

	protected function filterRangeCheck(array $activeFilter, array $productData): bool
	{
		foreach ($activeFilter['range'] as $k => $v) {
			$df = $productData['df'][$k] ?? null;
			if (!$df) {
				return false;
			}

			if (($v[0] && max($df) < $v[0]) || ($v[1] && min($df) > $v[1])) {
				return false;
			}
		}

		return true;
	}

	protected function loadFilters(): void
	{
		$this->eventDispatcher->dispatch(new ControlEvent($this), ProductsFilter::class . '::beforeLoadFilters');

		$filter = $this->getFilterValues();

		if (
			($filter !== null && $filter !== [])
			|| !$this->firstCategory
			|| !$this->firstCategory->ajaxFilterLoad
			|| $this->httpRequest->getQuery('loadFilters') === '1'
			|| $this->httpRequest->isAjax()
		) {
			$filterNoPrice = $filter;
			unset($filterNoPrice['priceRange']);
			if ($this->categories !== []) {
				$this->productIdsNoPriceFilter = $this->productsService->getProductsIdInCategory($this->categories, null, null, null, $filterNoPrice);
			} else if ($this->searchQuery) {
				$this->productIdsNoPriceFilter = $this->productsService->getProductsIdBySearch($this->searchQuery, null, null, null, $filterNoPrice);
			}

			$this->applyFilters();
		} else {
			$this->canLoadFilters = false;
		}
	}

	/**
	 * @return FilterGroup[]
	 */
	public function applyFilters(): ?array
	{
		if ($this->cFilters) {
			return $this->cFilters;
		}

		if ($this->productIds === []) {
			return [];
		}

		$activeFilters = $this->getFilterValues();
		$filterGroups  = [];

		if (Config::load('productsList.showAvailabilityFilter')) {
			$availabilityGroup        = new FilterGroup;
			$availabilityGroup->type  = 'av';
			$availabilityGroup->id    = 0;
			$availabilityGroup->name  = $this->t('eshopCatalogFront.filter.availability');
			$availabilityGroup->items = $this->filterService->getAvailability($this->productIds);
			$filterGroups['av_0']     = $availabilityGroup;
		}

		$tagGroup              = new FilterGroup;
		$tagGroup->type        = 'tag';
		$tagGroup->id          = 0;
		$tagGroup->name        = $this->t('eshopCatalogFront.filter.tags');
		$tagGroup->items       = $this->filterService->getTags($this->productIds);
		$filterGroups['tag_0'] = $tagGroup;

		$manuGroup              = new FilterGroup;
		$manuGroup->type        = 'manu';
		$manuGroup->id          = 0;
		$manuGroup->name        = $this->t('eshopCatalogFront.filter.manufacturer');
		$manuGroup->items       = $this->filterService->getManufacturers($this->productIds);
		$filterGroups['manu_0'] = $manuGroup;

		$availableFeatures = [];
		if ($this->categories !== []) {
			foreach ($this->categories as $catId) {
				$cat = $this->categoriesService->get((int) $catId);

				if ($cat instanceof Category) {
					foreach ($this->filterService->getCategoryFilters((int) $catId, (bool) $cat->filtersFromParent) as $fea) {
						if (!isset($availableFeatures[$fea->id]) && $fea->name) {
							$availableFeatures[$fea->id] = $fea;
						}
					}
				}
			}
		} else {
			$availableFeatures = $this->filterService->getFeatures($this->productIds);
			uasort($availableFeatures, static fn($a, $b): int => $a->position <=> $b->position);
		}

		$productsFilters = $this->getProductsCountInFilter()['filters'];
		$products        = $this->getProductsCountInFilter()['p'];
		$productsDf      = $this->getProductsCountInFilter()['df'];

		if ($this->allowedFeatures) {
			$tmpF = [];
			foreach ($this->allowedFeatures as $v) {
				if (isset($availableFeatures[$v])) {
					$tmpF[$v] = $availableFeatures[$v];
				}
			}

			$availableFeatures = $tmpF;
			$tmpF              = null;
		}
		foreach ($availableFeatures as $k => $v) {
			if ($v->valueType === Feature::TYPE_CHECK) {
				$filterGroups['feature_' . $k] = $v;
			} else {
				$tmp      = $productsDf[$k] ?? null;
				$groupKey = $v->valueType . '_' . $k;

				$filterGroups[$groupKey] = $v;
				if ($tmp) {
					$filterGroups[$groupKey]->setMinMaxValue($tmp['min'], $tmp['max']);
				}

				$af = $activeFilters[$v->valueType][$k] ?? null;
				if ($af) {
					$filterGroups[$groupKey]->setMinMaxCurrentValue(...$af);
				}
			}
		}

		foreach ($filterGroups as $feature) {
			foreach ($feature->items as $ki => $item) {
				$count = count($productsFilters[$feature->type . '_' . $ki] ?: []);

				if ($count !== 0) {
					$item->productsCountInit = $count;
				} else {
					unset($feature->items[$ki]);
				}
			}
		}

		$prodsForFilter        = [];
		$filteredProducts      = [];
		$filterCountsForActive = [];
		if (isset($activeFilters['tag']) || isset($activeFilters['manu']) || isset($activeFilters['feature']) || isset($activeFilters['av'])) {
			foreach ($activeFilters as $afGroup => $afGroupValue) {
				foreach ($afGroupValue as $afGroupValuesKey => $afValues) {
					$countsForActiveKey = $afGroup . '_' . $afGroupValuesKey;

					if (in_array($afGroup, ['tag', 'manu', 'av'])) {
						$afValues           = $afGroupValue;
						$countsForActiveKey = $afGroup;
					}

					$group = $filterGroups[$afGroup . '_' . $afGroupValuesKey] ?? null;

					foreach ($afValues as $value) {
						if (isset($productsFilters[$afGroup . '_' . $value])) {
							$prodsForFilter[$afGroup . $afGroupValuesKey] = array_merge($prodsForFilter[$afGroup . $afGroupValuesKey] ?? [], $productsFilters[$afGroup . '_' . $value] ?? []);
						}

						if ($group && isset($group->items[$value])) {
							$group->items[$value]->isActive = true;
						}
					}

					$filterCountsForActive[$countsForActiveKey] = $prodsForFilter[$afGroup . $afGroupValuesKey];
				}
			}

			if ($prodsForFilter !== []) {
				$prodsForFilter = count($prodsForFilter) > 1 ? array_intersect(...array_values($prodsForFilter)) : array_values($prodsForFilter)[0];
			}
		} else {
			$prodsForFilter = $this->productIds;
		}

		// Vypocet poctu pro aktivni filtry
		if (isset($activeFilters['tag']) || isset($activeFilters['manu']) || isset($activeFilters['feature']) || isset($activeFilters['av'])) {
			foreach ($activeFilters as $afGroup => $afGroupValue) {
				foreach ($afGroupValue as $afGroupValuesKey => $afValues) {
					$countsForActiveKey = $afGroup . '_' . $afGroupValuesKey;

					if (in_array($afGroup, ['tag', 'manu', 'av'])) {
						$afValues           = $afGroupValue;
						$countsForActiveKey = $afGroup;
					}

					foreach ($afValues as $value) {
						$group = $filterGroups[$afGroup . '_' . $afGroupValuesKey] ?? null;

						if ($group) {
							$tmpProds = $filterCountsForActive;

							// Odebrani skupiny ktera je zrovna aktivni a vybrani porduktu pouze z ostatnich filtru
							unset($tmpProds[$countsForActiveKey]);

							//							$lists = array_map(static fn($v) => is_array($v) ? $v : [], (array)$tmpProds);
							//
							//							$lists = array_filter($lists); // smaže prázdná pole
							//
							//							if ($lists === []) {
							//								$tmpProds = [];
							//							} elseif (count($lists) > 1) {
							//								$tmpProds = array_intersect(...array_values($lists));
							//							} else {
							//								$tmpProds = array_merge(...array_values($lists));
							//							}

							$lists = array_filter((array) $tmpProds);
							if ($lists === []) {
								$tmpProds = [];
							} else {
								$tmpProds = count($lists ?: []) > 1 // @phpstan-ignore-line
									? array_intersect(...array_values($lists ?: [])) // @phpstan-ignore-line
									: array_merge(...array_values($lists ?: [[]])); // @phpstan-ignore-line
							}

							foreach ($group->items as $itemK => $item) {
								if (count($filterCountsForActive) <= 1) {
									$item->productsCount = count($productsFilters[$afGroup . '_' . $itemK] ?? []);
								} else {
									$item->productsCount = count(array_intersect($tmpProds, $productsFilters[$afGroup . '_' . $itemK]));
								}
							}
						}
					}
				}
			}
		}

		if ($prodsForFilter) {
			$prodsForFilter = array_unique($prodsForFilter);
		}

		if (isset($activeFilters['priceRange']) || isset($activeFilters['range'])) {
			foreach ($prodsForFilter as $k => $prodId) {
				$prod = $products[$prodId] ?? null;

				if (!$prod || !$this->filterPriceCheck($activeFilters, $prod) || !$this->filterRangeCheck($activeFilters, $prod)) {
					unset($prodsForFilter[$k]);
					continue;
				}
			}
		}

		if (count($prodsForFilter) === count($this->productIds)) {
			foreach ($filterGroups as $feature) {
				foreach ($feature->items as $ki => $item) {
					$item->productsCount = count($productsFilters[$feature->type . '_' . $ki]);
				}
			}
			$this->productIdsAfterFilter = $this->productIds;
		} else {
			foreach ($prodsForFilter as $prodId) {
				$prod = $products[$prodId] ?? null;
				if (!$prod) {
					continue;
				}

				$filteredProducts[] = $prodId;

				foreach ($prod['filters'] as $k => $v) {
					$group = $filterGroups[$k] ?? null;
					if ($group) {
						if (is_array($v)) {
							foreach ($v as $v2) {
								$item = $group->items[$v2] ?? null;

								if ($item) {
									$item->productsCount++;
								}
							}
						} else {
							$item = $group->items[$v] ?? null;

							if ($item) {
								$item->productsCount++;
							}
						}
					}
				}
			}

			$this->productIdsAfterFilter = array_unique($filteredProducts);
		}

		$this->cFilters = $filterGroups;

		return $this->cFilters;
	}

	public function getMinMax(): ?array
	{
		if (!$this->cMinMax && $this->productIdsNoPriceFilter) {
			$min    = $this->getProductsCountInFilter()['priceMin'];
			$minVat = $this->getProductsCountInFilter()['priceMinVatRate'] ?? 21;
			$max    = $this->getProductsCountInFilter()['priceMax'];
			$maxVat = $this->getProductsCountInFilter()['priceMaxVatRate'] ?? 21;

			$this->cMinMax = [
				'min'    => $this->exchange->change((float) $min),
				'max'    => $this->exchange->change((float) $max),
				'minVat' => $minVat,
				'maxVat' => $maxVat,
			];
		}

		return $this->cMinMax;
	}

	/**
	 * Vrátí ID produktů které odpovídají filtru
	 */
	public function getProductIdsAfterFilter(): array
	{
		if ($this->resultAfterFilter === null) {
			$products = $this->productIdsAfterFilter
				? array_intersect($this->productIds, $this->productIdsAfterFilter)
				: ($this->getFilterValues() ? [] : $this->productIds);

			if (Config::load('groupVariantsInList', false)) {
				$allVariants = $this->variantsHelper->getBasicByProduct();

				$this->resultAfterFilter = ProductsHelper::filterGroupVariantsInList($products, $allVariants);
			} else {
				$this->resultAfterFilter = $products;
			}
		}

		return $this->resultAfterFilter;
	}

	public function getFilterValues(): ?array
	{
		if ($this->filterValues) {
			return $this->filterValues;
		}

		$return = [];

		if (isset($this->filter['c'])) {
			$return['categories'] = explode('|', (string) $this->filter['c']);
		}

		if (isset($this->filter['ff'])) {
			foreach ($this->filter['ff'] as $group => $vals) {
				$return['feature'][$group] = explode('|', (string) $this->filter['ff'][$group]);
			}
		}

		if (isset($this->fr)) {
			foreach (array_keys($this->fr) as $group) {
				$return['range'][$group] = explode('|', (string) $this->fr[$group]);
			}
		}

		if (isset($this->filter['fm'])) {
			foreach ($this->filter['fm'] as $group => $vals) {
				$return['manu'] = explode('|', (string) $this->filter['fm'][$group]);
			}
		}

		if (isset($this->filter['fa'])) {
			foreach ($this->filter['fa'] as $group => $vals) {
				$return['av'] = explode('|', (string) $this->filter['fa'][$group]);
			}
		}

		if (isset($this->filter['ft'])) {
			foreach ($this->filter['ft'] as $group => $vals) {
				$return['tag'] = explode('|', (string) $this->filter['ft'][$group]);
			}
		}

		if ($this->min) {
			$return['priceRange']['min'] = $this->min;
		}

		if ($this->max) {
			$return['priceRange']['max'] = $this->max;
		}

		if (isset($return['priceRange']['min']) && $return['priceRange']['min'] > 0) {
			if (Config::load('productPreview.basePriceWithoutVat', false)) {
				$return['priceRange']['min'] = ((float) $return['priceRange']['min'] / 100) * 121;
			}

			$return['priceRange']['min'] = $this->exchange->change((float) $return['priceRange']['min'], 'default', 'current');
		}

		if (isset($return['priceRange']['max']) && $return['priceRange']['max'] > 0) {
			if (Config::load('productPreview.basePriceWithoutVat', false)) {
				$return['priceRange']['max'] = ((float) $return['priceRange']['max'] / 100) * 121;
			}

			$return['priceRange']['max'] = $this->exchange->change((float) $return['priceRange']['max'], 'default', 'current');
		}

		if ($this->sort) {
			$this->sort     = urldecode($this->sort);
			$this->sort     = str_replace(' ', '+', $this->sort);
			$return['sort'] = $this->sort;
		}

		$this->filterValues = $return;

		return $this->filterValues;
	}

	protected function resetFilter(): void
	{
		$this->filterValues = null;
		$this->filter       = null;
		$this->min          = null;
		$this->max          = null;
		$this->sort         = null;
	}

	protected function getProductsCountInFilter(): ?array
	{
		$key = $this->categories ? 'c' . serialize($this->categories) : $this->searchQuery;

		if (!$key) {
			return [];
		}

		return $this->filterService->getProductsData($this->productIds, $key);
	}

	public function checkDisableIndexing(): bool
	{
		$filters = $this->getFilterValues();

		return $filters !== null && $filters !== [];
	}

	public function checkDisableIndexingAdvanced(): bool
	{
		$filters = $this->getFilterValues();

		if (
			isset($filters['priceRange'])
			|| isset($filters['range'])
			|| isset($filters['tag'])
			|| (isset($filters['manu']) && count($filters['manu']) >= 2)
			|| (isset($filters['av']) && count($filters['av']) >= 2)
		) {
			return true;
		}

		foreach ($filters['feature'] ?? [] as $values) {
			if (count($values) >= 2) {
				return true;
			}
		}

		return false;
	}

	public function getFilters(): ?array
	{
		return $this->applyFilters();
	}
}
