<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\AdminModule\Model\Sites;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Lang\Langs;
use EshopCatalog\Model\Config;
use Navigations\Model\Navigations;
use Nette\Caching\Cache;
use Nette\Caching\Storage;

class CacheService
{
	public Cache $defaultCache;
	public Cache $navigationCache;
	public Cache $productCache;
	public Cache $priceCache;
	public Cache $filterCache;
	public Cache $categoryCache;
	public Cache $documentsCache;
	public Cache $videosCache;
	public Cache $packagesCache;
	public Cache $tagsCache;
	public Cache $manufacturersCache;

	protected static array $cleaned = [];

	public function __construct(
		protected Storage                $cacheStorage,
		protected Langs                  $langs,
		protected Sites                  $sites,
		protected EntityManagerDecorator $em,
	)
	{
		$this->defaultCache       = new Cache($cacheStorage, 'eshopCatalog');
		$this->navigationCache    = new Cache($cacheStorage, Navigations::CACHE_NAMESPACE);
		$this->productCache       = new Cache($cacheStorage, Products::CACHE_NAMESPACE);
		$this->priceCache         = new Cache($cacheStorage, Products::CACHE_PRICE_NAMESPACE);
		$this->filterCache        = new Cache($cacheStorage, FilterService::CACHE_NAMESPACE);
		$this->categoryCache      = new Cache($cacheStorage, Categories::CACHE_NAMESPACE);
		$this->documentsCache     = new Cache($cacheStorage, ProductDocuments::CACHE_NAMESPACE);
		$this->videosCache        = new Cache($cacheStorage, ProductVideos::CACHE_NAMESPACE);
		$this->packagesCache      = new Cache($cacheStorage, Packages::cacheNamespace);
		$this->tagsCache          = new Cache($cacheStorage, Tags::cacheNamespace);
		$this->manufacturersCache = new Cache($cacheStorage, Manufacturers::CACHE_NAMESPACE);

		$this->langs = $langs;
	}

	public function clean(string $type, array $params): void
	{
		if (isset($params[Cache::Tags])) {
			$tags   = $params[Cache::Tags];
			$exists = self::$cleaned[$type] ?? [];

			if ($exists) {
				foreach ($tags as $kTag => $tag) {
					if (in_array($tag, $exists, true)) {
						unset($tags[$kTag]);
					} else {
						self::$cleaned[$type][] = $tag;
						$exists[]               = $tag;
					}
				}
			} else {
				self::$cleaned[$type] = $tags;
			}

			if ($tags) {
				$this->{$type . 'Cache'}->clean([Cache::TAGS => $tags]);
			}
		}
	}

	public function cleanCategories(array $ids = []): void
	{
		$sites = $this->sites->getOptionsForSelect();
		$roots = $this->em->getConnection()->fetchFirstColumn('SELECT id FROM eshop_catalog__category WHERE lvl = 0');

		foreach ($this->langs->getLangs(false) as $lang) {
			$l = $lang->getTag();

			$this->categoryCache->remove('fullUrl_' . $l);
			$this->categoryCache->remove('categoriesQueryPath:' . $l);

			foreach (array_keys($sites) as $site) {
				foreach ($ids as $catId) {
					$this->cacheStorage->remove('eshopCatalog_eshopNav_default_' . $l . '_' . $catId . '_' . $site . '_');
				}
			}

			foreach ($roots as $root) {
				$this->categoryCache->remove('structured_' . $l . '-' . $root);
			}
		}

		foreach ($ids as $catId) {
			$this->categoryCache->remove('categoryFilters_' . $catId);
			$this->categoryCache->remove('categoryFilters_' . $catId . '_deep');
			$this->filterCache->remove('filtersPrepareData_' . md5((string) $catId));
			$this->categoryCache->clean([Cache::Tags => [
				'category/' . $catId,
			]]);
		}

		$this->cacheStorage->clean([Cache::Tags => [
			'eshopNavigation',
			'navigation',
			'categories',
			\EshopCatalog\FrontModule\Model\Categories::CACHE_NAMESPACE,
		]]);
	}

	public function removeDocuments(array $productsIds): void
	{
		foreach ($productsIds as $id) {
			$this->documentsCache->remove('documents/' . $id);
		}
	}

	public function removeVideos(array $productsIds): void
	{
		foreach ($productsIds as $id) {
			$this->videosCache->remove('videos/' . $id);
		}
	}

	public function removePackage(array $packageIds): void
	{
		foreach ($packageIds as $id) {
			$this->videosCache->remove('package/' . $id);
		}
	}

	public function clearFeatures(array $ids = []): void
	{
		foreach ($this->langs->getLangs(false) as $lang) {
			$this->defaultCache->remove('features/' . $lang->getTag());
			$this->defaultCache->remove('featureData/' . $lang->getTag());
		}

		foreach ($this->langs->getLangs(false) as $lang) {
			foreach ($ids as $id) {
				$this->productCache->remove('dynamicFeature/' . $id . '/' . $lang->getTag());;
			}
		}
	}

	public function clearFeaturesValues(): void
	{
		$this->defaultCache->remove('featureIdValues');
		foreach ($this->langs->getLangs(false) as $lang) {
			$this->defaultCache->remove('featureValues_' . $lang->getTag());
			$this->defaultCache->remove('featureValueTexts_' . $lang->getTag());
		}
		$this->defaultCache->clean([
			Cache::Tags => ['featureValues'],
		]);
	}

	public function clearTagsCache(): void
	{
		foreach ($this->langs->getLangs(false) as $lang) {
			$this->tagsCache->remove('allTags/' . $lang->getTag());
		}

		$this->tagsCache->clean([Cache::Tags => [Tags::cacheNamespace]]);
	}

	public function clearManufacturersCache(): void
	{
		foreach ($this->langs->getLangs(false) as $lang) {
			$this->manufacturersCache->remove('manufacturers/' . $lang->getTag());
		}
	}

	public function clearProduct(int $id): void
	{
		foreach ($this->langs->getLangs(false) as $lang) {
			$lang = $lang->getTag();
			$this->defaultCache->remove('productsExtraFields_' . $lang);
			$this->productCache->remove('product/' . $id . '/' . $lang);
			$this->productCache->remove('link/' . $id . '/' . $lang);
		}

		if (Config::load('allowRelatedProducts')) {
			$this->productCache->remove('related/' . $id);
		}

		$this->defaultCache->clean([Cache::TAGS => ['productsByTag']]);

		$this->productCache->remove('variantBasicByProduct');
		$this->priceCache->remove('retailPrice/' . $id);
		$this->productCache->remove('dynamicFeatures/' . $id);
	}
}
