<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Application\AppState;
use Core\Model\Entities\ExtraField;
use Core\Model\Event\DaoEvent;
use Core\Model\Event\Event;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Sites;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\AbstractQuery;
use Doctrine\ORM\Query\Parameter;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Category;
use EshopCatalog\Model\Entities\CategoryRelated;
use EshopOrders\Model\Entities\Customer;
use Gedmo\Tree\Entity\Repository\NestedTreeRepository;
use Navigations\Model\Entities\Navigation;
use Nette\Application\LinkGenerator;
use Nette\Caching\Cache;
use Nette\Utils\Json;
use Symfony\Component\EventDispatcher\EventDispatcher;
use Throwable;
use Tracy\Debugger;

/**
 * @method NestedTreeRepository getEr()
 */
class Categories extends BaseFrontEntityService
{
	public const CACHE_NAMESPACE = 'eshopCatalogCategories';
	public const MODE_ESHOP      = 'eshop';
	public const MODE_CHECKOUT   = 'checkout';

	protected static string $mode              = self::MODE_ESHOP;
	public static bool      $allowGenerateLink = true;
	public static ?string   $overrideLocale    = null;
	protected array         $cLinkGenerator    = [];

	protected $entityClass = Category::class;

	public array $cacheDep = [
		Cache::Tags   => ['categories'],
		Cache::Expire => '1 week',
	];

	protected ?array      $cNavs                           = null;
	protected ?array      $cSiteRoot                       = null;
	protected array       $cStructured                     = [];
	protected array       $cPath                           = [];
	protected array       $cQueryPath                      = [];
	protected array       $cFlat                           = [];
	protected array       $cFullUrl                        = [];
	protected ?array      $cExtraFields                    = null;
	protected array       $cCategoriesLinks                = [];
	protected array       $cCategoriesLinksWithoutEndSlash = [];
	protected ?array      $cCategoriesLinksWithoutDomain   = [];
	public ?LinkGenerator $linkGenerator                   = null;

	public function __construct(
		protected CacheService    $cacheService,
		protected Sites           $sitesService,
		protected EventDispatcher $eventDispatcher,
	)
	{
	}

	public function getRootIdForSite(?string $siteIdent = null): ?int
	{
		if ($siteIdent === null) {
			$siteIdent = $this->sitesService->getCurrentSite()->getIdent();
		}

		return $this->getAllRootIds()[$siteIdent] ?? null;
	}

	public function findInAll(int $catId): ?Dao\Category
	{
		foreach ($this->getAllRootIds() as $rootId) {
			$cat = $this->getCategories($rootId)[$catId] ?? null;

			if ($cat) {
				return $cat;
			}
		}

		return null;
	}

	public function getAllRootIds(): array
	{
		if ($this->cSiteRoot === null) {
			$this->cSiteRoot = $this->cacheService->categoryCache->load('allRootIds', function(&$dep) {
				$dep  = [Cache::Expire => '1 month'];
				$data = [];

				foreach ($this->em->getConnection()->executeQuery("SELECT c.id as id, ct.alias as alias 
							FROM eshop_catalog__category c 
						    INNER JOIN eshop_catalog__category_texts ct ON ct.id = c.id 
							WHERE c.lvl = 0")->iterateAssociative() as $row) {
					$data[$row['alias']] = (int) $row['id'];
				}

				return $data;
			});
		}

		return $this->cSiteRoot;
	}

	/**
	 * @return array<int, int>
	 */
	public function getRootsForIds(): array
	{
		$arr = [];

		foreach ($this->getEr()->createQueryBuilder('c')
			         ->select('c.id, IDENTITY(c.root) as root')
			         ->getQuery()->getArrayResult() as $row) {
			$arr[(int) $row['id']] = (int) $row['root'];
		}

		return $arr;
	}

	public function getStructured(?int $rootId = null, ?string $lang = null): array
	{
		$lang   = $lang ?: $this->translator->getLocale();
		$rootId = $rootId ?: $this->getRootIdForSite();
		$cKey   = $lang . '-' . ($rootId && self::$mode !== self::MODE_CHECKOUT ? $rootId : 'all');

		if (!isset($this->cStructured[$cKey])) {
			$data = $this->cacheService->categoryCache->load('structured_' . $cKey, function(&$dep) use ($rootId, $lang) {
				$dep = [
					Cache::Tags   => [self::CACHE_NAMESPACE],
					Cache::Expire => '1 week',
				];

				$allowRelated = Config::load('allowRelatedCategories', false);
				$fullUrls     = Config::load('allowCategoryFullUrlField', false) ? $this->getCategoryFullUrl(null, $lang) : [];
				/** @var Dao\Category[]|null[] $flat */
				$flat = [];
				/** @var Dao\Category[][] $grouped */
				$grouped = [];

				$related = [];
				foreach ($allowRelated ? $this->em->createQueryBuilder()->select('IDENTITY(cr.category) as id, cr.target, cr.targetKey')
					->from(CategoryRelated::class, 'cr')
					->orderBy('cr.category', 'ASC')->addOrderBy('cr.position', 'ASC')
					->getQuery()->getScalarResult() : [] as $row) {
					$related[$row['id']][] = $row;
				}

				$qb = $this->getEr()->createQueryBuilder('c')
					->addSelect('ct, IDENTITY(c.parent) as parent')
					->join('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
					->andWhere('c.root = :root')
					->setParameters(new ArrayCollection([new Parameter('lang', $lang), new Parameter('root', $rootId)]))
					->addOrderBy('c.lft');

				if (Config::load('category.publishedByLang')) {
					$qb->andWhere('ct.isPublished = 1');
				} else {
					$qb->andWhere('c.isPublished = 1');
				}

				$published = $qb->getQuery()
					->getResult(AbstractQuery::HYDRATE_ARRAY);

				foreach ($published as $v) {
					$cat                = $v[0];
					$cat['texts']       = $cat['categoryTexts'][$lang];
					$cat['parent']      = $v['parent'];
					$cat['extraFields'] = $this->getAllExtraFields()[$cat['id']] ?? [];

					$dao = $this->fillDao($cat);
					if ($fullUrls && $fullUrls[$dao->getId()]) {
						$dao->addAttr('originAlias', $dao->alias);
						$dao->alias = $fullUrls[$dao->getId()];
					}

					if (!$cat['parent']) {
						$cat['parent'] = 0;
					}

					$grouped[$cat['parent']][$cat['id']] = $dao;
					$flat[$cat['id']]                    = &$grouped[$cat['parent']][$cat['id']];
				}

				$createTree = static function(&$list, $parent) use (&$createTree) {
					$tree = [];
					foreach ($parent as $k => $l) {
						if (isset($list[$l->id])) {
							if ($l->lvl >= 1) {
								foreach ($list[$l->id] as $kk => $v) {
									$list[$l->id][$kk]->setParent($parent[$k]);
								}
							}
							$l->setChild($createTree($list, $list[$l->id]));
						}

						$tree[] = $l;
					}

					return $tree;
				};

				$dao     = (new Dao\Category);
				$dao->id = $rootId;
				$dao->setParentId(0);

				$grouped[0][] = $dao;

				/** @var Dao\Category[] $tree */
				$tree = $createTree($grouped, $grouped[0])[0]->getChild();

				$related = [];
				if ($allowRelated) {
					foreach ($this->em->createQueryBuilder()->select('IDENTITY(cr.category) as id, cr.target, cr.targetKey')
						         ->from(CategoryRelated::class, 'cr')
						         ->orderBy('cr.category', 'ASC')->addOrderBy('cr.position', 'ASC')
						         ->getQuery()->getScalarResult() as $row) {
						$related[$row['id']][] = $row;
					}

					$event = new Event([
						'related' => &$related,
						'lang'    => $lang,
					]);
					$this->eventDispatcher->dispatch($event, self::class . '::loadRelated');
				}

				$domain        = $this->sitesService->getCurrentSite()->getCurrentDomain();
				$linkGenerator = $domain && $this->linkGenerator ? $this->getLinkGenerator($domain->getDomain()) : null;

				foreach ($flat as &$v) {
					if (!$v) {
						continue;
					}

					$v->link = $linkGenerator ? $this->generateLink($v->id, $linkGenerator) : null;

					if (($related[$v->id] ?? []) !== []) {
						foreach ($related[$v->id] as $rel) {
							if (isset($rel['targetEntity'])) {
								$v->related[$rel['target']] = $rel['targetEntity'];
							} else if (isset($flat[$rel['target']])) {
								$v->related[$rel['target']] = &$flat[$rel['target']];
							}
						}
					}
				}

				return [
					'flat' => $flat,
					'tree' => $tree,
				];
			});

			$this->cFlat[$cKey]       = $data['flat'];
			$this->cStructured[$cKey] = $data['tree'];
		}

		return $this->cStructured[$cKey];
	}

	/**
	 * @return Dao\Category[]
	 */
	public function getCategories(?int $rootId = null, ?string $lang = null): array
	{
		$lang = $lang ?: self::$overrideLocale ?: $this->translator->getLocale();

		if (!$rootId && self::$mode === self::MODE_ESHOP) {
			$rootId = $this->getRootIdForSite();
		}

		$cKey = $lang . '-' . ($rootId ?: 'all');

		if ($this->cFlat[$cKey] === null) {
			$this->getStructured($rootId, $lang);
		}

		return $this->cFlat[$cKey] ?? [];
	}

	protected function getLinkGenerator(string $domain): ?LinkGenerator
	{
		if (!isset($this->cLinkGenerator[$domain])) {
			$this->cLinkGenerator[$domain] = $this->linkGenerator?->withReferenceUrl('https://' . $domain);
		}

		return $this->cLinkGenerator[$domain];
	}

	protected function generateLink(int $id, LinkGenerator $linkGenerator): ?string
	{
		$link = $linkGenerator->link('EshopCatalog:Front:Default:category', ['id' => $id]);

		if (Config::load('category.logLinkCreating')) {
			Debugger::log(Json::encode([
				'link'      => $link,
				'id'        => $id,
				'http_host' => $_SERVER['HTTP_HOST'],
				'siteIdent' => $this->sitesService->getCurrentSite()->getIdent(),
				'siteLang'  => $this->sitesService->getCurrentSite()->currentLang,
				'url'       => $_SERVER['REQUEST_URI'],
				'appState'  => AppState::getState('eshopCatalogLinkTest'),
			]), 'eshop-catalog_category-link-' . $this->sitesService->getCurrentSite()->getIdent() . '-' . (new \DateTime())->format('Y-m-d'));
		}

		return $link;
	}

	/**
	 * @return Dao\Category[]
	 */
	public function getPath(Dao\Category $category): array
	{
		if (!$this->cPath[$category->id]) {
			$path = [$category];
			$p    = $category->getParent();

			while ($p) {
				$path[] = $p;
				$p      = $p->getParent();
			}
			$this->cPath[$category->id] = array_reverse($path);
		}

		return $this->cPath[$category->id];
	}

	public function getQueryPath(int $id): array
	{
		$lang = $this->translator->getLocale();
		$key  = 'path/' . $id . '/' . $lang;

		if (!isset($this->cQueryPath[$lang])) {
			$this->cQueryPath[$lang] = $this->cacheService->categoryCache->load('categoriesQueryPath:' . $lang, function(&$dep) use ($lang) {
				$dep    = [Cache::TAGS => [self::CACHE_NAMESPACE], Cache::EXPIRE => '1 week'];
				$result = [];

				foreach ($this->getEr()->createQueryBuilder('c')
					         ->select('c.id, IDENTITY(c.parent) as parent, ct.alias')
					         ->leftJoin('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
					         ->setParameter('lang', $lang)
					         ->groupBy('c.id')
					         ->getQuery()
					         ->getArrayResult() as $row) {
					$result[$row['id']] = $row;
				}

				return $result;
			});
		}

		if (!$this->cQueryPath[$key]) {
			$flat = $this->cQueryPath[$lang];
			$path = [];
			$i    = 0;
			$cat  = $flat[$id];

			while ($cat && $i < 20) {
				$path[$cat['id']] = $cat;
				$cat              = $flat[$cat['parent']] ?? null;
				$i++;
			}

			$this->cQueryPath[$key] = array_reverse($path, true);
		}

		return $this->cQueryPath[$key];
	}

	public function get(int $id): ?Dao\Category
	{
		return $this->getCategories()[$id] ?? null;
	}

	/**
	 * @return Dao\Category[]
	 */
	public function findByAlias(string $alias): array
	{
		$result = [];

		foreach ($this->getCategories() as $c) {
			if ($c->alias == $alias) {
				$result[] = $c;
			}
		}

		return $result;
	}

	public function findNavigationId(int $categoryId): ?int
	{
		if ($this->cNavs === null) {
			$this->cNavs = $this->cacheService->categoryCache->load('navigationsById', function(&$dep) {
				$dep = [
					Cache::Tags       => ['eshopNavigation'],
					Cache::EXPIRATION => '1 week',
				];

				$arr = [];
				foreach ($this->em->getRepository(Navigation::class)->createQueryBuilder('n')
					         ->select('n.id, n.componentParams')
					         ->where('n.componentType = :type')->setParameter('type', 'eshopCatalog.navigation.category')
					         ->getQuery()->getArrayResult() as $row) {
					$arr[$row['componentParams']['category']] = (int) $row['id'];
				}

				return $arr;
			});
		}

		return $this->cNavs[$categoryId] ?? null;
	}

	/**
	 * @return Dao\Category[]
	 */
	public function getBreadcrumb(Dao\Category $category, DaoNavigationItem $navigation): array
	{
		$path = $this->getPath($category);

		$path[0]->name = $navigation->title;

		return $path;
	}

	/**
	 * @return array|string|null
	 * @throws Throwable
	 */
	public function getCategoryFullUrl(?int $catId, string $lang)
	{
		if (!array_key_exists($lang, $this->cFullUrl)) {
			$this->cFullUrl[$lang] = [];

			$cats = $this->cacheService->categoryCache->load('fullUrl_' . $lang, function(&$dep) use ($lang) {
				$dep = [Cache::TAGS => ['categories'], Cache::EXPIRE => '1 week'];

				return $this->em->getRepository(ExtraField::class)->createQueryBuilder('ef')
					->select('ef.sectionKey, ef.value')
					->where('ef.sectionName = :sn')
					->andWhere('ef.key = :key')
					->andWhere('ef.lang = :lang')
					->setParameters(new ArrayCollection([
						new Parameter('sn', Category::EXTRA_FIELD_SECTION),
						new Parameter('key', 'fullUrl'),
						new Parameter('lang', $lang),
					]))
					->getQuery()->getArrayResult();
			});

			foreach ($cats as $row) {
				$this->cFullUrl[$lang][(int) $row['sectionKey']] = $row['value'];
			}
		}

		return $catId ? $this->cFullUrl[$lang][$catId] ?? null : $this->cFullUrl[$lang];
	}

	public function getAllExtraFields(): array
	{
		if ($this->cExtraFields === null) {
			$lang               = $this->translator->getLocale();
			$this->cExtraFields = [];

			$query = $this->em->getRepository(ExtraField::class)->createQueryBuilder('ef')
				->select('ef.sectionKey, ef.key, ef.value')
				->where('ef.sectionName = :secName')
				->andWhere('ef.lang IS NULL OR ef.lang = :lang')
				->setParameters(new ArrayCollection([
					new Parameter('lang', $lang),
					new Parameter('secName', Category::EXTRA_FIELD_SECTION),
				]))->getQuery();

			foreach ($query->getScalarResult() as $row) {
				$this->cExtraFields[$row['sectionKey']][$row['key']] = $row['value'];
			}
		}

		return $this->cExtraFields;
	}

	protected function fillDao(array $category): Dao\Category
	{
		$c                      = new Dao\Category;
		$c->id                  = (int) $category['id'];
		$c->name                = $category['texts']['name'];
		$c->nameH1              = $category['texts']['nameH1'];
		$c->alias               = $category['texts']['alias'];
		$c->shortDescription    = $category['texts']['shortDescription'];
		$c->description         = (string) $category['texts']['description'];
		$c->image               = $category['texts']['image'] ?: $category['image'];
		$c->hideInMobileMenu    = (bool) $category['texts']['hideInMobileMenu'];
		$c->hideInDesktopMenu   = (bool) $category['texts']['hideInDesktopMenu'];
		$c->defaultImage        = $category['image'];
		$c->lvl                 = (int) $category['lvl'];
		$c->created             = $category['created'];
		$c->modified            = $category['modified'] ?: $category['created'];
		$c->filtersFromParent   = (int) $category['filtersFromParent'];
		$c->rod                 = $category['rod'];
		$c->activeProducts      = $category['hasActiveProducts'] !== null ? (int) $category['hasActiveProducts'] : null;
		$c->disableRegisterSale = (int) $category['disableRegisterSale'];

		$c->setSafetyWarningText($category['texts']['safetyWarningText']);
		$c->setSafetyWarningImage($category['texts']['safetyWarningImage']);
		$c->setSeo($category['texts']['seo']);
		$c->setAttrs($category['attrs'] ?: []);
		$c->setExtraFields($category['extraFields']);

		$tmp = [];
		foreach (explode(',', (string) $category['customerGroupRestriction']) as $v) {
			if (is_numeric($v)) {
				$tmp[] = (int) $v;
			}
		}
		$c->allowedCustomerGroups = $tmp;

		$c->canProductsAddToCart = (bool) $category['canProductsAddToCart'];
		$c->disablePickupPoints  = (bool) $category['disablePickupPoints'];
		if ($category['canProductsAddToCart'] === null) {
			$c->canProductsAddToCart = true;
		}

		if ($category['parent']) {
			$c->setParentId((int) $category['parent']);
		}

		if (Config::load('enableEmptyCategoryCustomText')) {
			$c->emptyText = $category['texts']['emptyText'];
		}

		if (Config::load('productsList.allowAjaxFilterLoad')) {
			$c->ajaxFilterLoad = (bool) $category['ajaxFilterLoad'];
		}

		$this->eventDispatcher->dispatch(new DaoEvent($c, $category), Categories::class . '::fillDao');

		return $c;
	}

	public static function setMode(string $mode): void
	{
		self::$mode = $mode;
		Products::setMode(Products::MODE_CHECKOUT);
	}

	public function getAllCategoriesLinks(string $siteIdent, string $lang): array
	{
		$arrayKey = $siteIdent . '_' . $lang;
		if (!array_key_exists($arrayKey, $this->cCategoriesLinks)) {
			$this->cCategoriesLinks[$arrayKey] = [];
			$catRoot                           = $this->getRootIdForSite($siteIdent);

			foreach ($this->getCategories($catRoot, $lang) as $cat) {
				$tmp = explode('/', (string) $cat->link, 4);

				if (isset($tmp[3])) {
					$this->cCategoriesLinks[$arrayKey][$cat->getId()] = '/' . $tmp[3];
				}
			}
		}

		return $this->cCategoriesLinks[$arrayKey];
	}

	public function getAllCategoriesLinksWithoutEndSlash(string $siteIdent, string $lang): array
	{
		$arrayKey = $siteIdent . '_' . $lang;
		if (!array_key_exists($arrayKey, $this->cCategoriesLinksWithoutEndSlash)) {
			$this->cCategoriesLinksWithoutEndSlash[$arrayKey] = [];

			foreach ($this->getAllCategoriesLinks($siteIdent, $lang) as $id => $link) {
				$this->cCategoriesLinksWithoutEndSlash[$arrayKey][$id] = rtrim($link, '/');
			}
		}

		return $this->cCategoriesLinksWithoutEndSlash[$arrayKey];
	}

	public function getCategoriesPath(?int $rootId = null): array
	{
		$result     = [];
		$allRootIds = $this->getAllRootIds();

		foreach ($allRootIds as $rootAlias => $root) {
			if ($rootId && $rootId !== $root) {
				continue;
			}

			$prefix = count($allRootIds) > 1 && !$rootId ? $rootAlias . ' > ' : '';

			foreach ($this->getCategories($root) as $cat) {
				if ($cat->name && $cat->name !== $rootAlias) {
					$path                  = $cat->getParentPathStringFlipped();
					$result[$cat->getId()] = $prefix . ($path ? $path . ' > ' : '') . $cat->name;
				}
			}
		}

		return $result;
	}

	public function checkCategoryRestrictionAccess(array $allowedGroups, ?Customer $customer): bool
	{
		if (!Config::load('allowCategoryCustomerGroupRestriction')) {
			return true;
		}

		if ($allowedGroups === []) {
			$customerGroup = $customer instanceof Customer ? $customer->getGroupCustomers() : null;

			return !($customerGroup && $customerGroup->accessOnlyToAllowedCategories);
		}

		if (!$customer || !$customer->getGroupCustomers()) {
			return false;
		}

		return Arrays::contains($allowedGroups, $customer->getGroupCustomers()->getId());
	}
}
