<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model\Dao;

use Core\Model\Dao\Traits\ExtraFields;
use Core\Model\Entities\TSeo;
use DateTimeInterface;

class Category
{
	use TSeo;
	use ExtraFields;

	public ?int    $id                    = null;
	public ?string $name                  = null;
	public ?string $nameH1                = null;
	public ?string $alias                 = null;
	public ?string $shortDescription      = null;
	public ?string $description           = null;
	public ?string $emptyText             = null;
	public ?string $image                 = null;
	public ?string $defaultImage          = null;
	public ?int    $lvl                   = null;
	public bool    $ajaxFilterLoad        = false;
	public array   $allowedCustomerGroups = [];

	protected ?string $safetyWarningImage = null;
	protected ?string $safetyWarningText  = null;

	public bool $hideInMobileMenu  = false;
	public bool $hideInDesktopMenu = false;

	/** @var self[] */
	public array $child = [];

	public ?int  $parentId = null;
	public ?self $parent   = null;

	public ?string            $link                 = '#';
	public ?DateTimeInterface $created              = null;
	public ?DateTimeInterface $modified             = null;
	public int                $filtersFromParent    = 0;
	public array              $attrs                = [];
	public ?string            $rod                  = null;
	public bool               $canProductsAddToCart = true;
	public bool               $disablePickupPoints  = false;
	public ?int               $activeProducts       = null;
	public int                $disableRegisterSale  = 0;

	/** @var self[] */
	public array $related = [];

	public function getId(): int { return $this->id; }

	public function getNameH1(): string { return $this->nameH1 ?: $this->name; }

	public function setParent(Category $category): self
	{
		$this->parent = &$category;

		return $this;
	}

	public function getParent(): ?Category { return $this->parent; }

	/**
	 * @return self[]
	 */
	public function getParentPath(): array
	{
		$path = [];

		$parent = $this->getParent();
		while ($parent) {
			$path[$parent->getId()] = $parent;
			$parent                 = $parent->getParent();
		}

		return $path;
	}

	public function getParentPathString(): string
	{
		return implode(' > ', array_map(static fn($cat): ?string => $cat->name, $this->getParentPath()));
	}

	public function getParentPathStringFlipped(): string
	{
		return implode(' > ', array_map(static fn($cat): ?string => $cat->name, array_reverse($this->getParentPath())));
	}

	public function setParentId(int $id): self
	{
		$this->parentId = $id;

		return $this;
	}

	public function getParentId(): ?int { return $this->parentId; }

	public function setChild(array $arr): self
	{
		$this->child = $arr;

		return $this;
	}

	public function getRoot(): ?Category
	{
		$r = $this->getParent();

		while ($r && $r->getParent()) {
			$r = $r->getParent();
		}

		return $r;
	}

	public function getRootId(): ?int
	{
		$cat = $this;
		while ($cat) {
			if (!$cat->getParent() instanceof \EshopCatalog\FrontModule\Model\Dao\Category) {
				return $cat->getParentId();
			}
			$cat = $cat->getParent();
		}

		/** @phpstan-ignore-next-line */
		return $this->getParentId();
	}

	/**
	 * @return Category[]
	 */
	public function getChild(): array { return $this->child ?: []; }

	public function addAttr(string $key, mixed $value): self
	{
		$this->attrs[$key] = $value;

		return $this;
	}

	public function setAttrs(array $attrs): self
	{
		$this->attrs = $attrs;

		return $this;
	}

	/**
	 * @return mixed|null
	 */
	public function getAttr(string $key) { return $this->attrs[$key] ?? null; }

	public function getAttrs(): array { return $this->attrs ?: []; }

	public function hasDisabledPickupPoints(): bool
	{
		if ($this->disablePickupPoints) {
			return true;
		}

		$parent = $this->getParent();
		while ($parent) {
			if ($parent->disablePickupPoints) {
				return true;
			}

			$parent = $parent->getParent();
		}

		return false;
	}

	public function setSafetyWarningImage(?string $image): void { $this->safetyWarningImage = $image; }

	public function getSafetyWarningImage(): ?string
	{
		$result = $this->safetyWarningImage;

		if (!$result) {
			$parent = $this->getParent();
			while ($parent && !$result) {
				$result = $parent->safetyWarningImage;

				$parent = $parent->getParent();
			}
		}

		return $result;
	}

	public function setSafetyWarningText(?string $text): void { $this->safetyWarningText = $text; }

	public function getSafetyWarningText(): ?string
	{
		$result = $this->safetyWarningText;

		if (!$result) {
			$parent = $this->getParent();
			while ($parent && !$result) {
				$result = $parent->safetyWarningText;

				$parent = $parent->getParent();
			}
		}

		return $result;
	}
}
