<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Sites;
use Currency\Model\Exchange;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\Query\Parameter;
use EshopCatalog\FrontModule\Model\Dao\Availability;
use EshopCatalog\FrontModule\Model\Dao\Category;
use EshopCatalog\FrontModule\Model\Dao\FilterGroup;
use EshopCatalog\FrontModule\Model\Dao\FilterItem;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\CategoryFilter;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureProduct;
use EshopCatalog\Model\Entities\Product;
use Exception;
use Nette\Caching\Cache;
use Nette\SmartObject;
use Nette\Utils\DateTime;

/**
 * TODO misto hledání podle productsId používat třeba kategorie
 */
class FilterService
{
	use SmartObject;

	public static array $prepareDataCategories = [];
	public static array $cacheTags             = [];

	public const CACHE_NAMESPACE = 'eshopCatalogFilter';

	protected ?array $cProductsData = null;
	protected array  $cacheDep      = [
		Cache::Tags   => ['filters'],
		Cache::EXPIRE => '15 minutes',
	];

	protected ?array $cFeatureProducts = null;
	protected ?array $cProducts        = null;
	protected array  $cCategoryFilters = [];

	public function __construct(
		protected EntityManagerDecorator $em,
		protected CacheService           $cacheService,
		protected Manufacturers          $manufacturers,
		protected Sites                  $sites,
		protected Features               $featuresService,
		protected Config                 $config,
		protected Tags                   $tags,
		protected Exchange               $exchange,
		protected Categories             $categories,
		protected AvailabilityService    $availabilityService,
		protected Translator             $translator,
	)
	{
	}

	/**
	 * @param int[] $productIds
	 */
	public function getFeatures(array $productIds = []): array
	{
		$groups = [];

		$sql = "SELECT fv.feature_id as featureId, fp.id_feature_value as valueId
				FROM eshop_catalog__feature_product fp
				INNER JOIN eshop_catalog__feature_value fv ON fv.id = fp.id_feature_value AND fv.is_published = 1";

		if ($productIds !== []) {
			$sql .= ' WHERE fp.id_product IN (' . implode(',', array_values($productIds)) . ')';
		}

		$sql .= ' ORDER BY fv.position ASC';

		$stmt = $this->em->getConnection()->executeQuery($sql);

		foreach ($stmt->fetchAllAssociative() as $row) {
			if (!isset($groups[$row['featureId']])) {
				$tmp = $this->featuresService->getFeatureById($row['featureId']);

				if (!$tmp['useAsFilter'] || !$tmp['isPublished']) {
					continue;
				}

				if ($tmp && $tmp['name']) {
					$tmpV              = new FilterGroup;
					$tmpV->id          = $tmp['id'];
					$tmpV->name        = $tmp['name'];
					$tmpV->type        = 'feature';
					$tmpV->valueType   = $tmp['type'];
					$tmpV->unit        = $tmp['unit'] ? (string) $tmp['unit'] : null;
					$tmpV->position    = (int) $tmp['position'];
					$tmpV->seoNoFollow = (int) $tmp['seoNoFollow'];

					$groups[$row['featureId']] = $tmpV;
				}
			}

			$g = $groups[$row['featureId']];

			if (!isset($g->items[$row['valueId']])) {
				$tmp = $this->featuresService->getFeatureValueById($row['valueId']);

				if ($tmp && $tmp['name']) {
					$tmpV              = new FilterItem('featureValue', $tmp['id'], (string) $tmp['name']);
					$tmpV->rodM        = $tmp['rodM'];
					$tmpV->rodZ        = $tmp['rodZ'];
					$tmpV->rodS        = $tmp['rodS'];
					$tmpV->seoNoFollow = $tmp['seoNoFollow'];

					$g->items[$row['valueId']] = $tmpV;
				}
			}
		}

		uksort($groups, static fn($a, $b): int => $groups[$a]->position <=> $groups[$b]->position);

		return $groups;
	}

	/**
	 * @param int[] $productIds
	 *
	 * @return FilterItem[]
	 */
	public function getManufacturers($productIds): array
	{
		$toUppercase = $this->config->get('filterManufacturersUppercase', false);
		$data        = [];

		$tmp = array_intersect_key($this->manufacturers->getManufacturers(), $this->getProductsData($productIds)['m'] ?? []);
		uasort($tmp, static fn($a, $b): int => $a->name <=> $b->name);

		foreach ($tmp as $m) {
			$data[$m->id] = new FilterItem('manufacturer', $m->id, $toUppercase ? mb_strtoupper($m->name) : $m->name);
		}

		return $data;
	}

	/**
	 * @param int[] $productIds
	 *
	 * @return FilterItem[]
	 */
	public function getAvailability(array $productIds): array
	{
		$data = [];

		$tmp = array_intersect_key($this->availabilityService->getAll(), $this->getProductsData($productIds)['av'] ?? []);
		uasort($tmp, static fn(Availability $a, Availability $b): int => $a->position <=> $b->position);

		foreach ($tmp as $av) {
			if (!$av->showInFilter) {
				continue;
			}

			$data[$av->getId()] = new FilterItem('availability', $av->getId(), $av->getName());
		}

		return $data;
	}

	/**
	 * @param int[] $productIds
	 *
	 * @return FilterItem[]
	 */
	public function getTags(array $productIds): array
	{
		$data = [];

		$tmp = array_intersect_key($this->tags->getAll(), $this->getProductsData($productIds)['t'] ?? []);
		uasort($tmp, static fn($a, $b): int => $a->name <=> $b->name);

		foreach ($tmp as $tag) {
			$data[$tag->id] = new FilterItem('tag', $tag->id, $tag->name);

			$data[$tag->id]->customData['tagColor'] = [
				'text' => $tag->textColor,
				'bg'   => $tag->bgColor,
			];
		}

		return $data;
	}

	/**
	 * @return FilterGroup[]
	 */
	public function getCategoryFilters(int $id, bool $deep = true): array
	{
		$key = 'categoryFilters_' . $id . ($deep ? '_deep' : '');

		if (array_key_exists($key, $this->cCategoryFilters)) {
			return $this->cCategoryFilters[$key];
		}

		$data = [];

		$features = $this->cacheService->categoryCache->load($key, function(&$dep) use ($id, $deep) {
			$dep                = $this->cacheDep;
			$dep[Cache::Expire] = '1 week';
			$dep[Cache::Tags][] = 'category/' . $id;

			$find = fn(int $id) => $this->em->getRepository(CategoryFilter::class)->createQueryBuilder('f')
				->select('IDENTITY(f.feature) as featureId, f.isHidden')
				->andWhere('f.category = :category')->setParameters(new ArrayCollection([new Parameter('category', $id)]))
				->orderBy('f.position')
				->getQuery()->getArrayResult();

			$features = $find($id);

			if (!$features && $deep) {
				try {
					$stmt = $this->em->getConnection()->prepare("
						SELECT T2.id 
						FROM (
							SELECT @r AS _id,
								(SELECT @r := parent_id FROM eshop_catalog__category WHERE id = _id) AS parent_id, @l := @l + 1 AS lvl 
								FROM (SELECT @r := {$id}, @l := 0) vars, eshop_catalog__category h WHERE @r <> 0) T1 
						JOIN eshop_catalog__category T2 ON T1._id = T2.id ORDER BY T1.lvl DESC
						");
					$tmp  = $stmt->executeQuery()->fetchAllAssociative();
				} catch (Exception) {
					$tmp = [];
				}
				array_shift($tmp);
				array_pop($tmp);
				$categories = [];
				foreach ($tmp as $row) {
					$dep[Cache::Tags][] = 'category/' . $row['id'];
					$categories[]       = $row['id'];
				}

				/** @phpstan-ignore-next-line */
				while ($categories !== [] && empty($features)) {
					$id       = array_pop($categories);
					$features = $find((int) $id);

					if ($features) {
						break;
					}
				}
			}

			return $features;
		});

		if (!$features) {
			return $data;
		}

		foreach ($features as $featureV) {
			$featureId = $featureV['featureId'];
			$tmp       = $this->featuresService->getFeatureById($featureId);

			$group              = new FilterGroup;
			$group->id          = (int) $tmp['id'];
			$group->name        = $tmp['name'] ? (string) $tmp['name'] : null;
			$group->type        = 'feature';
			$group->valueType   = $tmp['type'];
			$group->unit        = $tmp['unit'] ? (string) $tmp['unit'] : null;
			$group->position    = (int) $tmp['position'];
			$group->decimals    = (int) $tmp['decimals'] ?: 0;
			$group->isHidden    = (int) $featureV['isHidden'];
			$group->seoNoFollow = (int) $tmp['seoNoFollow'];

			$data[$featureId] = $group;

			if ($group->valueType !== Feature::TYPE_RANGE) {
				foreach ($this->featuresService->getValuesByFeatureId((int) $featureId) as $tmp) {
					$item              = new FilterItem('featureValue', (int) $tmp['id'], (string) $tmp['name']);
					$item->seoNoFollow = $tmp['seoNoFollow'];

					if ($tmp['valueType']) {
						$item->valueType = $tmp['valueType'];
					}

					if ($item->valueType === Feature::VALUE_TYPE_COLOR) {
						$item->customData['colors']      = $tmp['extraFields']['colors'];
						$item->customData['colorsStyle'] = $tmp['extraFields']['colorsStyle'];
					}

					$data[$featureId]->items[$tmp['id']] = $item;
				}
			}
		}

		$this->cCategoryFilters[$key] = $data;

		return $this->cCategoryFilters[$key];
	}

	protected function buildBaseData(array &$data, array $v, array $vatRates): void
	{
		$df = [];
		if ($v['dynamicFeatures']) {
			foreach (explode(';', (string) $v['dynamicFeatures']) as $kv) {
				$kv = explode('|', $kv);
				if (count($kv) === 2) {
					$df[(int) $kv[0]][] = (float) $kv[1];
				}
			}
		}

		$id      = (int) $v['id'];
		$price   = (float) $v['price'];
		$vatRate = (int) $vatRates[$v['vatRate']];
		$tags    = $v['tags'] ? array_map(static fn($v2): int => (int) $v2, explode(',', (string) $v['tags'])) : [];
		$manu    = $v['m'] ? (int) $v['m'] : null;
		$av      = $v['av'] ? (int) $v['av'] : null;

		$data['p'][$id] = [
			'id'      => $id,
			'price'   => $price,
			'vatRate' => $vatRate,
			'av'      => $v['av'],
			'df'      => $df,
		];

		foreach ($tags as $t) {
			$data['t'][$t][]                    = $id;
			$data['filters']['tag_' . $t][]     = $id;
			$data['p'][$id]['filters']['tag_0'] = $t;
		}

		$data['av'][$v['av']][$id]            = $id;
		$data['filters']['av_' . (int) $av][] = $id;
		$data['p'][$id]['filters']['av_0']    = $av;

		$data['m'][(int) $manu][$id]              = $id;
		$data['filters']['manu_' . (int) $manu][] = $id;
		$data['p'][$id]['filters']['manu_0']      = $manu;

		if (!isset($data['priceMin']) || $data['priceMin'] > $price) {
			$data['priceMin']        = $price;
			$data['priceMinVatRate'] = $vatRate;
		}
		if (!isset($data['priceMax']) || $data['priceMax'] < $price) {
			$data['priceMax']        = $price;
			$data['priceMaxVatRate'] = $vatRate;
		}

		foreach ($df as $dfK => $dfV) {
			foreach ($dfV as $dfV2) {
				if (!isset($data['df'][$dfK]['min']) || $data['df'][$dfK]['min'] > $dfV2) {
					$data['df'][$dfK]['min'] = $dfV2;
				}
				if (!isset($data['df'][$dfK]['max']) || $data['df'][$dfK]['max'] < $dfV2) {
					$data['df'][$dfK]['max'] = $dfV2;
				}
			}
		}
	}

	protected function buildFeaturesData(array &$data, int $featureValue, int $productId, array $featureValuesParent): void
	{
		$featureId = $featureValuesParent[$featureValue] ?? null;
		if ($featureId) {
			$data['filters']['feature_' . $featureValue][]             = $productId;
			$data['p'][$productId]['filters']['feature_' . $featureId] = $featureValue;
		}
	}

	protected function prepareData(array $productsIds): array
	{
		$site = $this->sites->getCurrentSite();

		$catIds = self::$prepareDataCategories === [] ? null : implode(',', self::$prepareDataCategories);

		if ($catIds) {
			$cacheKey = 'filtersPrepareData_' . md5($catIds) . '_' . md5(implode(',', array_keys($productsIds)));

			$result = $this->cacheService->filterCache->load($cacheKey);

			if ($result) {
				return $result;
			}
		}

		$data = [];
		$now  = (new DateTime)->format('Y-m-d H:i:00');
		$conn = $this->em->getConnection();

		$allowedFilters = [];
		foreach (self::$prepareDataCategories as $catId) {
			$cat = $this->categories->get((int) $catId);

			if ($cat instanceof Category) {
				foreach ($this->getCategoryFilters((int) $catId, (bool) $cat->filtersFromParent) as $filter) {
					$allowedFilters[] = $filter->id;
				}
			}
		}

		$avs = [];
		if ($catIds) {
			foreach ($this->availabilityService->getAll() as $av) {
				if ($av->canShowOnList()) {
					$avs[$av->getId()] = $av->getId();
				}
			}
		} else {
			foreach ($this->availabilityService->getAll() as $av) {
				if ($av->canShowOnSearch()) {
					$avs[$av->getId()] = $av->getId();
				}
			}
		}

		$featureValuesParent = [];
		foreach ($conn->executeQuery("SELECT feature_id as feature, id 
					FROM eshop_catalog__feature_value
					" . ($allowedFilters === [] ? '' : 'WHERE feature_id IN (' . implode(',', $allowedFilters) . ')'))->iterateAssociative() as $row) {
			$featureValuesParent[$row['id']] = $row['feature'];
		}

		$vatRates = [];
		foreach ($conn->executeQuery("SELECT id, rate FROM eshop_catalog__vat_rate")->iterateAssociative() as $row) {
			$vatRates[$row['id']] = $row['rate'];
		}

		$qb = $this->em->createQueryBuilder()
			->select('p.id, p.price, IDENTITY(p.availability) as av, IDENTITY(p.manufacturer) as m, IDENTITY(p.vatRate) as vatRate,
				GROUP_CONCAT(DISTINCT CONCAT(IDENTITY(dfp.feature),\'|\',dfp.value) SEPARATOR \';\') as dynamicFeatures,
				GROUP_CONCAT(DISTINCT IDENTITY(pt.tag)) as tags')
			->from(Product::class, 'p')
			->andWhere('p.price IS NOT NULL')
			->leftJoin('p.dynamicFeatures', 'dfp')
			->leftJoin('p.productTags', 'pt', Join::WITH, '(pt.validFrom IS NULL OR pt.validFrom <= \'' . $now . '\') AND (pt.validTo IS NULL OR pt.validTo >= \'' . $now . '\')')
			->groupBy('p.id');

		if (Config::load('product.publishedByLang')) {
			$qb->innerJoin('p.productTexts', 'ptx', Join::WITH, 'ptx.lang = \'' . $this->translator->getLocale() . '\' AND ptx.isPublished = 1');
		} else {
			$qb->andWhere('p.isPublished = 1');
		}

		if ($avs !== []) {
			$qb->andWhere('p.availability IN (' . implode(',', $avs) . ')');
		}

		$qbFp = $this->em->createQueryBuilder()
			->select('IDENTITY(fp.product) as id, IDENTITY(fp.featureValue) as featureValue')
			->from(FeatureProduct::class, 'fp');

		if ($catIds) {
			$prods = [];

			$processBaseData = function($qb) use (&$data, $vatRates, &$prods): void {
				$query = $qb->getQuery();
				$sql   = $query->getSQL();

				$paramValues = [];
				foreach ($query->getParameters() as $param) {
					$paramValues[$param->getName()] = $param->getValue();
				}

				foreach ($this->em->getConnection()->executeQuery($sql, $paramValues)->iterateAssociative() as $row) {
					$row = array_values($row);

					$v = [
						'id'              => (int) $row[0],
						'price'           => $row[1],
						'av'              => $row[2],
						'm'               => $row[3],
						'vatRate'         => $row[4],
						'dynamicFeatures' => $row[5],
						'tags'            => $row[6],
					];

					$prods[$v['id']] = $v['id'];
					$this->buildBaseData($data, $v, $vatRates);
				}
			};

			$qb2 = clone $qb;

			$qb->innerJoin('p.categoryProducts', 'cp', Join::WITH, 'cp.category IN (' . $catIds . ')');

			$processBaseData($qb);
			$qb = null;

			$qb2->innerJoin('p.sites', 's', Join::WITH, 's.site = \'' . $site->getIdent() . '\' AND s.isActive = 1 AND s.category IN (' . $catIds . ')');

			$processBaseData($qb2);
			$qb2 = null;

			$processFeatures = false;
			if ($featureValuesParent && count($featureValuesParent) < 1000) {
				$processFeatures = "SELECT fp.id_product as id, fp.id_feature_value as featureValue
					FROM eshop_catalog__feature_product fp
					WHERE fp.id_feature_value IN (" . implode(',', array_keys($featureValuesParent)) . ")
					AND fp.id_product IS NOT NULL";
			} else if ($allowedFilters) {
				$processFeatures = "SELECT fp.id_product as id, fp.id_feature_value as featureValue
					FROM eshop_catalog__feature_product fp
					INNER JOIN eshop_catalog__feature_value fv ON fv.id = fp.id_feature_value AND fv.feature_id IN (" . implode(',', $allowedFilters) . ")
					WHERE fp.id_product IS NOT NULL";
			}

			if ($processFeatures) {
				$stmt = $this->em->getConnection()->executeQuery($processFeatures);
				foreach ($stmt->iterateAssociative() as $v) {
					if (!isset($prods[$v['id']])) {
						continue;
					}

					$this->buildFeaturesData($data, (int) $v['featureValue'], (int) $v['id'], $featureValuesParent);
				}
			}

			gc_collect_cycles();
		} else {
			foreach (array_chunk($productsIds, 999) as $chunk) {
				$qb2   = clone $qb;
				$qbFp2 = clone $qbFp;

				$qb2->andWhere('p.id IN (' . implode(',', $chunk) . ')');
				$qbFp2->andWhere('fp.product IN (' . implode(',', $chunk) . ')');

				foreach ($qb2->getQuery()->getScalarResult() as $v) {
					$this->buildBaseData($data, $v, $vatRates);
				}

				foreach ($qbFp2->getQuery()->getScalarResult() as $v) {
					$this->buildFeaturesData($data, (int) $v['featureValue'], (int) $v['id'], $featureValuesParent);
				}
			}
		}

		if ($catIds) {
			$this->cacheService->filterCache->save($cacheKey, $data, $this->cacheDep);
		}

		return $data;
	}

	public function getProductsData(array $productsIds, string $cacheKey = ''): ?array
	{
		if (!$this->cProductsData) {
			$this->cProductsData = $this->prepareData($productsIds);
		}

		return $this->cProductsData;
	}

}
