<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Countries;
use Core\Model\Helpers\BaseFrontEntityService;
use EshopCatalog\Model\Entities\Manufacturer;
use Nette\Caching\Cache;
use Throwable;

class Manufacturers extends BaseFrontEntityService
{
	public const CACHE_NAMESPACE = 'manufacturers';

	protected $entityClass = Manufacturer::class;

	protected ?array $cManufacturers = null;
	protected array  $cacheDep       = [
		Cache::Tags   => ['manufacturers'],
		Cache::EXPIRE => '1 week',
	];

	public function __construct(
		protected CacheService $cacheService,
		protected Countries    $countries,
	)
	{
	}

	public function getCache(): Cache
	{
		if ($this->cache === null) {
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);
		}

		return $this->cache;
	}

	/**
	 * @return Dao\Manufacturer[]
	 * @throws Throwable
	 */
	public function getManufacturers(): array
	{
		if ($this->cManufacturers === null) {
			$this->cManufacturers = [];

			$locale = $this->translator->getLocale();
			$key    = 'manufacturers/' . $locale;

			$this->cManufacturers = $this->cacheService->defaultCache->load($key, function(&$dep) use ($locale) {
				$dep    = $this->cacheDep;
				$result = [];

				foreach ($this->em->getConnection()->fetchAllAssociative("SELECT m.id, m.logo, m.name, m.website,
	                    m.country_id, m.street, m.city, m.zip_code, m.email, m.contact_link, m.phone,
	                    mt.description
					FROM eshop_catalog__manufacturer m
					LEFT JOIN eshop_catalog__manufacturer_texts mt ON m.id = mt.id AND mt.lang = :lang
					WHERE m.is_published = 1", [
					'lang' => $locale,
				]) as $row) {
					$result[$row['id']] = $this->fillDao($row);
				}

				return $result;
			});
		}

		return $this->cManufacturers;
	}

	public function get(int $id): ?Dao\Manufacturer
	{
		return $this->getManufacturers()[$id] ?? null;
	}

	protected function fillDao(array $a): Dao\Manufacturer
	{
		$manufacturer              = new Dao\Manufacturer($a['id'], $a['name']);
		$manufacturer->logo        = $a['logo'];
		$manufacturer->description = $a['description'];
		$manufacturer->website     = $a['website'];
		$manufacturer->country     = $a['country_id'] ? $this->countries->getDao()[$a['country_id']] : null;
		$manufacturer->street      = $a['street'];
		$manufacturer->city        = $a['city'];
		$manufacturer->zipCode     = $a['zip_code'];
		$manufacturer->email       = $a['email'];
		$manufacturer->contactLink = $a['contact_link'];
		$manufacturer->phone       = $a['phone'];

		return $manufacturer;
	}

}
