<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use EshopCatalog\FrontModule\Model\Dao\Document;
use EshopCatalog\Model\Entities\ProductDocument;
use Nette\Caching\Cache;

class ProductDocuments
{
	public const CACHE_NAMESPACE = 'eshopCatalogProductsDocuments';

	/** @var Dao\Document[][] */
	protected array $cDocuments = [];

	public function __construct(
		protected EntityManagerDecorator $em,
		protected Translator             $translator,
		protected CacheService           $cacheService,
	)
	{
	}

	/**
	 * @return Dao\Document[][]
	 */
	public function getDocuments(array $ids): array
	{
		$whereIds = [];
		/** @var Dao\Document[][] $result */
		$result = [];
		$locale = $this->translator->getLocale();
		$keys   = [];

		foreach ($ids as $id) {
			if (isset($this->cDocuments[$id])) {
				$result[$id] = $this->cDocuments[$id];
			} else {
				$keys[] = 'documents/' . $id;
			}
		}

		foreach ($this->cacheService->documentsCache->bulkLoad($keys) as $key => $documents) {
			$id = explode('/', $key)[1];

			if ($documents !== null) {
				$this->cDocuments[$id] = $documents;
				$result[$id]           = $documents;
			} else {
				$whereIds[] = $id;
			}
		}

		if ($whereIds !== []) {
			$qb = $this->em->getRepository(ProductDocument::class)->createQueryBuilder('pd')
				->select('IDENTITY(pd.product) as product, pd.id, pd.name, pd.file, pd.lang')
				->where('pd.product IN (:ids)')
				->setParameters(new ArrayCollection([new Parameter('ids', $whereIds)]));

			foreach ($qb->getQuery()->getArrayResult() as $row) {
				$this->cDocuments[$row['product']][] = $this->fillDao($row);
			}

			foreach ($whereIds as $id) {
				if (!isset($this->cDocuments[$id])) {
					$this->cDocuments[$id] = [];
				} else {
					$result[$id] = $this->cDocuments[$id];
				}

				$this->cacheService->documentsCache->save('documents/' . $id, $this->cDocuments[$id], [
					Cache::EXPIRATION => '14 days',
				]);
			}
		}

		foreach ($result as $id => $documents) {
			foreach ($documents as $k => $v) {
				if ($v->getLang() !== null && $v->getLang() !== $locale) {
					unset($result[$id][$k]);
				}
			}
		}

		return $result;
	}

	protected function fillDao(array $data): Document
	{
		return new Document(
			(int) $data['id'],
			(string) $data['name'],
			(string) $data['file'],
			$data['lang'],
		);
	}

}
