<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Application\AppState;
use Core\Model\Entities\EntityRepository;
use Core\Model\Entities\ExtraField;
use Core\Model\Event\DaoEvent;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Helpers\Strings;
use Core\Model\Lang\DefaultLang;
use Core\Model\Sites;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\Query\Parameter;
use EshopCatalog\FrontModule\Model\Event\ProductsEvent;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\FeatureProduct;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductInSite;
use EshopCatalog\Model\Entities\ProductPriceLevelCountry;
use EshopCatalog\Model\Entities\ProductTexts;
use EshopCatalog\Model\Entities\ProductVariant;
use EshopCatalog\Model\Helpers\Helper;
use EshopCatalog\Model\Helpers\ProductsHelper;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\FrontModule\Model\GroupCustomers;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Helpers\OrderHelper;
use EshopOrders\Model\ProductsPriceLevel;
use Gallery\FrontModule\Model\Albums;
use Nette\Application\LinkGenerator;
use Nette\Caching\Cache;
use Nette\DI\Attributes\Inject;
use Nette\DI\Container;
use Nette\Http\Session;
use Nette\Utils\DateTime;
use Throwable;
use Users\Model\Security\User;
use Users\Model\Users;

/**
 * TODO možnost přesunutí do fasády
 * @method EntityRepository getEr()
 */
class Products extends BaseFrontEntityService
{
	public const CACHE_NAMESPACE       = 'eshopCatalogProducts';
	public const CACHE_PRICE_NAMESPACE = 'eshopCatalogPrice';
	public const MODE_ESHOP            = 'eshop';
	public const MODE_CHECKOUT         = 'checkout';

	protected static string $mode        = self::MODE_ESHOP;
	public static ?string   $loadKey     = null;
	protected               $entityClass = Product::class;

	#[Inject]
	public DefaultLang $defaultLang;

	#[Inject]
	public Albums $albumsService;

	#[Inject]
	public Manufacturers $manufacturersService;

	#[Inject]
	public Sites          $sitesService;
	public ?LinkGenerator $linkGenerator = null;

	/** @var callable[] */
	public array $customQB = [];

	protected array  $cProductsIdInCategory = [];
	protected array  $cProductsIdBySearch   = [];
	protected array  $cProductsIdAll        = [];
	protected array  $cGroupCustomer        = [];
	protected ?array $cExtraFields          = null;
	protected array  $cIdsExtraFields       = [];
	protected array  $cQuantity             = [];
	protected ?array $cExtraFields2         = null;
	protected array  $cForNavigation        = [];

	/** @var callable[] */
	public array $onAfterProductsGet = [];

	public array $cacheDep = [
		Cache::Tags   => ['products'],
		Cache::Expire => '15 minutes',
	];

	public function __construct(
		public FeatureProducts            $featureProductsService,
		protected CacheService            $cacheService,
		protected ProductDocuments        $productDocuments,
		protected Users                   $users,
		protected User                    $user,
		protected Config                  $config,
		protected EventDispatcher         $eventDispatcher,
		protected AvailabilityService     $availabilityService,
		protected Helper                  $helper,
		protected ProductVideos           $productVideos,
		protected Tags                    $tagsService,
		protected AppState                $appState,
		protected Customers               $customers,
		protected GroupCustomers          $groupCustomers,
		protected ProductsPriceLevel      $productsPriceLevel,
		protected Session                 $session,
		protected DynamicFeaturesProducts $dynamicFeaturesProducts,
		protected Container               $container
	)
	{
		$this->onAfterProductsGet[] = function($products): void {
			/** @var Dao\Product[] $products */
			$this->loadPriceLevels($products,
				$this->user->getIdentity() ? $this->user->getIdentity()->getId() : null);
		};
	}

	public function findIdByAlias(string $alias): ?array
	{
		$qb = $this->em->createQueryBuilder()
			->select('IDENTITY(pt.id) as id, IDENTITY(site.category) as category')
			->from(ProductTexts::class, 'pt')
			->innerJoin('pt.id', 'p')
			->innerJoin('p.sites', 'site', Join::WITH, 'site.site = :site')
			->where('pt.alias = :alias')
			->setParameters(new ArrayCollection([new Parameter('alias', $alias), new Parameter('site', $this->sitesService->getCurrentSite()->getIdent())]));

		if (Config::load('product.publishedByLang')) {
			$qb->andWhere('pt.isPublished = 1');
		} else {
			$qb = $qb->andWhere('p.isPublished = 1');
		}

		return $qb->getQuery()->getArrayResult() ?: [];
	}

	public function getForNavigation(int $id, bool $withCategoryId = false, ?string $lang = null): ?array
	{
		if (!$lang) {
			$lang = $this->translator->getLocale();
		}

		if (!array_key_exists($lang, $this->cForNavigation)) {
			$this->cForNavigation[$lang] = [];
		}

		if (($this->cForNavigation[$lang][$id] ?? null) !== null) {
			return $this->cForNavigation[$lang][$id] ?: null;
		}

		$forLoad = [];
		foreach (array_merge(ProductsFacade::$allLoadedIds, [$id]) as $lId) {
			if (($this->cForNavigation[$lang][$lId] ?? null) === null) {
				$forLoad[]                         = $lId;
				$this->cForNavigation[$lang][$lId] = false;
			}
		}

		foreach (array_chunk($forLoad, 900) as $chunk) {
			$qb = $this->em->createQueryBuilder()->select('IDENTITY(pt.id) as id, pt.name, pt.alias, pt.lang')
				->from(ProductTexts::class, 'pt')
				->where('pt.id IN (' . implode(',', $chunk) . ')');

			if ($withCategoryId) {
				$qb->innerJoin('pt.id', 'p')
					->innerJoin('p.sites', 'site', Join::WITH, 'site.site = :site')
					->setParameter('site', $this->sitesService->getCurrentSite()->getIdent())
					->addSelect('IDENTITY(site.category) as category');
			}

			$resultCache = Config::loadInt('productsList.enableResultCacheOther') ?: 0;

			$query = $qb->getQuery();
			if ($resultCache) {
				$query->setQueryCacheLifetime($resultCache);
				$query->enableResultCache($resultCache, md5(serialize(($withCategoryId ? 'withC' : '') . implode(',', $chunk))));
			}

			foreach ($query->getArrayResult() as $row) {
				if (!$row['alias'] && $row['name'] && $row['lang']) {
					$row['alias'] = Strings::webalize($row['name']);

					$this->em->getConnection()->update('eshop_catalog__product_texts', [
						'alias' => $row['alias'],
					], [
						'id'   => $row['id'],
						'lang' => $row['lang'],
					]);

					$this->cacheService->productCache->remove('product/' . $row['id'] . '/' . $row['lang']);
				}

				$this->cForNavigation[$row['lang']][$row['id']] = $row;
			}
		}

		return $this->cForNavigation[$lang][$id] ?: null;
	}

	/**
	 * @param int|int[] $categoryId
	 *
	 * @return int[]
	 */
	public function getProductsIdInCategory(
		$categoryId,
		?int $start = 0,
		?int $limit = null,
		?string $sort = null,
		?array $filter = [],
		bool $orderByPosition = true
	): array
	{
		if ($sort) {
			$sort = str_replace('%2B', ' ', $sort);
		}

		if ($sort === 'recommended') {
			$sort = null;
		}

		$lang    = $this->translator->getLocale();
		$site    = $this->sitesService->getCurrentSite();
		$keyBase = $site->getIdent() . '/' . md5(serialize([$categoryId, $sort, $filter,
				$orderByPosition])) . '/' . $lang;

		if (isset($this->cProductsIdInCategory[$keyBase])) {
			if ($limit) {
				return array_slice($this->cProductsIdInCategory[$keyBase], $start, $limit);
			}

			return $this->cProductsIdInCategory[$keyBase];
		}

		$loadData = function() use ($categoryId, $site, $sort) {
			if (is_array($categoryId) && count($categoryId) === 1) {
				$categoryId = (int) array_values($categoryId)[0];
			}

			$parameters = [];

			if (is_array($categoryId)) {
				$catIdQuery = ' IN (' . implode(',', $categoryId) . ')';
			} else {
				$catIdQuery               = ' = :categoryId';
				$parameters['categoryId'] = (int) $categoryId;
			}

			$categoryPublishedByLang = Config::load('category.publishedByLang');
			$productPublishedByLang  = Config::load('product.publishedByLang');

			$selects      = [
				'p.id',
				'IFNULL(p.position, 999999) AS sPosition',
			];
			$joins        = [
			];
			$joinsSiteCat = [
				'INNER JOIN eshop_catalog__product_in_site pSites ON p.id = pSites.product_id AND pSites.is_active = 1' . (self::$mode === self::MODE_ESHOP ? " AND pSites.site = '{$site->getIdent()}'" : '') . ' AND pSites.category_id ' . $catIdQuery,
			];
			$joinsCats    = [
				'INNER JOIN eshop_catalog__category_product pCats ON p.id = pCats.id_product AND pCats.id_category ' . $catIdQuery,
				'INNER JOIN eshop_catalog__product_in_site pSites2 ON p.id = pSites2.product_id AND pSites2.is_active = 1' . (self::$mode === self::MODE_ESHOP ? " AND pSites2.site = '{$site->getIdent()}'" : ''),
			];
			$where        = ['p.is_deleted = 0 AND p.disable_listing = 0 AND p.price IS NOT NULL'];
			$subOrder     = [
				'sPosition ASC',
			];

			if ($categoryPublishedByLang) {
				$joinsCats[]        = 'INNER JOIN eshop_catalog__category_texts pCatsCatText ON pCatsCatText.id = pCats.id_category AND pCatsCatText.lang = :lang AND pCatsCatText.is_published = 1';
				$joinsCats[]        = 'INNER JOIN eshop_catalog__category pSitesCat2 ON pSitesCat2.id = pSites2.category_id AND pSitesCat2.is_published = 1';
				$parameters['lang'] = $this->translator->getLocale();
			} else {
				$joinsSiteCat[] = 'INNER JOIN eshop_catalog__category pSitesCat ON pSitesCat.id = pSites.category_id AND pSitesCat.is_published = 1';
				$joinsCats[]    = 'INNER JOIN eshop_catalog__category pCatsCat ON pCatsCat.id = pCats.id_category AND pCatsCat.is_published = 1';
			}

			if ($productPublishedByLang) {
				$joins[]            = 'INNER JOIN eshop_catalog__product_texts pTexts ON pTexts.id = p.id AND pTexts.lang = :lang AND pTexts.is_published = 1 AND pTexts.name != \'\'';
				$parameters['lang'] = $this->translator->getLocale();
			} else {
				$where[] = 'p.is_published = 1';
			}

			if (is_null($sort) || $sort === '' || $sort === 'recommended') {
				// Default sort
				$allTags = $this->tagsService->getAll();
				/** @var Dao\Tag|null $tag */
				$tag = $allTags[Config::loadString('productsList.defaultSortTopTag')] ?? null;

				$now = new DateTime;
				$now->setTime((int) $now->format("H"), (int) $now->format("i"), 0);
				$now    = $now->format('Y-m-d H:i:s');
				$tagKey = (string) $tag->id;

				$joins[]                          = 'LEFT JOIN eshop_catalog__product_tag pTags ON pTags.id_product = p.id AND pTags.id_tag = :pTagId' . $tagKey . ' AND (pTags.valid_from <= :pTagFrom' . $tagKey . ' OR pTags.valid_from IS NULL) AND (pTags.valid_to >= :pTagTo' . $tagKey . ' OR pTags.valid_to IS NULL)';
				$parameters['pTagFrom' . $tagKey] = $now;
				$parameters['pTagTo' . $tagKey]   = $now;
				$parameters['pTagId' . $tagKey]   = $tag->id;

				$selects[]  = 'IF(pTags.id_product IS NULL, 2, 1) AS sTag';
				$subOrder[] = 'sTag ASC';
			}

			$availabilities = $this->availabilityService->getAll();
			$showOnlyStock  = (bool) Config::loadScalar('onlyStockProductsInList');
			$avSort         = [1 => [], 0 => []];
			$avFilter       = [1 => [], 0 => []];

			foreach ($availabilities as $av) {
				$availabilities[$av->getIdent()] = $av;

				$avSort[$av->canAddToCart() ? 1 : 0][] = $av->getId();

				if ($showOnlyStock) {
					$avFilter[$av->canShowOnList() ? 1 : 0][] = $av->getId();
				}
			}

			if (!$sort || ($sort === 'recommended' && ($avSort[1] !== []))) {
				$selects[]  = 'IF(p.id_availability IN (' . implode(',', $avSort[1]) . '), 1, 0) AS avSort';
				$subOrder[] = 'avSort DESC';
			}

			if ($avFilter[1] !== []) {
				$where[] = 'p.id_availability IN (' . implode(',', $avFilter[1]) . ')';
			}

			$hideIfCountryPriceMissing = (array) Config::load('product.hideIfCountryPriceMissing') ?: [];
			if (!empty($hideIfCountryPriceMissing) && Config::load('enableCountryPrices')) {
				$country = $this->appState->getCountry();

				if (Arrays::contains($hideIfCountryPriceMissing, $country)) {
					$joins[]                 = 'INNER JOIN eshop_catalog__product_price pPrice ON pPrice.product = p.id AND pPrice.country = :cpCountry AND pPrice.price IS NOT NULL';
					$parameters['cpCountry'] = $country;
				}
			}

			$subOrder[] = 'id DESC';

			if (is_string($sort) && strlen($sort) > 0) {
				$sort = urldecode($sort);
				$sort = str_replace('+', ' ', $sort);
				$tmp  = explode(' ', $sort);

				$selects[] = $tmp[0] . ' as filterSort';
				$subOrder  = [
					'filterSort ' . (str_contains($sort, 'DESC') ? 'DESC' : 'ASC'),
					'id DESC',
				];
			}

			$ids        = [];
			$stmtResult = $this->em->getConnection()->executeQuery('SELECT * FROM'
				. '(SELECT ' . implode(', ', $selects) . ' FROM eshop_catalog__product p'
				. ' ' . implode("\n", $joinsSiteCat)
				. ' ' . implode("\n", $joins)
				. ' WHERE ' . implode(' AND ', $where) . ')'
				. ' as s1 UNION SELECT ' . implode(', ', $selects) . ' FROM eshop_catalog__product p'
				. ' ' . implode("\n", $joinsCats)
				. ' ' . implode("\n", $joins)
				. ' WHERE ' . implode(' AND ', $where)
				. ' ORDER BY ' . implode(', ', $subOrder), $parameters);
			while (($row = $stmtResult->fetchAssociative()) !== false) {
				$ids[$row['id']] = $row['id'];
			}

			return $ids;
		};

		$enableResultCache = (int) Config::loadScalar('productsList.enableResultCache') ?: 0;
		if ($enableResultCache) {
			$this->cProductsIdInCategory[$keyBase] = $this->cacheService->productCache->load($keyBase, function(?array &$dep) use ($enableResultCache, $loadData) {
				$dep                = $this->cacheDep;
				$dep[Cache::EXPIRE] = $enableResultCache . ' seconds';

				return $loadData();
			});
		} else {
			$this->cProductsIdInCategory[$keyBase] = $loadData();
		}

		if ($limit) {
			return array_slice($this->cProductsIdInCategory[$keyBase], $start, $limit);
		}

		return $this->cProductsIdInCategory[$keyBase];
	}

	/**
	 * @param int $categoryId
	 */
	public function getProductsInCategoryCount($categoryId, ?array $filter = []): int
	{
		return count($this->getProductsIdInCategory($categoryId, 0, null, null, $filter));
	}

	/**
	 * @return int[]
	 */
	public function getProductsIdBySearch(
		string  $q,
		?int    $start = 0,
		?int    $limit = null,
		?string $sort = null,
		?array  $filter = []
	): array
	{
		$keyBase = md5(serialize([$q, $sort, $filter]));

		if (isset($this->cProductsIdBySearch[$keyBase])) {
			if ($limit) {
				return array_slice($this->cProductsIdBySearch[$keyBase], $start, $limit);
			}

			return $this->cProductsIdBySearch[$keyBase];
		}

		$query            = (new ProductQuery($this->defaultLang->locale));
		$query->siteIdent = $this->sitesService->getCurrentSite()->getIdent();
		$query->search($q)
			->selectIds()
			->disableListing(0)
			->disableAutoIdSort()
			->hasPrice()
			->inSite($query->siteIdent);

		$ids = $this->processProductsResult($query, $filter, $sort, 'search');

		if (is_null($sort) || $sort === '' || $sort === 'recommended') {
			$this->cProductsIdBySearch[$keyBase] = array_unique(array_merge(
				$this->getRecommendedIds($q, 0, null, null, $filter, 'search'),
				array_keys($ids),
			));
		} else {
			$ids = array_keys($ids);

			$this->cProductsIdBySearch[$keyBase] = $ids;
		}

		if ($limit) {
			return array_slice($this->cProductsIdBySearch[$keyBase], $start, $limit);
		}

		return $this->cProductsIdBySearch[$keyBase];
	}

	protected function processProductsResult(
		ProductQuery $query,
		?array       $filter,
		?string      $sort,
		string       $type = 'default',
		bool         $orderByPosition = true
	): array
	{
		$ids               = [];
		$enableResultCache = (int) Config::loadScalar('productsList.enableResultCache') ?: 0;
		$cacheKey          = [];
		$availabilities    = $this->availabilityService->getAll();
		$showOnlyStock     = (bool) Config::loadScalar('onlyStockProductsInList');
		$avSort            = [1 => [], 0 => []];

		foreach ($availabilities as $av) {
			$availabilities[$av->getIdent()]       = $av;
			$avSort[$av->canAddToCart() ? 1 : 0][] = $av->getId();
		}

		if ($filter) {
			$query->useFilter($this->prepareFilter($filter));
			$cacheKey['filter'] = $filter;
		}

		if (is_string($sort)) {
			$sort             = urldecode($sort);
			$cacheKey['sort'] = $sort;
		}

		$hideIfCountryPriceMissing = (array) Config::load('product.hideIfCountryPriceMissing') ?: [];
		if (Config::load('enableCountryPrices') && !empty($hideIfCountryPriceMissing)) {
			$country = $this->appState->getCountry();
			if (Arrays::contains($hideIfCountryPriceMissing, $country)) {
				$query->countryPriceFilled($country);
			}
			$cacheKey['country'] = $country;
		}

		if ($sort) {
			$sort = str_replace('+', ' ', $sort);
		}

		$this->prepareSort($query, $sort);

		$isAutoIdSortDisabled = $query->disableAutoIdSort;
		$qb                   = $query->getQueryBuilder($this->getEr());

		foreach ($this->customQB as $cqb) {
			$cqb($qb);
			$cacheKey['customQb'] = true;
		}

		$qb->addSelect('IDENTITY(p.availability) as av');

		if (!$sort || ($sort === 'recommended' && ($avSort[1] !== []))) {
			$qb->addSelect('CASE WHEN p.availability IN (' . implode(',', $avSort[1]) . ') THEN 1 ELSE 0 END as avSort')
				->addOrderBy('avSort', 'DESC');
			$cacheKey['avSort'] = true;
		}

		if ($sort && $sort !== 'recommended') {
			$tmp = explode(' ', $sort);
			$qb->addSelect($tmp[0] . ' as sort');
		}

		if ($isAutoIdSortDisabled) {
			$qb->addOrderBy('p.id', 'DESC');
			$cacheKey['disabledAutoIdSort'] = true;
		}

		$query = $qb->getQuery();
		if ($enableResultCache) {
			$query->setQueryCacheLifetime($enableResultCache);
			$query->enableResultCache($enableResultCache, md5(serialize($cacheKey)));
		}

		foreach ($query->getScalarResult() as $v) {
			if (!isset($v['av']) || empty($v['av'])) {
				$v['av'] = Availability::SOLD_OUT;
			}

			if ($type === 'search') {
				if ($showOnlyStock && (!isset($availabilities[$v['av']]) || !$availabilities[$v['av']]->canShowOnSearch())) {
					continue;
				}
			} else if ($showOnlyStock && (!isset($availabilities[$v['av']]) || !$availabilities[$v['av']]->canShowOnList())) {
				continue;
			}

			$ids[$v['id']] = $v;
		}

		return $ids;
	}

	/**
	 * @return int[]
	 * @throws Throwable
	 */
	public function getProductsIdAll(?int $start = 0, ?int $limit = null, ?string $sort = null, ?array $filter = []): array
	{
		$site      = $this->sitesService->getCurrentSite();
		$keySuffix = $sort === null && ($filter === null || $filter === []) ? 'all' : md5(serialize([$sort, $filter]));
		$keyBase   = $site->getIdent() . '_' . $keySuffix;

		if (isset($this->cProductsIdAll[$keyBase])) {
			if ($limit) {
				return array_slice($this->cProductsIdAll[$keyBase], $start, $limit);
			}

			return $this->cProductsIdAll[$keyBase];
		}

		$ids = $this->cacheService->defaultCache->load($keyBase, function(&$dep) use ($sort, $filter, $site) {
			$dep = $this->cacheDep;

			if (isset($filter['manu'])) {
				$dep[Cache::TAGS][] = 'manufacturers';
			}
			if (isset($filter['features'])) {
				$dep[Cache::TAGS][] = 'features';
			}

			$query = (new ProductQuery($this->defaultLang->locale))
				->inSite($site->getIdent())
				->hasPrice()
				->orderByPosition();

			if ($filter) {
				$query->useFilter($this->prepareFilter($filter));
			}
			$this->prepareSort($query, $sort);

			$qb = $query->getQueryBuilder($this->getEr());
			$qb->addSelect('p.id, pv.variantId')
				->andWhere('p.isDeleted = 0')
				->leftJoin('p.isVariant', 'pv')
				->groupBy('p.id');
			foreach ($this->customQB as $cqb) {
				$cqb($qb);
			}

			$ids      = [];
			$variants = [];
			foreach ($qb->getQuery()->getScalarResult() as $v) {
				$ids[$v['id']]             = $v['id'];
				$variants[$v['variantId']] = $v['variantId'];
			}

			if ($variants !== []) {
				foreach (array_chunk($variants, 250) as $chunk) {
					$qbV = $this->em->getRepository(ProductVariant::class)
						->createQueryBuilder('pv')
						->select('IDENTITY(pv.product) as product')
						->where('pv.variantId IN (:variants)')
						->innerJoin('pv.product', 'p', Join::WITH, 'p.isDeleted = 0')
						->setParameter('variants', $chunk);

					if (Config::load('product.publishedByLang')) {
						$qbV->innerJoin('p.productTexts', 'pt', Join::WITH, 'pt.lang = :lang AND pt.isPublished = 1')
							->setParameter('lang', $this->translator->getLocale());
					} else {
						$qbV->andWhere('p.isPublished = 1');
					}
					foreach ($qbV->getQuery()->getArrayResult() as $row) {
						$ids[$row['product']] = $row['product'];
					}
				}
			}

			return $ids;
		});

		if (is_null($sort)) {
			$this->cProductsIdAll[$keyBase] = array_unique(array_merge($this->getRecommendedIds(null, 0, null, null, $filter), $ids));
		} else {
			$this->cProductsIdAll[$keyBase] = $ids;
		}

		if ($limit) {
			return array_slice($this->cProductsIdAll[$keyBase], $start, $limit);
		}

		return $this->cProductsIdAll[$keyBase];
	}

	/**
	 * @param string|int|null|array $q
	 *
	 * @throws Throwable
	 */
	public function getRecommendedIds($q = null, ?int $start = 0, ?int $limit = null, ?string $sort = null, ?array $filter = [], string $type = 'category'): array
	{
		$site = $this->sitesService->getCurrentSite();

		$allTags = $this->tagsService->getAll();
		/** @var Dao\Tag|null $tag */
		$tag = $allTags[Config::loadString('productsList.defaultSortTopTag')] ?? null;
		if (!$tag) {
			return [];
		}

		$query            = (new ProductQuery($this->defaultLang->locale));
		$query->siteIdent = $this->sitesService->getCurrentSite()->getIdent();
		$query->withTag($tag->id)
			->inSite($query->siteIdent)
			->hasPrice()
			->selectIds();

		if ((bool) Config::loadScalar('onlyStockProductsInList') === true) {
			$query->onlyInStockOrSupplier();
		}

		if ($filter) {
			$query->useFilter($this->prepareFilter($filter));
		}

		$ids = [];
		switch ($type) {
			case 'category':
				if ($q) {
					$query2 = clone $query;

					if (self::$mode === self::MODE_ESHOP) {
						$query->inSite($site->getIdent());
					}
					$query->inSiteCategory($q);
					$query2->inOtherCategories($q);
					$query->orderByPosition();
					$query2->orderByPosition();

					$ids = array_merge($this->endGetRecommended($query, $limit, $start), $this->endGetRecommended($query2, $limit, $start));
					$ids = array_unique($ids);
				}
				break;
			case 'search':
				$query->siteIdent = $this->sitesService->getCurrentSite()->getIdent();
				$query->search(is_array($q) ? (string) array_values($q)[0] : (string) $q);

				$ids = $this->endGetRecommended($query, $limit, $start);

				break;
		}

		return $ids;
	}

	protected function endGetRecommended(ProductQuery $query, ?int $limit = null, ?int $start = 0): array
	{
		$qb = $query->getQueryBuilder($this->getEr());
		foreach ($this->customQB as $cqb) {
			$cqb($qb);
		}

		$ids = [];

		$resultCache = (int) Config::load('productsList.enableResultCacheOther') ?: 0;
		$query       = $qb->getQuery();
		if ($resultCache) {
			$query->setQueryCacheLifetime($resultCache);
			$query->enableResultCache($resultCache, md5(serialize($query->getSQL()) . '_' . serialize($query->getParameters())));
		}

		foreach ($query->setMaxResults($limit)->setFirstResult($start)->getScalarResult() as $v) {
			$ids[] = $v['id'];
		}

		return $ids;
	}

	/**
	 * @param Dao\Product[] $products
	 *
	 * @throws NonUniqueResultException
	 */
	public function loadFresh(
		array   &$products,
		?int    $userId = null,
		bool    $loadRetailPrice = true,
		?string $country = null
	): void
	{
		if ($products === []) {
			return;
		}

		$country = $country ?: $this->appState->getCountry();
		$ids     = array_values(array_map(static fn($p): int => $p->getId(), $products));

		foreach (array_chunk($ids, 900) as $chunk) {
			$qb = $this->getEr()->createQueryBuilder('p')
				->select('p.id, p.price, IDENTITY(p.availability) as availability');

			if (Config::load('product.allowRetailPrice')) {
				$qb->addSelect('p.retailPrice');
			}

			if (count($chunk) === 1) {
				$qb->andWhere('p.id = ' . $chunk[0]);
			} else {
				$qb->andWhere('p.id IN (' . implode(',', $chunk) . ')');
			}

			$qb->andWhere('p.isDeleted = 0');

			if (Config::load('enableCountryPrices') && $country) {
				$qb->leftJoin('p.prices', 'cp', Join::WITH, 'cp.country = \'' . $country . '\'')
					->leftJoin('cp.vatRate', 'cpv')
					->addSelect('cp.price as cPrice, cp.currency as cCurrency, cpv.rate as cVatRate');

				if (!Config::load('enablePriceHistory')) {
					$qb->addSelect('cp.retailPrice as cRetailPrice');
				}
			}

			$query = $qb->getQuery();

			$cacheKey    = 'loadFresh/' . md5(serialize($query->getSQL()) . '_' . serialize($query->getParameters()));
			$resultCache = (int) Config::loadScalar('productsList.enableFreshResultCache');

			if ($resultCache) {
				$data = $this->cacheService->productCache->load($cacheKey, function(&$dep) use ($query, $resultCache) {
					$dep = [
						Cache::EXPIRE => $resultCache . ' seconds',
					];

					$sql = $query->getSQL();

					return $this->em->getConnection()->executeQuery(is_array($sql) ? array_values($sql)[0] : $sql)->fetchAllAssociative();
				});
			} else {
				$data = $query->getArrayResult();
			}

			foreach ($data as $row) {
				if (isset($row['id_0'])) {
					$raw = array_values($row);

					$row = [
						'id'           => $raw[0],
						'price'        => $raw[1],
						'availability' => $raw[2],
					];

					$i = 3;
					if (Config::load('product.allowRetailPrice')) {
						$row['retailPrice'] = $raw[$i++] ?? null;
					}

					if (Config::load('enableCountryPrices') && $country) {
						$row['cPrice']    = $raw[$i++] ?? null;
						$row['cCurrency'] = $raw[$i++] ?? null;
						$row['cVatRate']  = $raw[$i++] ?? null;

						if (!Config::load('enablePriceHistory')) {
							$row['cRetailPrice'] = $raw[$i++] ?? null;
						}
					}

					$raw = null;
				}

				if (!isset($products[$row['id']])) {
					continue;
				}

				// Skryti produktu pokud neobsahuje cenu pro zemi
				if (Arrays::contains((array) Config::load('product.hideIfCountryPriceMissing') ?: [], $country) && $row['cPrice'] === null) {
					unset($products[$row['id']]);
					continue;
				}

				$price             = (float) $row['price'];
				$retailPrice       = null;
				$canUseRetailPrice = Config::load('product.allowRetailPrice') && !Config::load('enablePriceHistory');

				if ($canUseRetailPrice) {
					$retailPrice = (float) $row['retailPrice'];
				}

				if ($row['cPrice']) {
					$price = (float) $row['cPrice'];

					if ($canUseRetailPrice) {
						$retailPrice = (float) $row['cRetailPrice'];
					}

					if ($row['cCurrency']) {
						$products[$row['id']]->currency = $row['cCurrency'];
					}
				}

				if ($row['cVatRate'] !== null) {
					$products[$row['id']]->setVatRate($row['cVatRate']);
				}

				if (Config::load('product.priceIsWithoutVat', false)) {
					$price = ProductsHelper::processPrice($price, $products[$row['id']]->getVatRate());

					/** @var bool $canUseRetailPrice */
					if ($retailPrice !== null && $canUseRetailPrice) {
						$retailPrice = ProductsHelper::processPrice($retailPrice, $products[$row['id']]->getVatRate());
					}
				}

				$products[$row['id']]->setPrice($price);
				$products[$row['id']]->setBasePrice($price);

				if ($retailPrice !== null && $canUseRetailPrice) {
					$products[$row['id']]->setRetailPrice($retailPrice);
					$products[$row['id']]->retailPriceInBaseCurrency = $retailPrice;
				}

				$products[$row['id']]->priceInBaseCurrency     = $price;
				$products[$row['id']]->basePriceInBaseCurrency = $price;
				$preorderAv                                    = $this->availabilityService->getByIdent(Availability::PREORDER);
				if ($products[$row['id']]->unlimitedQuantity && $preorderAv && $row['availability'] != $preorderAv->getId()) {
					$products[$row['id']]->setAvailability($this->availabilityService->getByIdent(Availability::IN_STOCK));
				} else if ($row['availability']) {
					$products[$row['id']]->setAvailability($this->availabilityService->get((int) $row['availability']));
				} else {
					$products[$row['id']]->setAvailability($this->availabilityService->getByIdent(Availability::SOLD_OUT));
				}
			}
		}

		if ($loadRetailPrice) {
			$this->loadRetailPrice($products);
		}

		$this->loadSite($products);
		$this->loadPriceLevels($products, $userId);
		$this->loadQuantity($products);

		$invoiceCountry = $this->appState->getCountry();
		if ($this->session->exists()) {
			$invoiceCountry = $this->session->getSection('eshopOrdersOrderForm')->orderFormData['country'] ?? $this->appState->getCountry();
		} else {
			$invoiceCountry = $this->appState->getCountry();
		}

		// vypocet ceny pokud to chceme pocitat z ceny bez dph
		if (
			Config::load('calculatePriceFromWithoutVat')
			&& !Config::load('enableCountryPrices')
			&& class_exists(OrderHelper::class)
		) {
			foreach ($products as $product) {
				$vatRate = $product->getVatRate();

				$newVatRate = OrderHelper::checkCountryVatRate($vatRate, $invoiceCountry, false);

				if ($newVatRate !== $vatRate) {
					if ($product->price) {
						$tmp                          = ProductsHelper::getPriceWithoutVat($product->price, $vatRate);
						$product->price               = ProductsHelper::processPrice($tmp, $newVatRate);
						$product->priceInBaseCurrency = $product->price;
					}

					if ($product->retailPrice) {
						$tmp                                = ProductsHelper::getPriceWithoutVat($product->retailPrice, $vatRate);
						$product->retailPrice               = ProductsHelper::processPrice($tmp, $newVatRate);
						$product->retailPriceInBaseCurrency = $product->retailPrice;
					}

					$product->setVatRate($newVatRate);
				}
			}
		}

		$this->eventDispatcher->dispatch(new ProductsEvent($products), Products::class . '::afterLoadFresh');
	}

	/**
	 * @param Dao\Product[] $products
	 *
	 * @throws Throwable
	 */
	public function loadRetailPrice(array &$products): void
	{
		if (!Config::load('enablePriceHistory') || !Config::load('product.allowLoadRetailPrice')) {
			return;
		}

		$conn            = $this->em->getConnection();
		$keys            = [];
		$keysCountry     = [];
		$whereIds        = [];
		$whereCountryIds = [];
		$country         = Config::load('enableCountryPrices') && $this->appState->getCountry()
			? $this->appState->getCountry()
			: null;

		foreach ($products as $product) {
			if ($product->currency === null || $product->currency === \Currency\Model\Config::load('default')) {
				$keys[$product->getId()] = 'retailPrice/' . $product->getId();
			} else if ($country) {
				$keysCountry[$product->getId()] = 'retailPriceCountry/' . Strings::lower($country) . '/' . $product->getId();
			}
		}

		if ($keysCountry !== []) {
			foreach ($this->cacheService->priceCache->bulkLoad($keysCountry) as $key => $retailPrice) {
				$tmp = explode('/', $key);
				$id  = $tmp[2];

				if (is_float($retailPrice)) {
					$products[$id]->retailPrice               = $retailPrice;
					$products[$id]->retailPriceInBaseCurrency = $retailPrice;
					/** @phpstan-ignore-next-line */
					unset($keys[$id]);
				} else if ($retailPrice === false) {
					$products[$id]->retailPrice               = null;
					$products[$id]->retailPriceInBaseCurrency = null;
				} else {
					$whereCountryIds[$id] = [
						'price' => $products[$id]->price,
					];
				}
			}
		}

		foreach (array_keys($whereCountryIds) as $id) {
			$currency = $products[$id]->currency;

			$in30Days = $conn->fetchAssociative("SELECT id, created, price FROM `eshop_catalog__product_price_history_country`
					where created >= date(now() - interval 30 day)
					and product = {$id} and country = '{$country}' and currency_code " . ($currency ? "= '{$currency}'" : " IS NULL") . "
					order by price ASC
					limit 1");

			// Vytvoření proměnné kdy se vyskytuje navýšení ceny
			$conn->executeStatement("set @compareId = (SELECT id FROM `eshop_catalog__product_price_history_country`
					where direction = 1 and product = {$id} and country = '{$country}' and currency_code " . ($currency ? "= '{$currency}'" : " IS NULL") . "
					order by created desc
					limit 1)");

			/** @var array $history */
			$history = $conn->fetchAssociative("SELECT * FROM `eshop_catalog__product_price_history_country`
					where id > case
					    when @compareId is not null then @compareId
					    else 0
				    end
					and product = {$id} and country = '{$country}' and currency_code " . ($currency ? "= '{$currency}'" : " IS NULL") . "
					order by created asc
					limit 1");

			// Porovnání nejnižší ceny za 30 dní vs stálá sleva podle navýšení
			$retailPrice = $in30Days && $in30Days['created'] < $history['created'] ? $in30Days['price'] : $history['price'];

			$retailPrice = $retailPrice
				? (float) $retailPrice
				: null;

			$products[$id]->retailPrice               = $retailPrice;
			$products[$id]->retailPriceInBaseCurrency = $products[$id]->retailPrice;
			/** @phpstan-ignore-next-line */
			unset($keys[$id]);

			$this->cacheService->priceCache->save('retailPriceCountry/' . Strings::lower($country) . '/' . $id, $retailPrice ?: false, [
				Cache::Expire => '1 week',
			]);
		}

		if ($keys !== []) {
			foreach ($this->cacheService->priceCache->bulkLoad($keys) as $key => $retailPrice) {
				$tmp = explode('/', $key);
				$id  = $tmp[1];

				if (is_float($retailPrice)) {
					$products[$id]->retailPrice               = $retailPrice;
					$products[$id]->retailPriceInBaseCurrency = $retailPrice;
				} else if ($retailPrice === false) {
					$products[$id]->retailPrice               = null;
					$products[$id]->retailPriceInBaseCurrency = null;
				} else {
					$whereIds[$id] = [
						'price' => $products[$id]->price,
					];
				}
			}
		}

		foreach (array_keys($whereIds) as $id) {
			$in30Days = $conn->fetchAssociative("SELECT id, created, price FROM `eshop_catalog__product_price_history`
					where created >= date(now() - interval 30 day)
					and product = {$id}
					order by price ASC
					limit 1");

			// Vytvoření proměnné kdy se vyskytuje navýšení ceny
			$conn->executeStatement("set @compareId = (SELECT id FROM `eshop_catalog__product_price_history`
					where direction = 1 and product = {$id}
					order by created desc
					limit 1)");

			/** @var array $history */
			$history = $conn->fetchAssociative("SELECT * FROM `eshop_catalog__product_price_history`
					where id > case
					    when @compareId is not null then @compareId
					    else 0
				    end
					and product = {$id}
					order by created asc
					limit 1");

			// Porovnání nejnižší ceny za 30 dní vs stálá sleva podle navýšení
			$retailPrice = $in30Days && $in30Days['created'] < $history['created'] ? $in30Days['price'] : $history['price'];

			$retailPrice = $retailPrice
				? (float) $retailPrice
				: null;

			if ($products[$id]->currency === null || $products[$id]->currency === \Currency\Model\Config::load('default')) {
				$products[$id]->retailPrice               = $retailPrice;
				$products[$id]->retailPriceInBaseCurrency = $products[$id]->retailPrice;
			}

			$this->cacheService->priceCache->save('retailPrice/' . $id, $retailPrice ?: false, [
				Cache::Expire => '1 week',
			]);
		}
	}

	/** @param Dao\Product[] $products */
	public function loadSite(array $products): void
	{
		$ids      = array_map(static fn($product): int => $product->getId(), $products);
		$variants = [];

		/** @var Categories $categoriesService */
		$categoriesService = $this->container->getService('eshopCatalog.front.categories');

		foreach (array_chunk($ids, 900) as $chunk) {
			$qb       = $this->em->getRepository(ProductInSite::class)->createQueryBuilder('ps')
				->select('IDENTITY(ps.product) as prod, IDENTITY(ps.site) as site, IDENTITY(ps.category) as cat, ps.isActive')
				->where('ps.product IN (:ids)')
				->andWhere('ps.isActive = 1')
				->setParameters(new ArrayCollection([new Parameter('ids', $chunk)]));
			$cacheKey = implode(',', $chunk);

			if (self::$mode === self::MODE_ESHOP) {
				$qb->andWhere('ps.site = :site')
					->setParameter('site', $this->sitesService->getCurrentSite()->getIdent());
				$cacheKey = $this->sitesService->getCurrentSite()->getIdent() . '_' . $cacheKey;
			}

			$resultCache = (int) Config::loadScalar('productsList.enableResultCacheOther');
			$query       = $qb->getQuery();
			if ($resultCache) {
				$query->setQueryCacheLifetime($resultCache);
				$query->enableResultCache($resultCache, $cacheKey);
			}

			foreach ($query->getScalarResult() as $row) {
				$products[$row['prod']]->defaultCategoryId = (int) $row['cat'];

				if ($products[$row['prod']]->defaultCategoryId) {
					$products[$row['prod']]->defaultCategory = $categoriesService->get($products[$row['prod']]->defaultCategoryId);
				}

				// Stav aktivni podle kategorie pouze tehdy pokud je produkt aktivni podle jazyka
				if ($products[$row['prod']]->isActive) {
					$products[$row['prod']]->isActive = (bool) $row['isActive'];
				}
			}

			foreach ($products as $product) {
				if ($product->variantOf) {
					$variants[$product->variantOf][] = $product->getId();
				}
			}

			// Projduti variant
			foreach ($variants as $k => $vals) {
				if (!isset($products[$k])) {
					continue;
				}

				foreach ($vals as $v) {
					$products[$v]->defaultCategoryId = $products[$k]->defaultCategoryId;
					$products[$v]->isActive          = $products[$k]->isActive;

					if ($products[$v]->defaultCategoryId) {
						$products[$v]->defaultCategory = $categoriesService->get($products[$v]->defaultCategoryId);
					}
				}
			}
		}
	}

	/**
	 * @param Dao\Product[] $products
	 *
	 * @throws NonUniqueResultException
	 */
	public function loadPriceLevels(array $products, ?int $userId = null): void
	{
		if (!$userId || !Config::load('enablePriceLevels', false)) {
			return;
		}

		$customer = $this->customers->getByUser((int) $userId);
		if (!$customer instanceof Customer) {
			return;
		}

		$priceIsWithoutVat = (bool) Config::loadScalar('product.priceIsWithoutVat');
		$ids               = [];
		$prices            = [];
		foreach ($products as $product) {
			$ids[$product->getId()]    = $product->getId();
			$prices[$product->getId()] = $product->getPrice();
		}

		if ($customer->getGroupCustomers() instanceof \EshopOrders\Model\Entities\GroupCustomers) {
			// Pokud je uživatel ve skupině a je u produktu nastavena cenová hladina, aplikujeme ji
			if (!isset($this->cGroupCustomer[$userId])) {
				$this->cGroupCustomer[$userId] = $this->groupCustomers->getGroupArray($customer->getGroupCustomers()->getId());
			}

			$group = $this->cGroupCustomer[$userId] ?? null;
			if ($group['id']) {
				$oIds              = $ids;
				$priceLevelGroupId = $group['id'];

				if ($group['useProductsSaleOnGroup'] && EshopOrdersConfig::load('customerGroup.allowUseSaleOnOtherGroupPrices')) {
					foreach ($this->productsPriceLevel->getPricesForGroup($oIds, $group['useProductsSaleOnGroup']) as $k => $v) {
						if ($priceIsWithoutVat) {
							$v = ProductsHelper::processPrice((float) $v, $products[$k]->getVatRate());
						}

						$prices[$k]             = $v;
						$products[$k]->currency = null;
					}
				}

				if ($group['inheritPricesGroup'] && EshopOrdersConfig::load('customerGroup.allowInheritGroupPrices')) {
					$priceLevelGroupId = $group['inheritPricesGroup'];
				}

				foreach ($this->productsPriceLevel->getPricesForGroup($oIds, $priceLevelGroupId) as $k => $v) {
					if ($priceIsWithoutVat) {
						$v = ProductsHelper::processPrice((float) $v, $products[$k]->getVatRate());
					}

					$prices[$k]             = (float) $v;
					$products[$k]->currency = null;
					unset($ids[$k]);
				}

				if (Config::load('enableCountryPrices') && $this->appState->getCountry() && $oIds) {
					$query = $this->em->getRepository(ProductPriceLevelCountry::class)->createQueryBuilder('pplc')
						->select('IDENTITY(pplc.product) as product, pplc.price, pplc.currency')
						->where('pplc.product IN (' . implode(',', array_values($oIds)) . ')')
						->andWhere('pplc.group = :groupId')
						->andWhere('pplc.country = :country')
						->setParameters(new ArrayCollection([new Parameter('groupId', $priceLevelGroupId), new Parameter('country', $this->appState->getCountry())]))
						->getQuery();

					$resultCache = (int) Config::loadScalar('productsList.enableFreshResultCache');

					if ($resultCache) {
						$query->setQueryCacheLifetime($resultCache);
						$query->enableResultCache($resultCache);
					}

					foreach ($query->getArrayResult() as $row) {
						if ($row['price']) {
							if ($priceIsWithoutVat) {
								$row['price'] = ProductsHelper::processPrice((float) $row['price'], $products[$row['product']]->getVatRate());
							}

							$prices[$row['product']]             = (float) $row['price'];
							$products[$row['product']]->currency = $row['currency'];
							unset($ids[$row['product']]);
						}
					}
				}
			}

			// Pokud je ve skupině nastavená sleva na všechny produkty, aplikuejeme ji na produkty. Buď na již zlevněné nebo ne
			if ($group['productsSale'] > 0) {
				$tmp = 1 - ((float) $group['productsSale'] / 100);

				$tmpLoop = $ids;
				if (EshopOrdersConfig::load('customerGroup.allowChangeProductSaleBasePrice') && $group['productsSaleUseGroupPrice']) {
					$tmpLoop = array_keys($prices);
				}

				foreach ($tmpLoop as $id) {
					$prices[$id] *= $tmp;
					unset($ids[$id]);
				}

				unset($tmpLoop);
			}
		}

		// Na neupravené produkty aplikujeme slevu hromadnou
		$user = $this->users->get($userId);
		if ($user) {
			$registerUserSale = $this->helper->getRegisterUserSale($user);
			if ($registerUserSale > 0) {
				$tmp = 1 - ($registerUserSale / 100);

				foreach ($ids as $id) {
					$product = $products[$id] ?? null;

					if ($product && ($product->disableRegisterSale || ($product->defaultCategory && $product->defaultCategory->disableRegisterSale))) {
						continue;
					}

					$prices[$id] *= $tmp;
					unset($ids[$id]);
				}
			}
		}

		// Aplikujeme slevu na produkty
		foreach ($prices as $product => $price) {
			$products[$product]->price               = $price;
			$products[$product]->priceInBaseCurrency = $price;
		}
	}

	/**
	 * @param Dao\Product[] $products
	 *
	 * @throws Throwable
	 */
	public function loadFeatures(array &$products): void
	{
		foreach ($this->featureProductsService->getFeaturesForProduct($products) as $k => $v) {
			$products[$k]->setFeatures($v);
		}

		foreach ($this->dynamicFeaturesProducts->getDaoByProducts(array_keys($products)) as $prodId => $rows) {
			foreach ($rows as $v) {
				foreach ($v as $k => $v2) {
					$products[$prodId]->features['dynamic_' . $k] = $v2;
				}
			}
		}
	}

	/**
	 * @param Dao\Product[] $products
	 */
	public function loadDocuments(array &$products): void
	{
		$ids = array_map(static fn(Dao\Product $product): int => $product->getId(), $products);

		foreach ($this->productDocuments->getDocuments($ids) as $productId => $documents) {
			foreach ($documents as $document) {
				$products[$productId]->addDocument($document);
			}
		}
	}

	/**
	 * @param Dao\Product[] $products
	 */
	public function loadVideos(array &$products): void
	{
		$ids = array_map(static fn(Dao\Product $product): int => $product->getId(), $products);

		foreach ($this->productVideos->getVideos($ids) as $productId => $videos) {
			foreach ($videos as $video) {
				$products[$productId]->addVideo($video);
			}
		}
	}

	/** @param Dao\Product[] $products */
	public function loadQuantity(array $products, bool $forceLoadData = false): void
	{
		if ($forceLoadData) {
			$this->cQuantity = [];
		}

		/** @var int[] $ids */
		$ids = [];
		foreach ($products as $p) {
			if (isset($this->cQuantity[$p->getId()])) {
				$products[$p->getId()]
					->setQuantity($this->cQuantity[$p->getId()]['local'])
					->setQuantityExternal($this->cQuantity[$p->getId()]['external']);
			} else {
				$ids[] = $p->getId();
			}
		}

		if (empty($ids)) {
			return;
		}

		$allowSuppliers = Config::load('allowSuppliers');

		foreach (array_chunk($ids, 900) as $chunk) {
			$data = [];

			if ($allowSuppliers) {
				foreach ($this->em->getConnection()->fetchAllAssociative(" SELECT p.id, p.quantity FROM eshop_catalog__product p WHERE p.id IN (" . implode(',', $chunk) . ")") as $row) {
					$data[$row['id']] = [
						'productQ'   => (int) $row['quantity'],
						'suppliersQ' => [],
						'suppliersA' => [],
					];
				}

				foreach ($this->em->getConnection()->fetchAllAssociative(" SELECT ps.id_product, ps.quantity, ps.id_availability_after_sold_out 
 						FROM eshop_catalog__product_supplier ps 
 						INNER JOIN eshop_catalog__supplier s ON s.id = ps.id_supplier AND s.allow_sale = 1
 						WHERE ps.id_product IN (" . implode(',', $chunk) . ")
 						AND ps.is_active = 1") as $row) {
					$data[$row['id_product']]['suppliersQ'][] = (int) $row['quantity'];
					$data[$row['id_product']]['suppliersA'][] = (int) $row['id_availability_after_sold_out'];
				}

				$availabilities = $this->availabilityService->getAll();

				foreach ($data as $id => $row) {
					if (!isset($row['productQ'])) {
						$row['productQ'] = 0;
					}

					$suppliersQ = $row['suppliersQ'];
					$suppliersA = null;

					foreach ($row['suppliersA'] as $k => $v) {
						if (!$suppliersA || ($suppliersQ[$k] > 0 && $suppliersA->getDelay() > $availabilities[$v]->getDelay())) {
							$suppliersA = $availabilities[$v];
						}
					}

					$suppliersQ = array_sum($suppliersQ);

					if (Config::load('sumLocalAndExternalQuantities')) {
						$row['productQ'] = (int) $row['productQ'] + (int) $suppliersQ;
					}

					$this->cQuantity[$id] = [
						'local'     => (int) $row['productQ'],
						'external'  => (int) $suppliersQ,
						'externalA' => $suppliersA,
					];
				}

				$allowExternal = (int) $this->settings->get('eshopCatalogDisableExternalStorage', 0) === 0;
				foreach (array_keys($data) as $id) {
					$products[$id]->setQuantity($this->cQuantity[$id]['local'])
						->setQuantityExternal($allowExternal ? $this->cQuantity[$id]['external'] : 0);
					$products[$id]->externalAvailability = $this->cQuantity[$id]['externalA'] ?? null;
				}
			} else {
				foreach ($this->em->getConnection()->fetchAllAssociative(" SELECT p.id, p.quantity FROM eshop_catalog__product p WHERE p.id IN (" . implode(',', $chunk) . ")") as $row) {
					$products[$row['id']]->setQuantity((int) $row['quantity']);

					$this->cQuantity[$row['id']] = [
						'local'     => (int) $row['quantity'],
						'external'  => 0,
						'externalA' => null,
					];
				}
			}
		}
	}

	public function fillDao(array $product): Dao\Product
	{
		$daoClass = Config::load('product.daoClass');
		/** @var Dao\Product $p */
		$p = new $daoClass($product['id']);
		$p->setQuantity((int) $product['quantity']);
		$p->setVatRate($product['vatRate']);

		if (Config::load('product.priceIsWithoutVat', false)) {
			foreach (['price', 'retailPrice'] as $c) {
				if ($product[$c]) {
					$product[$c] = ProductsHelper::processPrice((float) $product[$c], $p->getVatRate());
				}
			}
		}

		$p->setPrice((float) $product['price']);
		$p->setBasePrice((float) $product['price']);
		$p->setRecyclingFee((float) $product['recyclingFee']);

		if (Config::load('product.allowRetailPrice') && (float) $product['retailPrice'] > $p->getPrice()) {
			$p->setRetailPrice((float) $product['retailPrice']);
			$p->retailPriceInBaseCurrency = $p->retailPrice;
		} else {
			$p->retailPrice               = null;
			$p->retailPriceInBaseCurrency = null;
		}

		$p->priceInBaseCurrency     = $p->getPrice();
		$p->basePriceInBaseCurrency = $p->basePrice;

		if ($product['hsCustomsCode'] && Config::load('product.allowHSCustomsCode')) {
			$p->hsCustomsCode = (int) $product['hsCustomsCode'];
		}

		if ($product['countryOfOrigin'] && Config::load('product.allowCountryOfOrigin')) {
			$p->countryOfOrigin = $product['countryOfOrigin'];
		}

		$p->setName($product['productTexts']['name']);
		$p->setName2($product['productTexts']['name2']);
		$p->setDescription($product['productTexts']['description']);
		$p->setSeo($product['productTexts']['seo']);
		$p->setEan($product['ean'] ? (string) $product['ean'] : null);
		$p->setCode1($product['code1'] ? (string) $product['code1'] : null);
		$p->setCode2($product['code2'] ? (string) $product['code2'] : null);
		$p->setExtraFields($product['extraFields']);
		$p->setMaximumAmount($product['maximumAmount']);
		$p->minimumAmount     = (int) $product['minimumAmount'] ?: 1;
		$p->unlimitedQuantity = $product['unlimitedQuantity'] ? 1 : 0;
		$p->shortDescription  = $product['productTexts']['shortDescription'];
		$p->created           = $product['created'];
		$p->modified          = $product['modified'];
		$p->categories        = $product['categories'];
		$p->moreData          = $product['moreData'] ?? [];
		$p->tagsRaw           = $product['tags'] ?? [];
		$p->suppliersIds      = $product['suppliersIds'] ?? [];

		if ($product['variantId']) {
			$p->variantId = (string) $product['variantId'];
		}

		$p->variantName = $product['variantName'];
		$p->helperData  = [
			'variantImage' => $product['variantImage'],
		];

		$p->disableCalculateFreeSpedition = (bool) $product['disableCalculateFreeSpedition'];
		$p->disableDeliveryBoxes          = (bool) ($product['moreData']['disableDeliveryBoxes'] ?? false);
		$p->defaultCategoryId             = $product['defaultCategory'] ? (int) $product['defaultCategory'] : null;
		$p->canAddToCart                  = (bool) $product['canProductsAddToCart'];
		$p->discountDisabled              = (bool) $product['discountDisabled'];
		$p->categoryGiftsAllowed          = (bool) $product['categoryGiftsAllowed'];
		$p->orderGiftsAllowed             = (bool) $product['orderGiftsAllowed'];
		$p->isAssort                      = (bool) $product['isAssort'];
		$p->condition                     = $product['condition'] ?: 'new';
		$p->conditionDescription          = $product['productTexts']['conditionDescription'] ?: null;
		$p->disablePickUpSpedition        = (bool) $product['disablePickUpSpedition'];
		$p->isDiscount                    = (bool) $product['isDiscount'];
		$p->discountType                  = $product['discountType'];
		$p->discountValue                 = $product['discountValue'] ? ((float) $product['discountValue']) : null;
		$p->isOversize                    = (bool) $product['isOversize'];
		$p->disabledSpeditions            = $product['disabledSpeditions'];
		$p->disabledPayments              = $product['disabledPayments'];
		$p->availabilityAfterSoldOut      = $product['availabilityAfterSoldOut']
			? $this->availabilityService->get($product['availabilityAfterSoldOut'])
			: null;
		$p->disableRegisterSale           = (int) $product['disableRegisterSale'];

		if (Config::load('product.allowVerifyAge')) {
			$p->verifyAge = (int) $product['verifyAge'];
		}

		foreach ($product['dynamicFeatures'] as $v) {
			foreach ($v as $k => $v2) {
				$p->features['dynamic_' . $k] = $v2;
			}
		}

		if (Config::load('product.allowPreorder')) {
			$p->preorderText = $product['productTexts']['preorderText'];
		}

		$p->width  = $product['width'];
		$p->height = $product['height'];
		$p->depth  = $product['depth'];
		$p->weight = $product['weight'];

		if ($product['gallery']) {
			$p->galleryId = $product['gallery'];
		}

		if ($product['manufacturer']) {
			$p->manufacturerId = $product['manufacturer'];
		}

		if (isset($product['inSites']) && !empty($product['inSites'])) {
			$p->inSites = $product['inSites'];
		}

		if (isset($product['packageId'])) {
			$p->packageId = (int) $product['packageId'];
		}

		$this->eventDispatcher->dispatch(new DaoEvent($p, $product), Products::class . '::fillDao');

		if ($p instanceof Dao\Product === false) {
			throw new \RuntimeException('Dao class must be instance of ' . Dao\Product::class);
		}

		return $p;
	}

	public function normalizeHydrateArray(array $product): array
	{
		$result                             = $product[0];
		$result['variantId']                = (string) $product['variantId'];
		$result['variantName']              = $product['variantName'];
		$result['variantImage']             = $product['variantImage'] ? (int) $product['variantImage'] : null;
		$result['productTexts']             = array_values($product[0]['productTexts'])[0];
		$result['packageId']                = $product['packageId'] ?? null;
		$result['availabilityAfterSoldOut'] = $product['availabilityAfterSoldOut'];

		foreach (['gallery', 'manufacturer', 'vatRate', 'defaultCategory', 'availability'] as $v) {
			$result[$v] = $product[$v];
		}

		if (isset($product['inSites'])) {
			$sites             = explode(',', (string) $product['inSites']);
			$result['inSites'] = array_unique($sites);
		}

		return $result;
	}

	protected function prepareFilter(array $filter): array
	{
		if (isset($filter['features'])) {
			$count = count($filter['features']);
			$ids   = [];
			$qb    = $this->em->getRepository(FeatureProduct::class)->createQueryBuilder('fp')->select('IDENTITY(fp.product) as product');
			$tmp   = [];

			foreach ($filter['features'] as $group => $vals) {
				$or  = [];
				$qb2 = clone $qb;
				foreach ($vals as $k => $v) {
					$or[] = "IDENTITY(fp.featureValue) = :fpfv_{$group}_{$k}";
					$qb2->setParameter("fpfv_{$group}_{$k}", $v);
				}
				$qb2->andWhere(implode(' OR ', $or));
				foreach ($qb2->getQuery()->getArrayResult() as $row) {
					$tmp[$row['product']] = isset($tmp[$row['product']]) ? $tmp[$row['product']] + 1 : 1;
				}
			}

			foreach ($tmp as $k => $v) {
				if ($v == $count) {
					$ids[] = $k;
				}
			}
			$tmp = null;

			$filter['features'] = $ids === [] ? 'empty' : $ids;
		}

		return $filter;
	}

	/**
	 * @param string|string[]|null $sort
	 */
	public function prepareSort(ProductQuery $query, $sort): void
	{
		if (is_null($sort)) {
			$sort = 'p.price ASC';
		}

		if ($sort === 'recommended') {
			$sort = '';
		}

		if (is_string($sort)) {
			$sort = explode(' ', $sort, 2);
			if (!in_array(strtolower($sort[1]), ['asc', 'desc'])) {
				$sort[1] = 'asc';
			}
			$query->addSortBy($sort[0], $sort[1]);
		} else if (is_array($sort)) {
			foreach ($sort as $k => $v) {
				$query->addSortBy($k, $v);
			}
		}
	}

	/**
	 * @deprecated
	 */
	public function getExtraFieldsByKey(string $key): array
	{
		if ($this->cExtraFields2[$key] === null) {
			$this->cExtraFields2[$key] = [];

			foreach ($this->em->getRepository(ExtraField::class)
				         ->createQueryBuilder('ef')
				         ->select('ef.sectionKey, ef.value')
				         ->where('ef.sectionName = :secName')
				         ->andWhere('ef.key = :key')
				         ->andWhere('ef.lang IS NULL OR ef.lang = :lang')
				         ->setParameters(new ArrayCollection([new Parameter('lang', $this->translator->getLocale()), new Parameter('secName', Product::EXTRA_FIELD_SECTION), new Parameter('key', $key)]))
				         ->getQuery()
				         ->getScalarResult() as $row) {
				$this->cExtraFields2[$key][$row['sectionKey']] = $row['value'];
			}
		}

		return $this->cExtraFields2[$key];
	}

	public function getExtraFieldsByIds(array $ids): array
	{
		$whereIds = [];
		$result   = [];
		$keys     = [];

		foreach ($ids as $id) {
			if (isset($this->cIdsExtraFields[$id])) {
				$result[$id] = $this->cIdsExtraFields[$id];
			} else {
				$keys[] = 'pExtraField/' . $id;
			}
		}

		if ($keys !== []) {
			foreach ($this->cacheService->defaultCache->bulkLoad($keys) as $key => $value) {
				$tmp = explode('/', $key);
				$id  = $tmp[1];

				if ($value) {
					$result[$id] = $value;
				} else {
					$whereIds[] = $id;
				}
			}
		}

		if ($whereIds !== []) {
			// WIP mozna optimalizace?
			//			$toCache = [];
			//			foreach ($this->em->getRepository(ExtraField::class)->createQueryBuilder('ef')
			//				         ->select('ef.sectionKey, ef.key, ef.value')
			//				         ->where('ef.sectionName = :secName')
			//				         ->andWhere('ef.lang IS NULL OR ef.lang = :lang')
			//				         ->setParameters([
			//					         'lang'    => $this->translator->getLocale(),
			//					         'secName' => Product::EXTRA_FIELD_SECTION,
			//				         ])->getQuery()->getArrayResult() as $row) {
			//				if (Arrays::contains($whereIds, $row['sectionKey'])) {
			//					$result[$row['sectionKey']][$row['key']] = $row['value'];
			//				}
			//
			//				$toCache[$row['sectionKey']][$row['key']] = $row['value'];
			//			}
			//
			//			foreach ($toCache as $k => $v) {
			//				$this->cacheService->defaultCache->save(
			//					'pExtraField/' . $k,
			//					$v,
			//					[
			//						Cache::EXPIRATION => '30 minutes',
			//					]
			//				);
			//			}

			foreach (array_chunk($whereIds, 200) as $chunk) {
				$query = $this->em->getRepository(ExtraField::class)->createQueryBuilder('ef')
					->select('ef.sectionKey, ef.key, ef.value')
					->where('ef.sectionName = :secName')
					->andWhere('ef.lang IS NULL OR ef.lang = :lang')
					->andWhere('ef.sectionKey IN (' . implode(',', $chunk) . ')')
					->setParameters(new ArrayCollection([new Parameter('lang', $this->translator->getLocale()), new Parameter('secName', Product::EXTRA_FIELD_SECTION)]))->getQuery();

				foreach ($query->getScalarResult() as $row) {
					$result[$row['sectionKey']][$row['key']] = $row['value'];
				}

				foreach ($chunk as $value) {
					$this->cacheService->defaultCache->save(
						'pExtraField/' . $this->translator->getLocale() . '/' . $value,
						$result[$value] ?: [],
						[
							Cache::EXPIRATION => '10 minutes',
						]
					);
				}
			}
		}

		return $result;
	}

	/**
	 * @deprecated
	 */
	public function getAllExtraFields(): array
	{
		if ($this->cExtraFields === null) {
			$lang = $this->translator->getLocale();

			$this->cExtraFields = $this->cacheService->defaultCache->load('productsExtraFields_' . $lang, function(&$dep) use ($lang) {
				$dep = [
					Cache::EXPIRATION => '10 minutes',
				];

				$arr   = [];
				$query = $this->em->getRepository(ExtraField::class)->createQueryBuilder('ef')
					->select('ef.sectionKey, ef.key, ef.value')
					->where('ef.sectionName = :secName')
					->andWhere('ef.lang IS NULL OR ef.lang = :lang')
					->setParameters(new ArrayCollection([new Parameter('lang', $lang), new Parameter('secName', Product::EXTRA_FIELD_SECTION)]))->getQuery();

				foreach ($query->getScalarResult() as $row) {
					$arr[$row['sectionKey']][$row['key']] = $row['value'];
				}

				return $arr;
			});
		}

		return $this->cExtraFields;
	}

	public static function setMode(string $mode): void { self::$mode = $mode; }

}
