<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Contributte\Translation\Translator;
use Core\Model\Countries;
use Core\Model\Event\EventDispatcher;
use Core\Model\Sites;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\Query\Parameter;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\Dao\RelatedGroup;
use EshopCatalog\FrontModule\Model\Event\ProductsEvent;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\RelatedProduct;
use Gallery\FrontModule\Model\Albums;
use Gallery\FrontModule\Model\Dao\Album;
use Gallery\FrontModule\Model\Dao\Image;
use Nette\Application\LinkGenerator;
use Nette\Caching\Cache;
use Nette\Utils\DateTime;
use Throwable;
use Users\Model\Security\User;

class ProductsFacade
{
	public const MODE_ESHOP    = 'eshop';
	public const MODE_CHECKOUT = 'checkout';

	protected static string $mode            = self::MODE_ESHOP;
	public static ?string   $forceLocale     = null;
	public static ?string   $forceLinkLocale = null;
	public static ?bool     $cacheEnabled    = true;
	public static bool      $loadVariants    = true;
	public LinkGenerator    $linkGenerator;

	/** @var Dao\Product[] */
	protected array $cProducts;

	/** @var Dao\Product[] */
	protected array $cVariants;

	public array        $cFreashLoaded = [];
	public static array $allLoadedIds  = [];
	public static bool  $loadFeatures  = true;

	public static bool $ignoreIsActive = false;

	public function __construct(
		protected Translator              $translator,
		protected CacheService            $cacheService,
		public Products                   $productsService,
		protected EventDispatcher         $eventDispatcher,
		protected User                    $user,
		protected Albums                  $albumsService,
		protected Manufacturers           $manufacturersService,
		protected Tags                    $tagsService,
		protected Sites                   $sitesService,
		public Categories                 $categoriesService,
		protected DynamicFeaturesProducts $dynamicFeaturesProducts,
		protected Countries               $countries,
		protected Packages                $packages,
		protected AvailabilityService     $availabilityService,
	)
	{
	}

	public function loadBase(array $ids): array
	{
		$ids      = array_unique($ids);
		$whereIds = [];
		$loaded   = [];
		$result   = [];
		$locale   = $this->getLocale();

		foreach ($ids as $k => $id) {
			if (isset($this->cProducts[$id])) {
				$loaded[$id] = $this->cProducts[$id];
				unset($ids[$k]);
			}
		}

		if (self::$cacheEnabled) {
			$keys = [];
			foreach ($ids as $id) {
				$keys[] = 'product/' . $id . '/' . $locale;
			}

			// Načtení z cache
			if ($keys !== []) {
				foreach ($this->cacheService->productCache->bulkLoad($keys) as $key => $product) {
					$tmp = explode('/', $key);
					$id  = $tmp[1];

					if ($product instanceof Product) {
						$result[$id] = $product;
					} else if ($product === null || (self::$ignoreIsActive && $product === false)) {
						$whereIds[] = $id;
					}
				}
			}
		} else {
			$whereIds = array_values($ids);
		}

		/** @var int[] $whereIds */
		$whereIds = array_filter($whereIds, static fn($v) => is_numeric($v) && $v > 0);

		// Vytvoření získání produktů co nejsou v cache
		if ($whereIds !== []) {
			$conn  = $this->productsService->em->getConnection();
			$query = (new ProductQuery($locale))
				->withTexts()
				->withVatRate()
				->loadSites()
				->withVariantParent()
				->hasPrice()
				->byId($whereIds);

			$rootCats = $this->categoriesService->getAllRootIds();

			$cats = [];
			foreach ($conn->executeQuery("SELECT cp.id_product, cp.id_category 
						FROM eshop_catalog__category_product cp 
						WHERE cp.id_product IN (" . implode(',', $whereIds) . ")")->iterateAssociative() as $cat) {
				$cats[$cat['id_product']][] = $cat['id_category'];
			}

			$qb = $query->getQueryBuilder($this->productsService->getEr());
			$qb->groupBy('p.id');

			foreach ($this->productsService->customQB as $cqb) {
				$cqb($qb);
			}

			$tags = [];
			foreach ($conn->executeQuery("SELECT id_product as product, id_tag as tag, valid_from as validFrom, valid_to as validTo
						FROM eshop_catalog__product_tag 
						WHERE id_product IN (" . implode(',', $whereIds) . ")")->iterateAssociative() as $row) {

				if ($row['validFrom']) {
					$row['validFrom'] = DateTime::from($row['validFrom']);
				}

				if ($row['validTo']) {
					$row['validTo'] = DateTime::from($row['validTo']);
				}

				$tags[$row['product']][] = $row;
			}

			$disabledSpeds = [];
			if (Config::load('product.allowModifySpedition')) {
				foreach ($conn->executeQuery("SELECT product_id, spedition_id
						FROM eshop_catalog__product_spedition
						WHERE product_id IN (" . implode(',', $whereIds) . ") AND delivery_disabled = 1")->iterateAssociative() as $row) {
					$disabledSpeds[$row['product_id']][] = $row['spedition_id'];
				}
			}

			$disabledPayments = [];
			if (Config::load('product.allowModifyPayment')) {
				foreach ($conn->executeQuery("SELECT product_id, payment_id
						FROM eshop_catalog__product_payment
						WHERE product_id IN (" . implode(',', $whereIds) . ") AND payment_disabled = 1")->iterateAssociative() as $row) {
					$disabledPayments[$row['product_id']][] = $row['payment_id'];
				}
			}

			if (Config::load('product.allowCountryOfOrigin')) {
				$qb->addSelect('IDENTITY(p.countryOfOrigin) as countryOfOrigin');
			}

			if (Config::load('product.allowPackage', false)) {
				$qb->addSelect('IDENTITY(p.package) as packageId');
			}

			$qbResult   = $qb->getQuery()->getArrayResult();
			$idsForLink = [];

			if (self::$cacheEnabled) {
				foreach (array_diff($whereIds, array_keys($qbResult)) as $notFound) {
					$cacheDep                = $this->productsService->cacheDep;
					$cacheDep[Cache::Tags][] = 'product/' . $notFound;
					$cacheDep[Cache::Expire] = '1 week';
					$this->cacheService->productCache->save('product/' . $notFound . '/' . $locale, false, $cacheDep);
				}
			}

			foreach ($qbResult as $row) {
				$idsForLink[] = $row[0]['id'];
			}

			self::$allLoadedIds = $idsForLink;

			$suppliers = [];
			if (!empty($idsForLink)) {
				foreach ($conn->fetchAllAssociative("SELECT id_product, id_supplier FROM eshop_catalog__product_supplier WHERE id_product IN (" . implode(',', $idsForLink) . ")") as $row) {
					/** @var array $row */
					$suppliers[$row['id_product']][] = $row['id_supplier'];
				}
			}

			$featureValuesIds = $this->productsService->featureProductsService->getFeatureValuesIdsForProducts($idsForLink);
			$dynamicFeatures  = $this->dynamicFeaturesProducts->getDaoByProducts($idsForLink);
			$extraFields      = $this->productsService->getExtraFieldsByIds($idsForLink);

			foreach ($qbResult as $row) {
				$row[0]['categories']      = $cats[$row[0]['id']] ?? [];
				$tmp                       = $this->productsService->normalizeHydrateArray($row);
				$tmp['extraFields']        = $extraFields[$tmp['id']] ?: [];
				$tmp['disabledSpeditions'] = $disabledSpeds[$row[0]['id']] ?? [];
				$tmp['disabledPayments']   = $disabledPayments[$row[0]['id']] ?? [];
				$tmp['dynamicFeatures']    = $dynamicFeatures[$tmp['id']] ?? [];
				$tmp['tags']               = $tags[$tmp['id']] ?? [];
				$tmp['suppliersIds']       = $suppliers[$row[0]['id']] ?? [];

				if ($row['countryOfOrigin']) {
					$tmp['countryOfOrigin'] = $this->countries->getDao()[$row['countryOfOrigin']];
				}

				$dao                   = $this->productsService->fillDao($tmp);
				$dao->featureValuesIds = $featureValuesIds[$dao->getId()] ?? [];

				foreach ($dao->inSites as $v) {
					if (isset($rootCats[$v])) {
						array_unshift($dao->categories, $rootCats[$v]);
					}
				}

				if (Config::load('product.publishedByLang')) {
					if ($row[0]['productTexts'][$locale]['isPublished'] === 0) {
						$dao->isActive = false;
					}
				} else if (isset($row[0]['isPublished']) && $row[0]['isPublished'] === 0) {
					$dao->isActive = false;
				}

				if (self::$cacheEnabled) {
					$cacheDep                = $this->productsService->cacheDep;
					$cacheDep[Cache::Tags][] = 'product/' . $dao->getId();
					$cacheDep[Cache::Expire] = '1 week';
					$this->cacheService->productCache->save('product/' . $dao->getId() . '/' . $locale, $dao, $cacheDep);
				}

				$result[$tmp['id']] = $dao;
			}
		}

		return [
			'prepared' => $result ?: [],
			'loaded'   => $loaded ?: [],
		];
	}

	/**
	 * Vrátí DAO produktů z cache nebo získá data z db a uloží do cache
	 *
	 * @param int[]|string[] $ids
	 *
	 * @return Dao\Product[]
	 * @throws Throwable
	 */
	public function getProducts(array $ids, ?string $siteIdent = null, bool $loadVariants = true, bool $loadFresh = true): array
	{
		$baseIds = $ids;
		$data    = $this->loadBase($ids);

		/** @var Dao\Product[] $loaded */
		$loaded = $data['loaded'];

		/** @var Dao\Product[] $result */
		$result = $data['prepared'];

		$userId = $this->user->getIdentity() ? $this->user->getIdentity()->getId() : null;

		if (!empty($result)) {
			$variants         = [];
			$variantParentIds = [];
			$variantIds       = [];

			if ($loadFresh && Config::load('groupVariantsInList', false)) {
				$this->productsService->loadFresh(
					$result,
					$userId,
					true
				);

				$this->cFreashLoaded = array_merge($this->cFreashLoaded, array_keys($result));
			}

			foreach ($result as $product) {
				if ($product->variantId) {
					$variantIds[$product->getId()] = $product->variantId;
				}

				if ($product->packageId) {
					$package = $this->packages->getPackages([$product->packageId])[$product->packageId] ?? null;
					if ($package) {
						$product->package = $package;

						$packageProductIds = array_map(static fn($v): int => $v->productId, $package->items);
						$packageProducts   = $this->getProducts($packageProductIds);
						foreach ($package->items as $item) {
							if ($packageProducts[$item->productId]) {
								$item->product = $packageProducts[$item->productId];
							}
						}

						$this->packages->setMaximumQuantity($product);

						if ($product->getQuantity() <= 0) {
							$product->setAvailability($this->availabilityService->getByIdent(Availability::SOLD_OUT));
						}
					}
				}
			}

			if ($variantIds !== []) {
				$tmp         = [];
				$resultCache = (int) Config::loadScalar('productsList.enableResultCache');

				$getVariantData = function() use ($resultCache, $variantIds) {
					$conn = $this->productsService->em->getConnection();
					$sql  = "SELECT product_id as product, variant_id as variantId, is_default as isDefault
						FROM eshop_catalog__product_variant
						WHERE variant_id IN ('" . implode("','", array_unique($variantIds)) . "')";

					if ($resultCache) {
						$cacheKey = 'facadeVariants/' . md5(implode(',', $variantIds));

						foreach ($this->cacheService->productCache->load($cacheKey, function(&$dep) use ($resultCache, $conn, $sql) {
							$dep = [Cache::Expire => $resultCache . ' seconds'];

							return $conn->executeQuery($sql)->fetchAllAssociative();
						}) as $v) {
							yield $v;
						}
					} else {
						foreach ($conn->executeQuery($sql)->iterateAssociative() as $v) {
							yield $v;
						}
					}
				};

				foreach ($getVariantData() as $row) {
					if ($row['isDefault'] == 1) {
						$variantParentIds[$row['variantId']] = $row['product'];
					}

					if (isset($variantIds[$row['variantId']]) && !is_array($variantIds[$row['variantId']])) {
						$variantIds[$row['variantId']] = [$variantIds[$row['variantId']]];
					}

					$tmp[$row['variantId']][] = $row['product'];
					$variants[]               = $row['product'];
				}
				$variantIds = $tmp;

				if (self::$loadVariants && $variants !== []) {
					self::$loadVariants = false;
					$tmp                = $this->getProducts(array_diff(array_unique($variants), $ids));
					self::$loadVariants = true;
					$result             += $tmp;
				}
			}

			foreach ($result as $product) {
				if (isset($variantParentIds[$product->variantId])) {
					$product->variantOf = (string) $variantParentIds[$product->variantId];
				}
			}

			$this->tagsService->loadTagsToProduct($result);

			if (!Config::load('groupVariantsInList', false)) {
				$this->productsService->loadFresh(
					$result,
					$userId,
					true
				);

				$this->cFreashLoaded = array_merge($this->cFreashLoaded, array_keys($result));
			}

			$now = (new DateTime)->format('Y-m-d H:i:00');
			foreach ($result as $k => $product) {
				foreach ($product->tagsRaw as $tagK => $tagV) {
					if (
						($tagV['validFrom'] === null || $tagV['validFrom']->format('Y-m-d H:i:00') <= $now)
						&& ($tagV['validTo'] === null || $tagV['validTo']->format('Y-m-d H:i:00') >= $now)
					) {
						$tag                       = $this->tagsService->get($tagV['tag']);
						$product->tags[$tag->type] = $tag;
					} else {
						unset($product->tagsRaw[$tagK]);
					}
				}

				uasort($product->features, static fn($a, $b): int => $a->position <=> $b->position);
			}

			$galIds = [];
			foreach ($result as $product) {
				if ($product->galleryId) {
					$galIds[] = $product->galleryId;
				}
			}
			$galleries = $this->albumsService->getAlbums($galIds, 'eshopCatalog%siteIdent%PlaceholderImage');

			$forLoadFeatures = $result;

			for ($i = 0; $i <= 1; $i++) {
				foreach ($result as $id => $product) {
					if (!$product->isActive && !self::$ignoreIsActive) {
						unset($result[$id]);
						continue;
					}

					if ($i === 0 && $product->variantOf && $product->variantOf !== $product->getId()) {
						continue;
					} else if ($i === 1 && !$product->variantOf) {
						continue;
					}

					if ($product->galleryId) {
						/** @var Album|null $gallery */
						$gallery = isset($galleries[$product->galleryId]) ? clone $galleries[$product->galleryId] : null;

						if ($gallery) {
							if ($product->helperData['variantImage']) {
								/** @var Image|null $img */
								$img = $gallery->getImages()[$product->helperData['variantImage']];

								foreach ($gallery->getImages() as $image) {
									$image->isCover = $img->id == $image->id;
								}
								if ($img) {
									$gallery->setCover($img);
								}
							}

							if ($gallery->getCover()) {
								$product->variantImage = $gallery->getCover();
							}

							$product->setGallery($gallery);
						}
					}

					if ($product->manufacturerId) {
						$product->setManufacturer($this->manufacturersService->get((int) $product->manufacturerId));
					}

					if (!$product->defaultCategoryId && isset($this->cProducts[$product->variantOf])) {
						$product->defaultCategoryId = $this->cProducts[$product->variantOf]->defaultCategoryId;
					}

					if ($product->defaultCategoryId) {
						$product->defaultCategory = $this->categoriesService->get($product->defaultCategoryId);
					}

					if ($product->defaultCategory) {
						$product->canAddToCart = (bool) $product->defaultCategory->canProductsAddToCart;
					}

					if (($product->canAddToCart || !$product->defaultCategoryId) && $product->getAvailability()) {
						$product->canAddToCart = (bool) $product->getAvailability()->canAddToCart();
					}

					$this->cProducts[$id] = $product;
					$loaded[$id]          = $product;
					$result[$id]          = null;
				}
			}

			if (self::$loadFeatures) {
				$this->productsService->loadFeatures($forLoadFeatures);
			}

			$this->eventDispatcher->dispatch(new ProductsEvent($loaded), Products::class . '::afterFillDao');

			$allLoaded          = $loaded;
			self::$allLoadedIds = array_keys($allLoaded);

			/** @phpstan-ignore-next-line */
			if (Config::load('canCacheProductsUrl') && $this->linkGenerator) {
				$linkKeys   = [];
				$whereLinks = [];
				$locale     = self::$forceLinkLocale ?: $this->getLocale();

				foreach ($allLoaded as $id => $product) {
					$linkKeys[] = 'link/' . $id . '/' . $locale;
				}

				foreach ($this->cacheService->productCache->bulkLoad($linkKeys) as $key => $link) {
					$tmp = explode('/', $key);
					$id  = $tmp[1];

					if ($link) {
						$allLoaded[$id]->link = $link;
					} else {
						$whereLinks[] = $id;
					}
				}

				foreach ($whereLinks as $id) {
					$product = $allLoaded[$id] ?? null;

					if (!$product) {
						continue;
					}

					$product->link = $this->generateLink($product, $locale);

					$cacheDep = [
						Cache::Tags   => 'eshopNavigation',
						Cache::Expire => '1 week',
					];
					$this->cacheService->productCache->save('link/' . $product->getId() . '/' . $locale, $product->link, $cacheDep);
				}
			}

			foreach ($allLoaded as $id => $product) {

				/** @phpstan-ignore-next-line */
				if (!Config::load('canCacheProductsUrl') && $this->linkGenerator) {
					$product->link = $this->generateLink($product, self::$forceLinkLocale ?: $this->getLocale());
				}

				$this->cProducts[$id] = $product;
			}

			foreach (array_diff(array_keys($loaded), $baseIds) as $v) {
				unset($loaded[$v]);
			}

			if (self::$mode !== self::MODE_CHECKOUT) {
				foreach ($loaded as $product) {
					if (!$product->variantId) {
						continue;
					}

					foreach ($variantIds[$product->variantId] as $v) {
						if ($allLoaded[$v]) {
							$product->variants[$v] = $allLoaded[$v];
						}
					}

					if (count($product->variants) <= 1) {
						$product->variants  = [];
						$product->variantId = null;
					} else {
						foreach ($product->variants as $k => $v) {
							$product->variants[$k]->variants = &$product->variants;
						}
					}
				}
			}
		}

		$result = [];
		foreach ($ids as $id) {
			if (isset($loaded[$id])) {
				$result[$id] = $loaded[$id];
			}
		}

		$loadFreshArr = [];
		foreach (array_diff(array_keys($result), $this->cFreashLoaded) as $id) {
			$loadFreshArr[$id] = &$result[$id];
		}

		if (!empty($loadFreshArr) && $loadFresh) {
			$this->productsService->loadFresh(
				$loadFreshArr,
				$userId,
				true
			);
		}

		return $result;
	}

	/**
	 * @throws Throwable
	 */
	public function getProduct(int $id, ?string $siteIdent = null): ?Product
	{
		return $this->getProducts([$id], $siteIdent)[$id] ?? null;
	}

	public function loadAlternative(Product $product, int $limit): void
	{
		$data       = $this->productsService->getProductsIdInCategory($product->defaultCategoryId, 0, null, null, [], false);
		$variantIds = array_keys($product->variants);

		foreach ($data as $k => $v) {
			if ($product->getId() == $v || in_array($v, $variantIds)) {
				unset($data[$k]);
			}
		}

		shuffle($data);
		$products = $this->getProducts(array_slice($data, 0, $limit));

		$product->setAlternatives($products);
	}

	public function loadRelated(Product $product, int $limit): void
	{
		if (!Config::load('allowRelatedProducts') || $product->getRelated() !== []) {
			return;
		}

		[$related, $ids, $groupedIds] = $this->cacheService->productCache
			->load('related/' . $product->getId(), function(&$dep) use ($product) {
				$dep = [
					Cache::Tags => ['related'],
				];

				$related    = [];
				$ids        = [];
				$groupedIds = [];
				foreach ($this->productsService->em->getRepository(RelatedProduct::class)->createQueryBuilder('rp')
					         ->select('IDENTITY(rp.product) as product, g.id as groupId, gt.name as groupName')
					         ->innerJoin('rp.group', 'g', Join::WITH, 'g.isPublished = 1')
					         ->innerJoin('g.texts', 'gt', Join::WITH, 'gt.lang = :lang')
					         ->andWhere('rp.origin = :id')
					         ->orderBy('g.position')
					         ->setParameters(new ArrayCollection([new Parameter('lang', $this->getLocale()), new Parameter('id', $product->getId())]))->getQuery()->getArrayResult() as $row) {
					if (!isset($related[$row['groupId']])) {
						$related[$row['groupId']] = new RelatedGroup($row['groupId'], $row['groupName']);
					}
					$ids[]                         = $row['product'];
					$groupedIds[$row['groupId']][] = $row['product'];
				}

				return [$related, $ids, $groupedIds];
			});

		if ($ids) {
			Products::$loadKey = 'related';
			$products          = $this->getProducts($ids);

			foreach ($groupedIds as $groupId => $ids) {
				shuffle($ids);
				$i = 0;

				foreach ($ids as $id) {
					if (!isset($products[$id])) {
						continue;
					}

					$related[$groupId]->addProduct($products[$id]);
					$i++;

					if ($i >= $limit) {
						break;
					}
				}
			}

			$product->setRelated($related);
			Products::$loadKey = null;
		}
	}

	public static function setMode(string $mode): void
	{
		self::$mode = $mode;
		Products::setMode(Products::MODE_CHECKOUT);
		Categories::setMode(Categories::MODE_CHECKOUT);
	}

	protected function getLocale(): string { return self::$forceLocale ?: $this->translator->getLocale(); }

	public function updateQuantitiesByIds(array $ids): void
	{
		$forUpdate = [];
		foreach ($ids as $id) {
			$product = $this->cProducts[$id] ?? null;

			if (!$product) {
				continue;
			}

			$forUpdate[$id] = &$product;
		}

		$this->productsService->loadQuantity($forUpdate, true);
	}

	protected function generateLink(Product $product, string $locale): string
	{
		$linkParams = [
			'id'     => $product->getId(),
			'locale' => $locale,
		];

		if (Config::load('useOldUrls') && $product->getExtraField((string) Config::loadScalar('product.oldUrlEFKey')) !== null
			&& $product->variantOf && $product->variantOf != $product->getId()) {
			$linkParams['variant'] = $product->getId();
		}

		return $this->linkGenerator->link('EshopCatalog:Front:Default:product', $linkParams);
	}

	public function clearTemp(): void
	{
		$this->cProducts     = [];
		$this->cVariants     = [];
		$this->cFreashLoaded = [];
	}

}
