<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model\Subscribers;

use Core\Model\Application\AppState;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\Event;
use Core\Model\Sites;
use Core\Model\UI\FrontPresenter;
use EshopCatalog\FrontModule\Components\ProductPreview;
use EshopCatalog\FrontModule\Model\AvailabilityService;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use Exception;
use Nette\Caching\Cache;
use Nette\Utils\Json;
use Override;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use voku\helper\HtmlDomParser;
use voku\helper\SimpleHtmlDomInterface;

class ContentSubscriber implements EventSubscriberInterface
{
	public function __construct(
		protected ProductsFacade         $productsFacade,
		protected EntityManagerDecorator $em,
		protected Sites                  $sites,
		protected CacheService           $cacheService,
		protected AvailabilityService    $availabilityService,
	)
	{
	}

	#[Override]
	public static function getSubscribedEvents(): array
	{
		return [
			'editorData.processString' => 'processString',
		];
	}

	public function processString(Event $event): void
	{
		if (!($event->data['string'] ?? null)) {
			return;
		}

		/** @var string $html */
		$html = &$event->data['string'];
		/** @var ?FrontPresenter $presenter */
		$presenter = AppState::getState('presenter');

		if (!$html || !$presenter) {
			return;
		}

		$htmlDom   = HtmlDomParser::str_get_html($html);
		$allIds    = [];
		$toReplace = [];

		// Box produktu podle id
		foreach (self::findEshopCatalogProduct($htmlDom) as $row) {
			try {
				$ids = self::findEshopCatalogProductData($row);

				foreach ($ids as $id) {
					$allIds[] = $id;
				}

				$toReplace[] = [
					'html' => $row->outerhtml,
					'ids'  => $ids,
				];
			} catch (Exception) {
				$toReplace[] = [
					'html' => $row->outerhtml,
					'ids'  => [],
				];
			}
		}

		// Box produktu podle kategorii a vlastnosti
		foreach (self::findEshopCatalogProductFromCat($htmlDom) as $row) {
			try {
				$limit = (int) ($row->getAttribute('data-limit') ?: 3);
				$ids   = self::findEshopCatalogProductFromCatData($this->em, $row, $this->cacheService, $this->availabilityService);

				if ($ids === []) {
					throw new Exception('No ids');
				}

				shuffle($ids);
				$ids = array_slice($ids, 0, $limit);

				foreach ($ids as $id) {
					$allIds[] = $id;
				}

				$toReplace[] = [
					'html' => $row->outerhtml,
					'ids'  => $ids,
				];
			} catch (Exception) {
				$toReplace[] = [
					'html' => $row->outerhtml,
					'ids'  => [],
				];
			}
		}

		// Nahrazení boxů produktů
		if ($toReplace !== []) {
			$products = $allIds !== [] ? $this->productsFacade->getProducts($allIds) : [];

			foreach ($toReplace as $item) {
				$item['html'] = str_replace(["\r\n", "\r", "\n"], '', $item['html']);
				$replaceHtml  = '';

				foreach ($item['ids'] as $id) {
					$product = $products[$id] ?? null;

					if ($product) {
						ob_start();
						/** @var ProductPreview $control */
						$control = $presenter->getComponent('productPreview-' . $id);
						$control->setProduct($product);
						$control->render();
						$replaceHtml .= ob_get_clean();
					}
				}

				if ($replaceHtml !== '' && $replaceHtml !== '0') {
					$replaceHtml = '<div class="products-auto-grid">' . $replaceHtml . '</div>';
				}

				$html = str_replace($item['html'], $replaceHtml, $html);
			}
		}
	}

	protected static function buildFeatureWhere(array $data, array &$params, string $field_name = 'fp.id_feature_value'): string
	{
		if (isset($data['op']) && !empty($data['children'] ?? [])) {
			$logic    = strtoupper((string) $data['op']);
			$children = [];

			foreach ($data['children'] as $child) {
				$children[] = self::buildFeatureWhere($child, $params, $field_name);
			}

			$children = array_filter($children, static fn($child): bool => (bool) strlen((string) $child));
			if ($children === []) {
				return '';
			}

			return '(' . implode(' ' . $logic . ' ', $children) . ')';
		}

		if (isset($data['operator'], $data['value'])) {
			$placeholder = '?';

			$params[] = is_numeric($data['value']) ? (int) $data['value'] : $data['value'];

			if ($data['operator'] === '=') {
				return "$field_name = $placeholder";
			}

			if ($data['operator'] === '!=') {
				return "$field_name != $placeholder";
			}
		}

		return '';
	}

	public function clearEditorDataCache(): void
	{
		$this->cacheService->productCache->clean([Cache::TAGS => ['editorData']]);
	}

	/**
	 * @return array<int, SimpleHtmlDomInterface>
	 */
	public static function findEshopCatalogProduct(HtmlDomParser $htmlDom): array
	{
		return iterator_to_array($htmlDom->find('.eshopcatalogproduct')->getIterator());
	}

	public static function findEshopCatalogProductData(SimpleHtmlDomInterface $row): array
	{
		$ids = [];
		foreach (Json::decode(rawurldecode(base64_decode($row->getAttribute('data-products'))), Json::FORCE_ARRAY) as $v) {
			$ids[] = (int) $v['id'];
		}

		return $ids;
	}

	/**
	 * @return array<int, SimpleHtmlDomInterface>
	 */
	public static function findEshopCatalogProductFromCat(HtmlDomParser $htmlDom): array
	{
		return iterator_to_array($htmlDom->find('.eshopcatalogproductfromcat')->getIterator());
	}

	public static function findEshopCatalogProductFromCatData(
		EntityManagerDecorator $em,
		SimpleHtmlDomInterface $row,
		CacheService           $cacheService,
		AvailabilityService    $availabilityService
	): array
	{
		$dataCategories    = $row->getAttribute('data-categories');
		$dataManufacturers = $row->getAttribute('data-manufacturers');
		$dataFeatures      = $row->getAttribute('data-features');

		$cacheKey = 'eshopcatalogproductfromcat/cat-' . $dataCategories . '/manu-' . $dataManufacturers . '/features-' . $dataFeatures;

		return $cacheService->productCache->load($cacheKey, function(&$dep) use ($dataCategories, $dataFeatures, $dataManufacturers, $availabilityService, $em) {
			$dep = [
				Cache::Tags   => ['editorData'],
				Cache::Expire => '1 hour',
			];

			$avs          = [];
			$featuresJoin = '';

			foreach ($availabilityService->getAll() as $av) {
				if ($av->canAddToCart()) {
					$avs[] = $av->getId();
				}
			}

			$categories = array_map(static fn($v): int => (int) $v['id'], Json::decode(rawurldecode(base64_decode($dataCategories)), Json::FORCE_ARRAY));
			if ($categories === []) {
				return [];
			}

			$params        = [];
			$categoriesIds = implode(',', $categories);

			$where = "(pis.category_id IN ({$categoriesIds}) OR cp.id_category IN ({$categoriesIds}))";

			if ($avs !== []) {
				$where .= " AND p.id_availability IN (" . implode(",", $avs) . ")";
			}

			if ($dataFeatures !== '' && $dataFeatures !== '0') {
				$features = Json::decode(rawurldecode(base64_decode($dataFeatures)), Json::FORCE_ARRAY);

				$whereSql = self::buildFeatureWhere($features, $params);

				if ($whereSql !== '' && $whereSql !== '0') {
					$featuresJoin = 'INNER JOIN eshop_catalog__feature_product fp ON fp.id_product = p.id AND ' . $whereSql;
				}
			}

			if ($dataManufacturers !== '' && $dataManufacturers !== '0') {
				$manufacturers = array_map(static fn($v): int => (int) $v['id'], Json::decode(rawurldecode(base64_decode($dataManufacturers)), Json::FORCE_ARRAY));
				if ($manufacturers !== []) {
					$where .= ' AND p.id_manufacturer IN (' . implode(',', $manufacturers) . ')';
				}
			}

			$ids = [];
			foreach ($em->getConnection()->executeQuery("SELECT DISTINCT p.id FROM eshop_catalog__product p
						LEFT JOIN eshop_catalog__product_in_site pis ON pis.product_id = p.id AND pis.is_active = 1
						LEFT JOIN eshop_catalog__category_product cp ON cp.id_product = p.id
                        {$featuresJoin}
						WHERE {$where}", $params)->iterateAssociative() as $cp) {
				$ids[] = (int) $cp['id'];
			}

			return $ids;
		});
	}
}
