<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Sites;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Tag;
use EshopOrders\Model\Helpers\EshopOrdersCache;
use Nette\Caching\Cache;
use Throwable;

class Tags extends BaseFrontEntityService
{
	public const cacheNamespace = 'eshopCatalogTags';

	protected $entityClass = Tag::class;

	protected ?Cache $productsCache = null;

	protected ?array $cTags            = null;
	protected ?float $cFreeFrom        = null;
	protected ?array $cTagsForProducts = null;
	protected ?array $freeFromArr      = null;
	protected array  $cacheDep         = [
		Cache::Tags   => [self::cacheNamespace, Products::CACHE_NAMESPACE],
		Cache::Expire => '1 week',
	];

	public function __construct(
		protected Sites            $sites,
		protected EshopOrdersCache $eshopOrdersCache,
		protected CacheService     $cacheService,
	)
	{
	}

	/** @param Dao\Product[] $products */
	public function loadTagsToProduct(array $products): void
	{
		$tagForPrice  = (string) Config::loadScalar('tagWhenPriceIsLowerThenRetail');
		$freeDelivery = $this->get('freeDelivery') instanceof \EshopCatalog\FrontModule\Model\Dao\Tag ? clone $this->get('freeDelivery') : null;

		foreach ($products as $prod) {
			$freeDeliveryMinimum = $this->getFreeDeliveryMinimumPrice($prod->disabledSpeditions ?: []);

			if ($tagForPrice && Config::load('product.allowRetailPrice') && $prod->getPrice() < $prod->getRetailPrice()) {
				$prod->tags[$tagForPrice] = $this->get($tagForPrice);
			}

			if ($freeDeliveryMinimum !== null && $freeDelivery && $prod->getPrice() >= $freeDeliveryMinimum) {
				$freeDelivery->isAuto       = 1;
				$prod->tags['freeDelivery'] = $freeDelivery;
			}
		}
	}

	/**
	 * @param int|string $id
	 *
	 * @throws Throwable
	 */
	public function get($id): ?Dao\Tag
	{
		return $this->getAll()[$id] ?? null;
	}

	/**
	 * @return Dao\Tag[]
	 * @throws Throwable
	 */
	public function getAll(): array
	{
		if ($this->cTags === null) {
			$lang        = $this->translator->getLocale();
			$this->cTags = $this->cacheService->tagsCache->load('allTags/' . $lang, function(&$dep) use ($lang) {
				$dep  = [
					Cache::TAGS       => [self::cacheNamespace],
					Cache::EXPIRATION => '1 month',
				];
				$data = [];

				foreach ($this->em->getConnection()->executeQuery("SELECT t.id, t.type, t.image, t.color, t.bg_Color as bgColor, t.hide_product_tag_name as hideProductTagName, tt.name, tt.description
						FROM eshop_catalog__tag t
						INNER JOIN eshop_catalog__tag_texts tt ON t.id = tt.tag AND tt.lang = ?", [$lang])->iterateAssociative() as $row) {
					$tag                = $this->fillDao($row);
					$data[$row['type']] = $tag;
					$data[$row['id']]   = $tag;
				}

				return $data;
			});
		}

		return $this->cTags ?? [];
	}

	protected function fillDao(array $tag): Dao\Tag
	{
		$dao              = new Dao\Tag($tag['id'], $tag['type']);
		$dao->name        = $tag['name'] ?? null;
		$dao->image       = $tag['image'] ?? null;
		$dao->bgColor     = $tag['bgColor'] ?? null;
		$dao->description = $tag['description'] ?? null;
		$dao->textColor   = $tag['color'] ?? null;

		return $dao;
	}

	protected function loadFreeFromArr(): array
	{
		if ($this->freeFromArr === null) {
			$this->freeFromArr = $this->eshopOrdersCache->getCache()->load('freeFrom/' . $this->sites->getCurrentSite()->getIdent(), function(&$dep) {
				$dep = [
					Cache::Tags       => ['freeFrom', 'spedition', 'payment'],
					Cache::EXPIRATION => '1 week',
				];

				$arr = [];

				foreach ($this->em->getConnection()->executeQuery("SELECT s.id, s.free_from FROM eshop_orders__spedition s
						INNER JOIN eshop_orders__payment_spedition ps ON ps.spedition_id = s.id
						INNER JOIN eshop_orders__payment_spedition_site pss ON pss.site_id = ? AND pss.ps_id = ps.id
						WHERE s.free_from > 0 AND s.free_from IS NOT NULL AND s.is_published = 1
						GROUP BY s.id", [$this->sites->getCurrentSite()->getIdent()])->iterateAssociative() as $row) {
					$arr[$row['id']] = (float) $row['free_from'];
				}

				return $arr;
			});
		}

		return $this->freeFromArr;
	}

	protected function getFreeDeliveryMinimumPrice(array $skipSpeditions = []): ?float
	{
		if ($skipSpeditions !== []) {
			$min            = 99999999;
			$skipSpeditions = array_flip($skipSpeditions);
			foreach ($this->loadFreeFromArr() as $id => $val) {
				if (isset($skipSpeditions[$id])) {
					continue;
				}

				if ($val < $min) {
					$min = $val;
				}
			}

			return $min;
		}

		if ($this->cFreeFrom === null && $this->loadFreeFromArr()) {
			$this->cFreeFrom = min($this->loadFreeFromArr());
		}

		return $this->cFreeFrom;
	}

}
