<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Presenters;

use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Application\AppState;
use Core\Model\Parameters;
use Currency\Model\Currencies;
use Currency\Model\Entities\Currency;
use EshopAdvancedFeature\DI\EshopAdvancedFeatureExtension;
use EshopCatalog\FrontModule\Components\CartAddForm;
use EshopCatalog\FrontModule\Components\ICartAddFormFactory;
use EshopCatalog\FrontModule\Components\IProductPreviewFactory;
use EshopCatalog\FrontModule\Components\IProductsFilterFactory;
use EshopCatalog\FrontModule\Components\IProductsListFactory;
use EshopCatalog\FrontModule\Components\Personalization\AlternativeProducts;
use EshopCatalog\FrontModule\Components\Personalization\BestSellingProducts;
use EshopCatalog\FrontModule\Components\Personalization\CrossSellProducts;
use EshopCatalog\FrontModule\Components\Personalization\IAlternativeProductsFactory;
use EshopCatalog\FrontModule\Components\Personalization\IBestSellingProductsFactory;
use EshopCatalog\FrontModule\Components\Personalization\ICrossSellProductsFactory;
use EshopCatalog\FrontModule\Components\Personalization\IProductListingFilterFactory;
use EshopCatalog\FrontModule\Components\Personalization\IRecentlyViewedProductFactory;
use EshopCatalog\FrontModule\Components\Personalization\ProductListingFilter;
use EshopCatalog\FrontModule\Components\Personalization\RecentlyViewedProduct;
use EshopCatalog\FrontModule\Components\ProductPreview;
use EshopCatalog\FrontModule\Components\ProductsFilter;
use EshopCatalog\FrontModule\Components\ProductsList;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Dao\Category;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\Event\CategoryHeadEvent;
use EshopCatalog\FrontModule\Model\Event\FilterLinkEvent;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\FrontModule\Model\SearchHistoryService;
use EshopCatalog\FrontModule\Model\Tags;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Helpers\ProductConditionEnum;
use EshopCatalog\Model\Navigation\Home;
use EshopOrders\FrontModule\Model\Customers;
use Gallery\FrontModule\Model\Albums;
use Nette\Application\Attributes\Persistent;
use Nette\Application\UI\Multiplier;
use Nette\DI\Container;
use Nette\Utils\DateTime;
use Nette\Utils\Strings;
use Pages\FrontModule\Model\Dao\DaoTemplatePage;
use Pages\FrontModule\Model\TemplatePages;
use Pages\Model\Paths;
use Spatie\SchemaOrg\OfferItemCondition;
use Spatie\SchemaOrg\Schema;

class DefaultPresenter extends BasePresenter
{
	public ?Category $category = null;

	#[Persistent]
	public ?int $variant = null;

	/** @var Category[] */
	public array $categories = [];

	public ?Product            $product      = null;
	protected ?DaoTemplatePage $templatePage = null;

	public function __construct(
		protected Categories           $categoriesService,
		protected Products             $productsService,
		protected TemplatePages        $templatePagesService,
		protected Paths                $pathsService,
		protected Albums               $albumsService,
		protected Tags                 $tagsService,
		protected Currencies           $currencies,
		protected SearchHistoryService $searchHistoryService,
		protected Customers            $customers,
		protected Container            $container,
		public ProductsFacade          $productsFacade
	)
	{
		parent::__construct();
	}

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionDefault(): void
	{
		$templatePage = $this->templatePagesService->get((int) $this->getActiveNavigation()->componentParams['templateHome']);

		if (!$templatePage) {
			$this->error('eshopCatalog.templatePage.notFound');
		}

		$this->template->pageClass          .= ' page-eshopcatalog-default';
		$this->template->templateTextValues = $templatePage->getTexts();
		$this->template->setFile($this->pathsService->getTemplatePagesFile($templatePage->getTemplate()));
	}

	public function actionProduct(int $id): void
	{
		$variant = $this->getParameter('variant');
		if ($variant) {
			$id = (int) $variant;
		}

		if ($this->container->hasService('eshopOrders.front.productReviewsFactory') && Parameters::load('eshopOrders.orderItemReviews.enable')) {
			$productReviewsFactory = $this->container->getService('eshopOrders.front.productReviewsFactory');
			$this->addComponent($productReviewsFactory->create($id), 'productReviews');
		}

		$this->product = $product = $this->productsFacade->getProduct($id);
		$currentSite   = $this->sitesService->getCurrentSite();

		if (!$product || !$product->defaultCategory || ($product->inSites && !$product->isInSite($currentSite->getIdent()))) {
			$this->error('eshopCatalog.product.notFound');
		}

		$customer = $this->getUser()->isLoggedIn()
			/** @phpstan-ignore-next-line */
			? $this->customers->getByUser($this->getUser()->getIdentity())
			: null;

		$hasCategoryAccess = true;
		if (!$this->categoriesService->checkCategoryRestrictionAccess($product->defaultCategory->allowedCustomerGroups, $customer)) {
			$hasCategoryAccess = false;
		}

		if (!$hasCategoryAccess) {
			foreach ($product->categories as $v) {
				$vCat = $this->categoriesService->get($v);
				if ($vCat && $this->categoriesService->checkCategoryRestrictionAccess($vCat->allowedCustomerGroups, $customer)) {
					$hasCategoryAccess = true;
					break;
				}
			}

			if (!$hasCategoryAccess) {
				$this->error('eshopCatalog.category.notFound');
			}
		}

		/** @var Product[] $products */
		$products = [
			$product->getId() => $product,
		];

		if (!$this->getActiveNavigation() instanceof DaoNavigationItem) {
			$navs = $this->navigationsService->getPublishedByComponent('eshopCatalog.navigation.home', $this->translator->getLocale());

			if ($navs !== []) {
				$this->setActiveNavigation(reset($navs));
			}
		}

		$this->templatePage = $templatePage = $this->templatePagesService->get((int) $this->getActiveNavigation()->componentParams['templateProduct']);

		$this->productsFacade->loadAlternative($product, (int) Config::loadScalar('config.productDetailAlternativeLimit') ?: 4);
		$this->productsService->loadFeatures($products);
		$this->productsService->loadDocuments($products);
		$this->productsService->loadVideos($products);
		$this->tagsService->loadTagsToProduct($products);

		if ($product->alternative) {
			$this->tagsService->loadTagsToProduct($product->alternative);
		}

		$this->productsFacade->loadRelated($product, (int) Config::loadScalar('productDetailRelatedLimit') ?: 4);

		$this->setTitle($product->getSeo('title') ?: $product->getName());

		if ($product->getSeo()) {
			$this->seoDataFromEntity = $product->getSeo();
		}

		$this->template->setFile($this->pathsService->getTemplatePagesFile($templatePage->getTemplate()));

		if ($product->defaultCategoryId) {
			$category = $this->categoriesService->get((int) $product->defaultCategoryId);

			if (!$category && $product->categories) {
				$category = $this->categoriesService->get((int) array_values($product->categories)[0]);
			}

			if ($category instanceof Category) {
				$this->category = $category;
			}
		}

		$url = $this->link('//this');

		$currency = $this->currencies->getCurrent();
		if (!$currency instanceof Currency) {
			$this->error('eshopCatalog.currency.notFound');
		}

		// Meta
		$this['meta']
			->setMeta('title', $product->getName())
			->setMeta('og:title', $product->getName(), 'property')
			->setMeta('og:type', 'og:product', 'property')
			->setMeta('og:description', Strings::truncate(strip_tags((string) $product->getDescription()), 255), 'property')
			->setMeta('og:url', $url, 'property')
			->setMeta('product:plural_title', $product->getName(), 'property')
			->setMeta('product:price:amount', (string) round($product->getPrice(), $currency->decimals), 'property')
			->setMeta('product:price:currency', $currency->getCode(), 'property')
			->setMeta('product:condition', ProductConditionEnum::toGoogle($product->condition), 'property');
		if ($product->getAvailability()) {
			$this['meta']->setMeta('product:availability', $product->getAvailability()->getOpenGraphText(), 'property');
		}

		$canonical = $product->getSeo('canonical');
		$httpUrl   = $this->getHttpRequest()->getUrl();
		if (!$canonical) {
			$httpUrl   = $httpUrl->withQuery(array_diff_key($httpUrl->getQueryParameters(), ['variant' => null]));
			$canonical = $httpUrl->getAbsoluteUrl();
		}

		if ($httpUrl->getQueryParameter('curr')) {
			$httpUrl   = $httpUrl->withQuery(array_diff_key($httpUrl->getQueryParameters(), ['curr' => null]));
			$canonical = $httpUrl->getAbsoluteUrl();
		}

		$this['meta']->setMeta('canonical', $canonical);

		if ($this->category instanceof Category) {
			$this->breadcrumbsAfter = array_merge($this->getBreadcrumb($this->category), [new DaoNavigationItem([
				'title' => $product->name,
				'link'  => $product->link,
			])]);
		}

		// Schema
		if (Config::load('allowRichSnippets')) {
			$schema = Schema::product()
				->productID((string) $product->getId())
				->name($product->getName())
				->category($product->defaultCategory->name)
				->description(Strings::truncate(strip_tags((string) $product->getDescription()), 255))
				->sku((string) ($product->getQuantity() + $product->getQuantityExternal()))
				->url($url);

			$offer = Schema::offer()
				->price($product->getBasePrice())
				->priceCurrency($this->currencies->getCurrent()->getCode())
				->itemCondition(Schema::offerItemCondition()->name(ProductConditionEnum::toGoogle($product->condition)))
				->priceValidUntil((new DateTime())->modify('+1 year')->setTime(0, 0, 0))
				->availability(Schema::itemAvailability()->name($product->getAvailability()->getSchemaUrl()))
				->url($url);

			$condition = match ($product->condition) {
				ProductConditionEnum::used, ProductConditionEnum::openBox => OfferItemCondition::UsedCondition,
				ProductConditionEnum::refurbished => OfferItemCondition::RefurbishedCondition,
				default => OfferItemCondition::NewCondition,
			};

			/** @phpstan-ignore-next-line */
			$offer->itemCondition($condition);

			$schema->offers([$offer]);

			$returnPolicyKey    = 'eshopCatalog' . Strings::firstUpper($currentSite->getIdent());
			$enableReturnPolicy = $this->settings->get($returnPolicyKey . 'EnableReturn');
			$returnPolicyData   = [
				'@type'             => 'MerchantReturnPolicy',
				'applicableCountry' => $this->settings->get($returnPolicyKey . 'ReturnApplicableCountry') ?: 'CZ',
			];

			if ($enableReturnPolicy === '') {
				$returnPolicyData['returnPolicyCategory'] = 'MerchantReturnUnspecified';
			} else if ((int) $enableReturnPolicy === 0) {
				$returnPolicyData += [
					'returnPolicyCategory' => 'MerchantReturnNotPermitted',
				];
			} else {
				$returnDays       = (int) $this->settings->get($returnPolicyKey . 'ReturnDays');
				$returnPolicyData += [
					'returnPolicyCategory' => $returnDays === 0 ? 'MerchantReturnUnlimitedWindow' : 'MerchantReturnFiniteReturnWindow',
					'returnMethod'         => $this->settings->get($returnPolicyKey . 'ReturnMethod'),
					'returnFees'           => $this->settings->get($returnPolicyKey . 'ReturnFees'),
				];

				if ($returnDays > 0) {
					$returnPolicyData['merchantReturnDays'] = $returnDays;
				}
			}

			$offer->setProperty('hasMerchantReturnPolicy', $returnPolicyData);

			if ($product->getEan()) {
				$schema->gtin13($product->getEan());
			}

			if ($product->getGallery() && $product->getGallery()->getCover()) {
				$imgUrl = $this->imagePipe->request($product->getGallery()->getCover()->getFilePath(), '1200x1200', 'fit');
				$schema->image($imgUrl);

				$this['meta']
					->setMeta('og:image', $imgUrl, 'property')
					->setMeta('image', $imgUrl);
			}

			if ($product->getManufacturer()) {
				$schema->brand(Schema::brand()
					->name($product->getManufacturer()->name)
					->logo($product->getManufacturer()->logo)
				);
				$schema->manufacturer(Schema::organization()
					->name($product->getManufacturer()->name)
					->logo($product->getManufacturer()->logo)
				);
			}

			$this['dataSchemaControl']->addSchema($schema);
		}
	}

	/**
	 * @param int|string $id
	 */
	public function actionCategory($id): void
	{
		$id = explode('|', (string) $id);
		if (count($id) > 1) {
			$this->autoCanonicalize = false;
			$category               = $this->categoriesService->get((int) $id[0]);
			$categories             = array_map(fn($c): ?Category => $this->categoriesService->get((int) $c), $id);
		} else {
			$category   = $this->categoriesService->get((int) $id[0]);
			$categories = [$category];
		}

		$customer = $this->getUser()->isLoggedIn()
			/** @phpstan-ignore-next-line */
			? $this->customers->getByUser($this->getUser()->getIdentity())
			: null;

		if (!$category || !$this->categoriesService->checkCategoryRestrictionAccess($category->allowedCustomerGroups, $customer)) {
			$this->error('eshopCatalog.category.notFound');
		}

		$this->category   = $category;
		$this->categories = $categories;

		if ($this->getParameter('loadFilters')) {
			$this['list']->setCategories($this->categories)
				->setFilterComponent($this['productsFilter']);

			$this->sendJson(['htmlFilters' => $this['productsFilter']->getHtmlFilters()]);
		}

		if (!$category->shortDescription && $category->description) {
			$category->shortDescription = $category->description;
			$category->description      = null;
		}

		$nav = $this->getActiveNavigation();
		if (!$nav) {
			$this->error('eshopCatalog.navigation.notFound');
		}

		if (
			$nav->getParam('virtualCategoryData')
			&& $nav->getParam('virtualCategoryData')['groups'][$nav->componentParams['group']]['templatePageItem']
		) {
			$this->template->setFile($this->pathsService->getTemplatePagesFile($nav->getParam('virtualCategoryData')['groups'][$nav->componentParams['group']]['templatePageItem']));
		} else {
			$templateId         = $nav->componentType === Home::COMPONENT_NAME ? $nav->componentParams['templateCategory'] : $nav->componentParams['templatePage'];
			$this->templatePage = $templatePage = $this->templatePagesService->get((int) $templateId);
			$this->template->setFile($this->pathsService->getTemplatePagesFile($templatePage->getTemplate()));
		}

		if ($category->getSeo()) {
			$this->seoDataFromEntity = $category->getSeo();
		}

		$showSubcategories = true;

		$this['list']->setCategories($this->categories)
			->setFilterComponent($this['productsFilter']);

		$nav = $this->getActiveNavigation();
		if ($nav && (count($this->categories) > 1 || $nav->getParam('virtualCategoryData'))) {
			$virtNavCategoryData = $nav->getParam('virtualCategoryData') ?? null;
			$showSubcategories   = false;

			if ($virtNavCategoryData
				&& is_array($virtNavCategoryData['params']['productsFilter-filter'])
				&& $category->getChild() && $this->getComponent('eshopNavigation', false)) {
				$links = [];
				foreach ($category->child as &$child) {
					$links[$child->getId()] = $this->link(':EshopCatalog:Front:Default:category', [
						'id'                    => $child->getId(),
						'productsFilter-filter' => $virtNavCategoryData['params']['productsFilter-filter'],
					]);
				}

				unset($child);
				$subcategoriesCacheSuffix                                    = md5(serialize($virtNavCategoryData['params']['productsFilter-filter']));
				$this['eshopNavigation']->editedLinks                        = $links;
				$this['eshopNavigation']->template->subcategoriesCacheSuffix = $subcategoriesCacheSuffix;
			}
		} else {
			$tmp = $this->getBreadcrumb($category);
			array_shift($tmp);
			$this->breadcrumbsAfter = $tmp;
		}

		$this->template->showSubcategories = $showSubcategories;

		$this->processCategory();
	}

	public function actionSearch(?string $q): void
	{
		if ($q) {
			$q = urldecode($q);

			$this['list']->setProductsBySearch($q)
				->setFilterComponent($this['productsFilter']);

			$currentPage = AppState::getState('list-page') ?: 1;
			if ($currentPage <= 1) {
				$this->searchHistoryService->saveSearch($q, $this->sitesService->getCurrentSite()->getIdent());
			}
		} else {
			$q = '';
		}

		$templatePage = $this->templatePagesService->get((int) $this->getActiveNavigation()->componentParams['templateResult']);

		if (!$templatePage) {
			$this->error('eshopCatalog.templatePage.notFound');
		}

		$this->templatePage = $templatePage;

		$this->template->q                  = $q;
		$this->template->templateTextValues = $templatePage->getTexts();
		$this->template->pageClass          .= ' page-eshopcatalog-search';

		$this->setTitle($this->translator->translate('eshopCatalogFront.search.searchQuery', null, ['q' => $q]));
		$this->template->setFile($this->pathsService->getTemplatePagesFile($templatePage->getTemplate()));

		$this['meta']->setMeta('robots', 'noindex, follow');

		if (Config::load('allowSearchInContent', false)) {
			$this->template->alowSearchInContent = true;
			$this->template->contentSeachLink    = $this->link(':Core:Front:Default:search', ['q' => $q]);
		}

		/** @var ProductsFilter|null $pf */
		$pf                            = $this['productsFilter'];
		$this->template->productsCount = $pf ? count($pf->getProductIdsAfterFilter()) : 0;
	}

	public function actionPersooSearch(): void
	{
		$currentPage = AppState::getState('list-page') ?: 1;
		if ($currentPage <= 1) {
			$this->searchHistoryService->saveSearch((string) $this->getParameter('q'), $this->sitesService->getCurrentSite()->getIdent());
		}

		$this['title']->setTitle($this->t('eshopCatalogFront.search.searchQuery', ['q' => $this->getParameter('q')]), false);
		$this['meta']->setMeta('robots', 'index, follow');
		$this['meta']->setMeta('canonical', $this->getHttpRequest()->getUrl()->getAbsoluteUrl());

		if (!Config::load('allowPersooSearchPage')) {
			$this->error();
		}
	}

	/*******************************************************************************************************************
	 * ============================== Render
	 */

	public function renderProduct(): void
	{
		$product      = $this->product;
		$templatePage = $this->templatePage;

		$this->template->templateTextValues = $templatePage->getTexts();
		$this->template->pageClass          .= ' page-eshopcatalog-product product-' . $product->id;
		$this->template->product            = $product;

		if ($this->getParameter('modal') !== null) {
			if (!$this->isAjax()) {
				$this->redirect('this');
			}

			$file      = $this->template->getFile();
			$tmp       = Strings::substring($file, 0, Strings::length($file) - 6);
			$modalFile = $tmp . '_modal.latte';

			$this->setLayout(false);
			echo $this->template->renderToString(file_exists($modalFile) ? $modalFile : $file);
			exit;
		}

		if (property_exists($this->presenter, 'addJsLibsFiles')) {
			$this->presenter->addJsLibsFiles['front/dist/js/fancybox.js'] = 'front/dist/js/fancybox.js';
		}
	}

	public function renderCategory(): void
	{
		$category     = $this->category;
		$templatePage = $this->templatePage;

		/** @var ProductsFilter|null $pf */
		$pf                                 = $this['productsFilter'];
		$this->template->templateTextValues = $templatePage instanceof DaoTemplatePage ? $templatePage->getTexts() : null;
		$this->template->pageClass          .= ' page-eshopcatalog-category';
		$this->template->category           = $category;
		$this->template->productsCount      = $pf ? count($pf->getProductIdsAfterFilter()) : 0;
		$this->template->filtersExist       = $pf->filtersExist();

		if (class_exists(EshopAdvancedFeatureExtension::class)) {
			if ($pf->checkDisableIndexingAdvanced()) {
				$this['meta']->setMeta('robots', 'noindex, nofollow');
			}
		} else if ($pf->checkDisableIndexing()) {
			$this['meta']->setMeta('robots', 'noindex, nofollow');
		}
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentCartAddForm(ICartAddFormFactory $factory): CartAddForm
	{
		$control = $factory->create();

		if ($this->product instanceof Product) {
			$control->setProduct($this->product);
		}
		$control->viewType = $control::VIEWTYPE_DETAIL;

		return $control;
	}

	protected function createComponentAddProductToCartForm(ICartAddFormFactory $factory): Multiplier
	{
		return new Multiplier(function($productId) use ($factory): CartAddForm {
			$control           = $factory->create();
			$control->viewType = $control::VIEWTYPE_DETAIL;
			$control->setProduct($this->productsFacade->getProduct((int) $productId));

			return $control;
		});
	}

	protected function createComponentList(IProductsListFactory $factory): ProductsList
	{
		return $factory->create();
	}

	protected function createComponentProductsFilter(IProductsFilterFactory $factory): ProductsFilter
	{
		$control = $factory->create();

		if ($this->isAjax()) {
			$this->redrawControl('breadcrumb');
			$this->redrawControl('categoryHead');
			$this->redrawControl('categoryMenu');
			$this->redrawControl('bestSellingProducts');
			$this->redrawControl('categoryFooter');
		}

		$this->eventDispatcher->addListener(ProductsFilter::class . '::onFilter', function(): void {
			$this->redrawControl('breadcrumb');
			$this->redrawControl('categoryHead');
			$this->redrawControl('categoryMenu');
			$this->redrawControl('bestSellingProducts');
			$this->redrawControl('categoryFooter');
			$this['list']->redrawControl('listWrap');
		});

		return $control;
	}

	protected function createComponentProductPreview(IProductPreviewFactory $factory): Multiplier
	{
		return new Multiplier(function(string $id) use ($factory): ProductPreview {
			$control = $factory->create();

			$control->setProductById((int) $id);

			return $control;
		});
	}

	protected function createComponentRecentlyViewedProduct(IRecentlyViewedProductFactory $factory): RecentlyViewedProduct
	{
		return $factory->create();
	}

	protected function createComponentAlternativeProducts(IAlternativeProductsFactory $factory): AlternativeProducts
	{
		return $factory->create();
	}

	protected function createComponentCrossSellProducts(ICrossSellProductsFactory $factory): CrossSellProducts
	{
		return $factory->create();
	}

	protected function createComponentBestSellingProducts(IBestSellingProductsFactory $factory): BestSellingProducts
	{
		return $factory->create();
	}

	protected function createComponentProductListingFilter(IProductListingFilterFactory $factory): ProductListingFilter
	{
		return $factory->create();
	}

	/*******************************************************************************************************************
	 * ==============================
	 */

	protected function getBreadcrumb(Category $category): array
	{
		$path    = $this->categoriesService->getPath($category);
		$baseNav = null;

		foreach ($path as $cat) {
			$tmp = $this->categoriesService->findNavigationId($cat->id);

			if (!$tmp) {
				continue;
			}

			$tmp = $this->navigationsService->getNavigation($tmp);
			if ($tmp instanceof DaoNavigationItem) {
				$baseNav = $tmp;
				break;
			}
		}

		$return = [];
		foreach ($baseNav instanceof DaoNavigationItem ? $this->categoriesService->getBreadcrumb($category, $baseNav) : [] as $cat) {
			if ($cat instanceof Category) {
				$return[] = new DaoNavigationItem([
					'title' => $cat->name,
					'link'  => $cat->link,
					'alias' => $cat->alias,
				]);
				/** @phpstan-ignore-next-line */
			} else if ($cat instanceof DaoNavigationItem) {
				$return[] = $cat;
			}
		}

		return $return;
	}

	protected function processCategory(): void
	{
		$category = $this->category;
		$nav      = $this->getActiveNavigation();

		if (!$nav instanceof DaoNavigationItem) {
			return;
		}

		$multiCats = count($this->categories) > 1;

		$paginator        = $this['list']->getPaginator();
		$shortDescription = (string) $this->category->shortDescription;
		$showDescription  = true;
		$metaDescription  = '';

		if (!$multiCats) {
			$categoryName    = $category->getNameH1();
			$title           = (string) ($category->getSeo('title') ?: $category->name);
			$metaDescription = (string) ($category->getSeo('description') ?: $category->shortDescription);
			$canonical       = $category->getSeo('canonical');

			if ($this['productsFilter']->getFilterValues()) {
				$category->shortDescription = null;
				$showDescription            = false;
			}

			if ($canonical) {
				$this['meta']->setMeta('canonical', $canonical);
			}
		} else {
			$categoryName = (string) $nav->title;
			$title        = (string) $nav->getSeo('title') ?: $nav->title;
		}

		$event            = new CategoryHeadEvent($title, $categoryName, $shortDescription, $metaDescription, $this['productsFilter']->getFilterValues(), $category);
		$event->presenter = $this;
		$this->eventDispatcher->dispatch($event, 'eshopCatalog.default.actionCategory.head');

		$titleSuffix = null;
		if ($paginator->getPage() > 1) {
			$titleSuffix                = $this->translator->translate('eshopCatalogFront.pageXOfY', [
				'x' => $paginator->getPage(),
				'y' => $paginator->getPageCount(),
			]);
			$title                      .= ' ' . $titleSuffix;
			$category->shortDescription = null;
			$showDescription            = false;
		}

		$this->setTitle($title);

		$categoryName = $event->categoryName ?: $category->getNameH1();
		if ($titleSuffix) {
			$categoryName .= ' ' . $titleSuffix;
		}

		$this->template->categoryName    = $categoryName;
		$this->template->showDescription = $showDescription;
		$this->template->pageTitle       = $categoryName;
		$this->template->page            = $paginator->getPage();

		$filterValues    = $this['productsFilter']->getFilterValues();
		$disableIndexing = false;
		$disableFollow   = false;

		if (isset($filterValues['features']) && $category->getAttr('featuresEnableIndexing') !== true) {
			if (count($filterValues['features']) >= 3) {
				$disableIndexing = true;
				$disableFollow   = true;
			}

			foreach ($filterValues['features'] as $vals) {
				if (count($vals) >= 2) {
					$disableIndexing = true;
					$disableFollow   = true;
					break;
				}
			}
		}

		if (isset($filterValues['sort']) || isset($filterValues['priceRange'])) {
			$disableIndexing = true;
			$disableFollow   = true;
		}

		$this['meta']->setMeta('robots', implode(', ', [$disableIndexing ? 'noindex' : 'index', $disableFollow ? 'nofollow' : 'follow']));

		if ($metaDescription) {
			$this['meta']->setMeta('description', Strings::truncate(str_replace('&nbsp;', ' ', strip_tags((string) $metaDescription)), 130));
		}

		$virtualCategoryData = $nav->getParam('virtualCategoryData') ?? [];
		if ($virtualCategoryData) {
			$this['meta']->setMeta('canonical', $this->getHttpRequest()->getUrl()->getBaseUrl() . ltrim($this->getHttpRequest()->getUrl()->getRelativePath(), '/'));

			$this->breadcrumbsAfter = $this->getBreadcrumb($category);

			if ($virtualCategoryData['groups']) {
				$this->breadcrumbsAfter = [new DaoNavigationItem([
					'title' => $nav->getParam('virtualCategoryData')['menu_title'] ?: $categoryName,
					'link'  => $this->link(':EshopAdvancedFeature:Front:Default:virtualCategoryGroup', ['id' => $nav->componentParams['group']]),
				])];
			}
		}
	}

	public function createFilterLink(int $featureId, int $featureValueId): ?string
	{

		$categoryId = $this->product->defaultCategoryId;
		$result     = null;

		if (Parameters::load('eshopAdvancedFeature.allowVirtualCategories')) {
			$event = new FilterLinkEvent([], [$categoryId], 'feature', (string) $featureId, (string) $featureValueId);
			$this->eventDispatcher->dispatch($event, ProductsFilter::class . '::createFilterLink');

			$result = $event->result;
		}

		if (!$result) {
			$result = $this->link(':EshopCatalog:Front:Default:category', [
				'id'                    => $categoryId,
				'productsFilter-filter' => [
					'ff' => [
						$featureId => $featureValueId,
					],
				],
				'do'                    => 'productsFilter-set',
			]);
		}

		return $result;
	}
}
