<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Entities\TId;
use Core\Model\Entities\TranslateListener;
use Core\Model\Entities\TTranslateListener;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;

#[ORM\Table(name: 'eshop_catalog__availability')]
#[ORM\Entity]
#[ORM\EntityListeners([TranslateListener::class, AvailabilityListener::class])]
class Availability
{
	use TId;
	use TTranslateListener;

	public const IN_STOCK = 'inStock';
	public const SOLD_OUT = 'soldOut';
	public const PREORDER = 'preorder';
	public const RESERVED = 'reserved';

	#[ORM\Column(type: Types::STRING, length: 255, nullable: false)]
	protected string $ident;

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['unsigned' => true, 'default' => 0])]
	public int $delay;

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['default' => 1])]
	public int $canShowOnList = 1;

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['default' => 1])]
	public int $canShowOnSearch = 1;

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['default' => 1])]
	public int $canAddToCart = 1;

	#[ORM\Column(type: Types::STRING, nullable: true)]
	public ?string $color = null;

	#[ORM\Column(type: Types::STRING, nullable: true)]
	public ?string $bgColor = null;

	/** @var Collection<string, AvailabilityText> */
	#[ORM\OneToMany(mappedBy: 'availability', targetEntity: AvailabilityText::class, indexBy: 'lang')]
	protected Collection $texts;

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['default' => 1])]
	protected int $canDelete = 1;

	#[ORM\Column(type: Types::STRING, length: 255, nullable: true)]
	public ?string $openGraphText = null;

	/** @Gedmo\SortablePosition */
	#[ORM\Column(type: Types::INTEGER)]
	protected int $position;

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['unsigned' => true, 'default' => 0])]
	public int $warehouseOverdraft = 0;

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['unsigned' => true, 'default' => 1])]
	public int $showInFilter = 1;

	public function __construct(string $ident, int $delay)
	{
		$this->setIdent($ident);
		$this->delay     = $delay;
		$this->texts     = new ArrayCollection;
		$this->canDelete = 1;
		$this->setPosition(-1);
	}

	public function getIdent(): string { return $this->ident; }

	public function setIdent(string $ident): self
	{
		$this->ident = $ident;

		return $this;
	}

	public function getText(string $lang = ''): ?AvailabilityText { return $this->texts[$lang ?: $this->locale]; }

	/**
	 * @return Collection<string, AvailabilityText>
	 */
	public function getTexts(): Collection { return $this->texts; }

	/**
	 * @param AvailabilityText[] $texts
	 */
	public function setTexts(array $texts): self
	{
		$this->texts = new ArrayCollection($texts);

		return $this;
	}

	public function canDelete(): bool { return (bool) $this->canDelete; }

	public function getPosition(): int { return $this->position; }

	public function setPosition(int $position): self
	{
		$this->position = $position;

		return $this;
	}

}
