<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Entities\Repository\NestedTreeRepository;
use Core\Model\Entities\TId;
use Core\Model\Entities\TranslateListener;
use Core\Model\Entities\TTranslateListener;
use Core\Model\Helpers\Traits\TExtraField;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;

/**
 * @Gedmo\Tree(type="nested")
 */
#[ORM\Table(name: 'eshop_catalog__category')]
#[ORM\Index(name: 'catalog_search', columns: ['id', 'is_published'])]
#[ORM\Entity(repositoryClass: NestedTreeRepository::class)]
#[ORM\EntityListeners([TranslateListener::class, CategoryListener::class])]
class Category
{
	use TId;
	use TTranslateListener;
	use TExtraField;

	public const EXTRA_FIELD_SECTION = 'eshopCatalogCategory';

	#[ORM\Column(name: 'image', type: Types::STRING, length: 255, nullable: true)]
	public ?string $image = null;

	#[ORM\Column(type: Types::SMALLINT, options: ['default' => 0])]
	public int $isPublished = 0;

	/** @var Collection<Category> */
	#[ORM\OneToMany(targetEntity: Category::class, mappedBy: 'parent')]
	#[ORM\OrderBy(['lft' => 'ASC'])]
	public Collection $children;

	/** @Gedmo\TreeParent */
	#[ORM\JoinColumn(name: 'parent_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\ManyToOne(targetEntity: Category::class, cascade: ['persist'], inversedBy: 'childrens')]
	public ?Category $parent = null;

	/** @Gedmo\TreeLeft */
	#[ORM\Column(type: Types::INTEGER)]
	private int $lft;

	/** @Gedmo\TreeLevel */
	#[ORM\Column(type: Types::INTEGER)]
	protected int $lvl;

	/** @Gedmo\TreeRight */
	#[ORM\Column(name: 'rgt', type: Types::INTEGER)]
	private int $gt;

	/** @Gedmo\TreeRoot */
	#[ORM\JoinColumn(name: 'tree_root', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\ManyToOne(targetEntity: Category::class)]
	private ?Category $root = null;

	/** @var Collection<CategoryTexts> */
	#[ORM\OneToMany(targetEntity: CategoryTexts::class, mappedBy: 'id', cascade: ['persist', 'remove'], indexBy: 'lang')]
	protected Collection $categoryTexts;

	/** @var Collection<int, CategoryProduct> */
	#[ORM\OneToMany(targetEntity: CategoryProduct::class, mappedBy: 'category', cascade: ['all'], indexBy: 'category')]
	protected Collection $categoryProducts;

	/** @var Collection<CategoryFilter> */
	#[ORM\OneToMany(targetEntity: CategoryFilter::class, mappedBy: 'category', indexBy: 'feature_id')]
	#[ORM\OrderBy(['position' => 'ASC'])]
	public Collection $filters;

	#[ORM\Column(type: Types::SMALLINT, options: ['default' => 1])]
	public int $filtersFromParent = 1;

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['default' => 0])]
	public int $ajaxFilterLoad = 0;

	#[ORM\Column(type: Types::DATETIME_MUTABLE, options: ['default' => 'CURRENT_TIMESTAMP'])]
	protected \DateTime $created;

	/** @Gedmo\Timestampable(on="update") */
	#[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: true)]
	protected ?\DateTime $modified = null;

	#[ORM\Column(type: Types::JSON, nullable: true)]
	protected ?array $attrs = null;

	#[ORM\Column(type: Types::STRING, length: 1, nullable: true)]
	public ?string $rod = null;

	#[ORM\Column(type: Types::SMALLINT, length: 1, options: ['default' => 1])]
	public int $canProductsAddToCart = 1;

	/** @var Collection<CategoryRelated> */
	#[ORM\OneToMany(targetEntity: CategoryRelated::class, mappedBy: 'category')]
	protected Collection $related;

	#[ORM\Column(type: Types::SMALLINT, options: ['default' => 0])]
	public int $disablePickupPoints = 0;

	/** @var Collection<FeatureCategory> */
	#[ORM\OneToMany(targetEntity: FeatureCategory::class, mappedBy: 'category', cascade: ['all'], indexBy: 'id_feature_value')]
	public Collection $features;

	/** @var Collection<DynamicFeatureCategory> */
	#[ORM\OneToMany(targetEntity: DynamicFeatureCategory::class, mappedBy: 'category', cascade: ['all'], indexBy: 'id')]
	public Collection $dynamicFeatures;

	#[ORM\Column(type: Types::STRING, length: 255, nullable: true)]
	public ?string $customerGroupRestriction = null;

	/**
	 * @deprecated
	 */
	#[ORM\Column(name: 'safety_warning_image', type: Types::STRING, length: 255, nullable: true)]
	public ?string $safetyWarningImage = null;

	#[ORM\Column(type: Types::SMALLINT, nullable: true, options: ['unsigned' => true])]
	public ?int $hasActiveProducts = null;

	#[ORM\Column(type: Types::STRING, length: 255, nullable: true)]
	public ?string $remoteId = null;

	#[ORM\Column(type: Types::SMALLINT, nullable: true, options: ['unsigned' => true, 'default' => 0])]
	public int $disableRegisterSale = 0;

	public function __construct()
	{
		$this->categoryTexts        = new ArrayCollection;
		$this->categoryProducts     = new ArrayCollection;
		$this->children             = new ArrayCollection;
		$this->filters              = new ArrayCollection;
		$this->isPublished          = 0;
		$this->created              = new DateTime;
		$this->filtersFromParent    = 1;
		$this->attrs                = [];
		$this->canProductsAddToCart = 1;
		$this->extraFields          = new ArrayCollection;
		$this->related              = new ArrayCollection;
		$this->features             = new ArrayCollection();
		$this->dynamicFeatures      = new ArrayCollection();
	}

	public function getLvl(): int { return $this->lvl; }

	public function getRoot(): ?Category { return $this->root; }

	public function addChildren(Category $child): void
	{
		$this->children->add($child);
	}

	public function addCategoryText(string $lang): void
	{
		$this->categoryTexts[$lang] = new CategoryTexts($this, $lang);
	}

	public function getCategoryText(?string $lang = null): ?CategoryTexts
	{
		return $this->categoryTexts[$lang ?? $this->locale];
	}

	/**
	 * @return Collection<string, CategoryTexts>
	 */
	public function getCategoryTexts(): Collection { return $this->categoryTexts; }

	/**
	 * @param CategoryTexts[] $categoryTexts
	 */
	public function setCategoryTexts(array $categoryTexts): self
	{
		$this->categoryTexts = new ArrayCollection($categoryTexts);

		return $this;
	}

	public function setCategoryText(CategoryTexts $categoryTexts): void
	{
		$this->categoryTexts->set($categoryTexts->getLang(), $categoryTexts);
	}

	public function getParent(): ?Category
	{
		return $this->parent;
	}

	public function setParent(?Category $parent): void
	{
		$this->parent = $parent;
	}

	public function addProduct(CategoryProduct $categoryProduct): void
	{
		$this->categoryProducts->add($categoryProduct);
	}

	public function getCreated(): \DateTime { return $this->created; }

	public function getModified(): \DateTime { return $this->modified ?: $this->getCreated(); }

	public function getAttrs(): array { return $this->attrs ? (array) $this->attrs : []; }

	/**
	 * @return mixed|null
	 */
	public function getAttr(string $key) { return $this->attrs[$key] ?? null; }

	/**
	 * @param mixed $value
	 */
	public function setAttr(string $key, $value): self
	{
		$this->attrs[$key] = $value;

		return $this;
	}

	public function removeAttr(string $key): self
	{
		unset($this->attrs[$key]);

		return $this;
	}

	public function getLft(): int { return $this->lft; }

	public function getRgt(): int { return $this->gt; }

	public function getAllowedCustomerGroup(): array
	{
		$groups = [];
		foreach (explode(',', (string) $this->customerGroupRestriction) as $group) {
			$groups[] = (int) $group;
		}

		return $groups;
	}

	public function setAllowedCustomerGroup(array $groups): void
	{
		if (empty($groups)) {
			$this->customerGroupRestriction = null;

			return;
		}

		$this->customerGroupRestriction = implode(',', $groups);
	}
}
