<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Event\EntityRemoveEvent;
use Core\Model\Event\EventDispatcher;
use Doctrine;
use Doctrine\ORM\Mapping as ORM;
use Doctrine\Persistence\Event\LifecycleEventArgs;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\FrontModule\Model\Categories;
use Nette\Caching\Cache;
use Nette\SmartObject;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class CategoryListener implements EventSubscriberInterface
{
	use SmartObject;

	protected ?int $objectId = null;

	public function __construct(
		protected CacheService    $cacheService,
		protected EventDispatcher $eventDispatcher,
	)
	{
	}

	public static function getSubscribedEvents(): array
	{
		return [];
	}

	#[ORM\PostUpdate]
	public function postUpdateHandler(Category $category, LifecycleEventArgs $event): void
	{
		$this->cacheService->clean('category', [Cache::Tags => ['categories']]);
	}

	#[ORM\PreRemove]
	public function preRemoveHandler(Category $category, LifecycleEventArgs $event): void
	{
		$this->objectId = $category->getId();
		$this->eventDispatcher->dispatch(new EntityRemoveEvent((string) $this->objectId, $category), Category::class . '::preRemove');
	}

	#[ORM\PostRemove]
	public function postRemoveHandler(Category $category, LifecycleEventArgs $event): void
	{
		$tags2 = [
			Categories::CACHE_NAMESPACE,
		];

		if ($category->getParent()) {
			$tags2[] = 'category/' . $category->getParent()->getId();
		}

		$this->cacheService->clean('category', [Cache::Tags => $tags2]);
		$this->eventDispatcher->dispatch(new EntityRemoveEvent((string) $this->objectId, $category), Category::class . '::remove');
	}
}
