<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Entities\TId;
use Core\Model\Entities\TranslateListener;
use Core\Model\Entities\TTranslateListener;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;

#[ORM\Table(name: 'eshop_catalog__feature')]
#[ORM\Index(name: 'published', columns: ['id', 'is_published'])]
#[ORM\Entity]
#[ORM\EntityListeners([FeatureListener::class, TranslateListener::class])]
class Feature
{
	public const TYPE_CHECK = 'check';
	public const TYPE_RANGE = 'range';

	public const VALUE_TYPE_TEXT  = 'text';
	public const VALUE_TYPE_COLOR = 'color';

	use TId;
	use TTranslateListener;

	/** @Gedmo\SortablePosition */
	#[ORM\Column(type: Types::INTEGER)]
	protected int $position = 0;

	#[ORM\Column(type: Types::SMALLINT, length: 1, options: ['default' => 1])]
	public int $isPublished = 1;

	#[ORM\Column(type: Types::SMALLINT, length: 1, options: ['default' => 1])]
	public int $showInProduct = 1;

	#[ORM\Column(type: Types::SMALLINT, length: 1, options: ['default' => 1])]
	public int $showInExport = 1;

	#[ORM\Column(name: 'use_filter', type: Types::SMALLINT, length: 1, options: ['default' => 0])]
	public int $useAsFilter = 0;

	#[ORM\Column(type: Types::SMALLINT, length: 1, options: ['default' => 1, 'unsigned' => true])]
	public int $useForVariantDiff = 1;

	/** @var Collection<string, FeatureTexts> */
	#[ORM\OneToMany(targetEntity: FeatureTexts::class, mappedBy: 'id', cascade: ['all'], indexBy: 'lang')]
	protected Collection $featureTexts;

	/** @var Collection<int, FeatureValue> */
	#[ORM\OneToMany(targetEntity: FeatureValue::class, mappedBy: 'feature', cascade: ['all'], indexBy: 'id')]
	protected Collection $featureValues;

	#[ORM\Column(type: Types::SMALLINT, options: ['default' => 0])]
	public int $beautyUrlActive = 0;

	#[ORM\Column(type: Types::STRING, length: 10, nullable: true)]
	public ?string $beaUrlPosition = null;

	#[ORM\Column(type: Types::INTEGER, length: 10, nullable: false, options: ['default' => 10])]
	public int $beaUrlPriority = 10;

	#[ORM\Column(name: 'bea_url_pattern', type: Types::STRING, nullable: false, options: ['default' => '%v%'])]
	public string $beaUrlPattern;

	#[ORM\Column(name: 'bea_url_position_multi', type: Types::STRING, length: 10, nullable: true)]
	public ?string $beaUrlPositionMulti = null;

	#[ORM\Column(type: Types::INTEGER, length: 10, nullable: false, options: ['default' => 10])]
	public int $beaUrlPriorityMulti;

	#[ORM\Column(type: Types::SMALLINT, length: 1, options: ['default' => 1])]
	public int $exportHeureka = 1;

	#[ORM\Column(type: Types::SMALLINT, length: 1, options: ['default' => 1])]
	public int $exportZbozi = 1;

	#[ORM\Column(type: Types::SMALLINT, length: 1, options: ['default' => 1])]
	public int $exportIdealo = 1;

	#[ORM\Column(name: 'zbozi_type', type: Types::STRING, nullable: true)]
	public ?string $zboziType = null;

	#[ORM\Column(name: 'zbozi_unit', type: Types::STRING, nullable: true)]
	public ?string $zboziUnit = null;

	#[ORM\Column(type: Types::SMALLINT, length: 1, options: ['default' => 1])]
	public int $exportGoogle = 1;

	#[ORM\Column(type: Types::STRING, nullable: true)]
	public ?string $googleType = null;

	#[ORM\Column(name: '`type`', type: Types::STRING, length: 50, nullable: false, options: ['default' => 'check'])]
	public string $type = self::TYPE_CHECK;

	#[ORM\Column(name: '`value_type`', type: Types::STRING, length: 50, nullable: false, options: ['default' => 'text'])]
	public string $valueType = self::VALUE_TYPE_TEXT;

	#[ORM\Column(type: Types::STRING, length: 10, nullable: true)]
	public ?string $unit = null;

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['unsigned' => true, 'default' => 0])]
	public int $decimals = 0;

	#[ORM\Column(type: Types::SMALLINT, nullable: true, options: ['unsigned' => true, 'default' => 0])]
	public int $seoNoFollow = 0;

	#[ORM\Column(type: Types::STRING, nullable: true)]
	public ?string $image = null;

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['unsigned' => true, 'default' => 0])]
	public int $showFilterLinkInProduct = 0;

	public function __construct()
	{
		$this->featureTexts  = new ArrayCollection;
		$this->featureValues = new ArrayCollection;
		$this->setIsPublished(1);
		$this->setUseAsFilter(0);
		$this->setPosition(-1);

		$this->beautyUrlActive     = 0;
		$this->beaUrlPattern       = '%v%';
		$this->beaUrlPriority      = 10;
		$this->beaUrlPriorityMulti = 10;
	}

	public function addFeatureText(string $lang): void
	{
		$this->featureTexts->set($lang, new FeatureTexts($this, $lang));
	}

	public function setFeatureText(FeatureTexts $featureTexts): void
	{
		$this->featureTexts->set($featureTexts->getLang(), $featureTexts);
	}

	public function setFeatureTexts(array $texts): void
	{
		$this->featureTexts = new ArrayCollection($texts);
	}

	public function getFeatureText(?string $lang = null): ?FeatureTexts { return $this->featureTexts[$lang ?: $this->locale] ?? null; }

	/**
	 * @return Collection<string, FeatureTexts>
	 */
	public function getFeatureTexts(): Collection { return $this->featureTexts; }

	/**
	 * @return Collection<int, FeatureValue>
	 */
	public function getFeatureValues(): Collection { return $this->featureValues; }

	public function setPosition(int $position): self
	{
		$this->position = $position;

		return $this;
	}

	public function getPosition(): int { return $this->position; }

	/******
	 * === Use filter
	 */

	/**
	 * @return int|bool
	 */
	public function getUseAsFilter() { return $this->useAsFilter; }

	/**
	 * @param int|bool $val
	 */
	public function setUseAsFilter($val): self
	{
		if ($val && $val == 1) {
			$this->useAsFilter = 1;
		} else {
			$this->useAsFilter = 0;
		}

		return $this;
	}

	/******
	 * === Publish
	 */

	/**
	 * @return int|bool
	 */
	public function getIsPublished() { return $this->isPublished; }

	/**
	 * @param int|bool $val
	 */
	public function setIsPublished($val): self
	{
		if ($val && $val == 1) {
			$this->isPublished = 1;
		} else {
			$this->isPublished = 0;
		}

		return $this;
	}

	public static function getStep(int $decimals): string
	{
		$step = '1';

		if ($decimals === 0) {
			return $step;
		}

		$step = str_pad($step, $decimals, '0', STR_PAD_LEFT);

		return '0.' . $step;
	}

}
