<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Entities\TId;
use Core\Model\Entities\TranslateListener;
use Core\Model\Entities\TTranslateListener;
use Core\Model\Helpers\Traits\TExtraField;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;

#[ORM\Table(name: 'eshop_catalog__feature_value')]
#[ORM\Index(name: 'published', columns: ['id', 'is_published'])]
#[ORM\Entity]
#[ORM\EntityListeners([FeatureValueListener::class, TranslateListener::class])]
class FeatureValue
{
	use TId;
	use TTranslateListener;
	use TExtraField;

	public const EXTRA_FIELD_SECTION = 'eshopCatalogFeatureValue';

	/** @Gedmo\SortableGroup */
	#[ORM\JoinColumn(name: 'feature_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\ManyToOne(targetEntity: Feature::class, cascade: ['persist'], inversedBy: 'id')]
	protected ?Feature $feature = null;

	/** @var Collection<string, FeatureValueTexts> */
	#[ORM\OneToMany(targetEntity: FeatureValueTexts::class, mappedBy: 'id', indexBy: 'lang')]
	protected Collection $featureValueTexts;

	/** @Gedmo\SortablePosition */
	#[ORM\Column(type: Types::INTEGER)]
	private int $position;

	#[ORM\Column(type: Types::SMALLINT, nullable: true, options: ['default' => 0])]
	public int $isPublished = 0;

	#[ORM\Column(name: 'value1', type: Types::STRING, length: 255, nullable: true)]
	public ?string $value1 = null;

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['default' => 0])]
	public int $showAsTag = 0;

	#[ORM\Column(type: Types::STRING, nullable: true)]
	public ?string $moreLink = null;

	#[ORM\Column(type: Types::STRING, nullable: true)]
	public ?string $image = null;

	#[ORM\Column(type: Types::STRING, nullable: true)]
	public ?string $tagTextColor = null;

	#[ORM\Column(type: Types::STRING, nullable: true)]
	public ?string $tagBgColor = null;

	#[ORM\Column(type: Types::STRING, nullable: true)]
	public ?string $googleType = null;

	#[ORM\Column(type: Types::SMALLINT, nullable: true, options: ['unsigned' => true])]
	public ?int $seoNoFollow = null;

	#[ORM\Column(type: Types::STRING, nullable: true)]
	public ?string $zboziUnit = null;

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['unsigned' => true, 'default' => 0])]
	public int $showFilterLinkInProduct = 0;

	public function __construct()
	{
		$this->featureValueTexts = new ArrayCollection;
		$this->isPublished       = 1;
		$this->setPosition(-1);
		$this->extraFields = new ArrayCollection;
	}

	public function addFeatureValueText(string $lang): void
	{
		$this->featureValueTexts->set($lang, new FeatureValueTexts($this, $lang));
	}

	public function getFeatureValueText(?string $lang = null): ?FeatureValueTexts { return $this->featureValueTexts[$lang ?? $this->locale] ?: null; }

	/**
	 * @return Collection<string, FeatureValueTexts>
	 */
	public function getFeatureValueTexts(): Collection { return $this->featureValueTexts; }

	public function setFeature(Feature $feature): void
	{
		$this->feature = $feature;
	}

	public function setFeatureValueText(FeatureValueTexts $featureValueTexts): void
	{
		$this->featureValueTexts->set($featureValueTexts->getLang(), $featureValueTexts);
	}

	public function setTexts(array $texts): void
	{
		$this->featureValueTexts = new ArrayCollection($texts);
	}

	public function getFeature(): ?Feature
	{
		return $this->feature;
	}

	public function setPosition(int $position): self
	{
		$this->position = $position;

		return $this;
	}

	public function getPosition(): int { return $this->position; }

}
