<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Contributte\Translation\Translator;
use Core\Model\Event\EntityRemoveEvent;
use Core\Model\Event\EventDispatcher;
use Doctrine;
use Doctrine\ORM\Mapping as ORM;
use Doctrine\Persistence\Event\LifecycleEventArgs;
use EshopCatalog\FrontModule\Model\CacheService;
use Nette\SmartObject;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class FeatureValueListener implements EventSubscriberInterface
{
	use SmartObject;

	protected ?int $objectId = null;

	public function __construct(
		protected CacheService    $cacheService,
		protected Translator      $translator,
		protected EventDispatcher $eventDispatcher,
	)
	{
	}

	public static function getSubscribedEvents(): array
	{
		return [];
	}

	/**
	 * @param FeatureValue|FeatureValueTexts $entity
	 */
	#[ORM\PostUpdate]
	public function onPostUpdate($entity, LifecycleEventArgs $event): void
	{
		$featureIds = [];

		if ($entity instanceof FeatureValue && $entity->getFeature() && $entity->getFeature()->getId() !== null) {
			$featureIds[] = $entity->getFeature()->getId();
		}

		$this->cacheService->clearFeatures($featureIds);

		$this->cacheService->clearFeaturesValues();
	}

	/**
	 * @param FeatureValue|FeatureValueTexts $entity
	 */
	#[ORM\PreRemove]
	public function preRemoveHandler($entity, LifecycleEventArgs $event): void
	{
		if ($entity instanceof FeatureValue) {
			$this->objectId = $entity->getId();
			$this->eventDispatcher->dispatch(new EntityRemoveEvent((string) $this->objectId, $entity), FeatureValue::class . '::preRemove');
		}
	}

	/**
	 * @param FeatureValue|FeatureValueTexts $entity
	 */
	#[ORM\PostRemove]
	public function postRemoveHandler($entity, LifecycleEventArgs $event): void
	{
		if ($entity instanceof FeatureValue) {
			$this->eventDispatcher->dispatch(new EntityRemoveEvent((string) $this->objectId, $entity), FeatureValue::class . '::remove');
		}
	}
}
