<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Contributte\Translation\Translator;
use Core\Model\Event\EntityRemoveEvent;
use Core\Model\Event\EventDispatcher;
use Doctrine;
use Doctrine\ORM\Event\PreFlushEventArgs;
use Doctrine\ORM\Mapping as ORM;
use Doctrine\Persistence\Event\LifecycleEventArgs;
use EshopCatalog\FrontModule\Model\CacheService;
use Nette\SmartObject;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class ManufacturerListener implements EventSubscriberInterface
{
	use SmartObject;

	protected ?int $objectId = null;

	public function __construct(
		protected CacheService    $cacheService,
		protected Translator      $translator,
		protected EventDispatcher $eventDispatcher,
	)
	{
	}

	public static function getSubscribedEvents(): array
	{
		return [];
	}

	/**
	 * @param Manufacturer|ManufacturerTexts $entity
	 */
	#[ORM\PreFlush]
	public function onPreFlush($entity, PreFlushEventArgs $args): void
	{
		$this->cacheService->clearManufacturersCache();
	}

	/**
	 * @param Manufacturer|ManufacturerTexts $entity
	 */
	#[ORM\PostUpdate]
	public function preUpdate($entity, LifecycleEventArgs $args): void
	{
		$this->cacheService->clearManufacturersCache();
	}

	/**
	 * @param Manufacturer|ManufacturerTexts $manufacturer
	 */
	#[ORM\PreRemove]
	public function preRemoveHandler($manufacturer, LifecycleEventArgs $event): void
	{
		if ($manufacturer instanceof Manufacturer) {
			$this->objectId = $manufacturer->getId();
			$this->eventDispatcher->dispatch(new EntityRemoveEvent((string) $this->objectId, $manufacturer), Manufacturer::class . '::preRemove');
		}
	}

	/**
	 * @param Manufacturer|ManufacturerTexts $manufacturer
	 */
	#[ORM\PostRemove]
	public function postRemoveHandler($manufacturer, LifecycleEventArgs $event): void
	{
		$this->cacheService->clearManufacturersCache();

		if ($manufacturer instanceof Manufacturer) {
			$this->eventDispatcher->dispatch(new EntityRemoveEvent((string) $this->objectId, $manufacturer), Manufacturer::class . '::remove');
		}
	}
}
