<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Entities\Country;
use Core\Model\Entities\TId;
use Core\Model\Entities\TranslateListener;
use Core\Model\Entities\TTranslateListener;
use Core\Model\Helpers\Traits\TExtraField;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use EshopCatalog\Model\Config;
use Gallery\Model\Entities\Album;
use Gallery\Model\Entities\Image;
use Gedmo\Mapping\Annotation as Gedmo;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;

#[ORM\Table(name: 'eshop_catalog__product')]
#[ORM\Index(name: 'catalog_search', columns: ['is_published', 'is_deleted', 'disable_listing', 'id_availability'])]
#[ORM\Entity]
#[ORM\EntityListeners([TranslateListener::class, ProductListener::class])]
class Product
{
	public const UNITS = ['pieces', 'kg', 'g', 'l', 'dl', 'ml'];

	use TId;
	use TTranslateListener;
	use TExtraField;

	public const EXTRA_FIELD_SECTION = 'eshopCatalogProduct';

	#[ORM\Column(type: Types::SMALLINT, length: 1, options: ['default' => 0])]
	public int $isPublished = 0;

	#[ORM\Column(type: Types::SMALLINT, length: 1)]
	public int $inStock = 1;

	#[ORM\Column(type: Types::INTEGER, nullable: false, options: ['default' => 0])]
	public int $quantity;

	#[ORM\Column(type: Types::INTEGER, nullable: false, options: ['default' => 1, 'unsigned' => true, 'comment' => 'Minimum quantity of products per order'])]
	public int $minimumAmount = 1;

	#[ORM\Column(type: Types::INTEGER, nullable: true, options: ['unsigned' => true, 'comment' => 'Maximum quantity of products per order'])]
	public ?int $maximumAmount = null;

	/**
	 * @var double|null
	 */
	#[ORM\Column(type: Types::DECIMAL, precision: 10, scale: 2, nullable: true)]
	public $price;

	/**
	 * @var double|null
	 */
	#[ORM\Column(type: Types::DECIMAL, precision: 10, scale: 2, nullable: true)]
	public $retailPrice;

	/**
	 * @var double|null
	 */
	#[ORM\Column(type: Types::DECIMAL, precision: 10, scale: 2, nullable: true)]
	public $purchasePrice;

	/** @var Collection<string, ProductPrice> */
	#[ORM\OneToMany(targetEntity: ProductPrice::class, mappedBy: 'product', cascade: ['persist', 'remove'], indexBy: 'country')]
	public Collection $prices;

	/** @var Collection<ProductPriceLevelCountry> */
	#[ORM\OneToMany(targetEntity: ProductPriceLevelCountry::class, mappedBy: 'product', cascade: ['persist', 'remove'])]
	public Collection $priceLevelCountries;

	/**
	 * @var double|null
	 */
	#[ORM\Column(type: Types::DECIMAL, precision: 10, scale: 2, nullable: true)]
	public $recyclingFee;

	#[ORM\JoinColumn(name: 'id_manufacturer', referencedColumnName: 'id', nullable: true, onDelete: 'SET NULL')]
	#[ORM\ManyToOne(targetEntity: Manufacturer::class, cascade: ['persist'])]
	protected ?Manufacturer $manufacturer = null;

	/** @var Collection<string, ProductTexts> */
	#[ORM\OneToMany(targetEntity: ProductTexts::class, mappedBy: 'id', cascade: ['all'], indexBy: 'lang')]
	protected Collection $productTexts;

	/** @var Collection<CategoryProduct> */
	#[ORM\OneToMany(targetEntity: CategoryProduct::class, mappedBy: 'product', cascade: ['all'], indexBy: 'id_category')]
	public Collection $categoryProducts;

	#[ORM\JoinColumn(name: 'id_category_default', referencedColumnName: 'id', onDelete: 'SET NULL')]
	#[ORM\ManyToOne(targetEntity: Category::class, cascade: ['persist'])]
	protected ?Category $idCategoryDefault = null;

	/** @var Collection<FeatureProduct> */
	#[ORM\OneToMany(targetEntity: FeatureProduct::class, mappedBy: 'product', cascade: ['all'], indexBy: 'featureValue')]
	protected Collection $featureProducts;

	/** @var Collection<DynamicFeatureProduct> */
	#[ORM\OneToMany(targetEntity: DynamicFeatureProduct::class, mappedBy: 'product', cascade: ['all'], indexBy: 'id')]
	public Collection $dynamicFeatures;

	/** @var Collection<ProductTag> */
	#[ORM\OneToMany(targetEntity: ProductTag::class, mappedBy: 'product', cascade: ['all'], indexBy: 'id_tag')]
	public Collection $productTags;

	/** @var Collection<ProductSupplier> */
	#[ORM\OneToMany(targetEntity: ProductSupplier::class, mappedBy: 'product', cascade: ['all'], indexBy: 'id_supplier')]
	public Collection $suppliers;

	#[ORM\JoinColumn(name: 'id_vat_rate', referencedColumnName: 'id', onDelete: 'SET NULL')]
	#[ORM\ManyToOne(targetEntity: VatRate::class)]
	public ?VatRate $vatRate = null;

	#[ORM\JoinColumn(name: 'id_availability', referencedColumnName: 'id', onDelete: 'SET NULL')]
	#[ORM\ManyToOne(targetEntity: Availability::class)]
	protected ?Availability $availability = null;

	#[ORM\JoinColumn(name: 'id_availability_after_sold_out', referencedColumnName: 'id', nullable: true, onDelete: 'SET NULL')]
	#[ORM\ManyToOne(targetEntity: Availability::class)]
	public ?Availability $availabilityAfterSoldOut = null;

	#[ORM\Column(type: Types::STRING, length: 20, nullable: true)]
	public ?string $ean = null;

	#[ORM\Column(name: 'code1', type: Types::STRING, length: 60, nullable: true)]
	public ?string $code1 = null;

	#[ORM\Column(name: 'code2', type: Types::STRING, length: 60, nullable: true)]
	public ?string $code2 = null;

	#[ORM\JoinColumn(name: 'gallery_id', referencedColumnName: 'id', nullable: true, onDelete: 'SET NULL')]
	#[ORM\ManyToOne(targetEntity: Album::class, cascade: ['persist'])]
	protected ?Album $gallery = null;

	#[ORM\Column(type: Types::DATETIME_MUTABLE)]
	protected \DateTime $created;

	/** @Gedmo\Timestampable(on="update") */
	#[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: true)]
	protected ?\DateTime $modified;

	/** @var Collection<ProductPriceLevel> */
	#[ORM\OneToMany(targetEntity: ProductPriceLevel::class, mappedBy: 'productId', indexBy: 'group_id')]
	public Collection $priceLevels;

	#[ORM\Column(type: Types::JSON, nullable: true)]
	protected ?array $moreData = [];

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['default' => 0])]
	public int $unlimitedQuantity = 0;

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['default' => 0])]
	public int $discountDisabled = 0;

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['default' => 1])]
	protected int $categoryGiftsAllowed = 0;

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['default' => 1])]
	public int $orderGiftsAllowed = 1;

	/** @var Collection<string, ProductInSite> */
	#[ORM\OneToMany(targetEntity: ProductInSite::class, mappedBy: 'product', indexBy: 'site')]
	public Collection $sites;

	/** @var Collection<string, ProductVariant>|null */
	public ?Collection $variants = null;

	/** @var Collection<string, ProductVariant> */
	#[ORM\OneToMany(targetEntity: ProductVariant::class, mappedBy: 'product', cascade: ['persist'])]
	public Collection $isVariant;

	/** @var Collection<ProductDocument> */
	#[ORM\OneToMany(targetEntity: ProductDocument::class, mappedBy: 'product', cascade: ['all'])]
	public Collection $documents;

	/** @var Collection<ProductVideo> */
	#[ORM\OneToMany(targetEntity: ProductVideo::class, mappedBy: 'product', cascade: ['all'])]
	public Collection $videos;

	/** @var Collection<RelatedProduct> */
	#[ORM\OneToMany(targetEntity: RelatedProduct::class, mappedBy: 'origin', indexBy: 'product_id')]
	public Collection $relatedProducts;

	#[ORM\Column(type: Types::SMALLINT, length: 1, options: ['default' => 0])]
	public int $isAssort = 0;

	#[ORM\Column(name: '`condition`', type: Types::STRING, length: 255, nullable: true, options: ['default' => 'new'])]
	public ?string $condition = 'new';

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['default' => 0, 'unsigned' => true])]
	public int $disablePickUpSpedition = 0;

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['default' => 0, 'unsigned' => true])]
	public int $isOversize = 0;

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['default' => 0, 'unsigned' => true])]
	public int $disableStackInCart = 0;

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['default' => 0, 'unsigned' => true])]
	public int $disableCalculateFreeSpedition = 0;

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['default' => 0, 'unsigned' => true])]
	public int $isDiscount = 0;

	#[ORM\Column(type: Types::STRING, nullable: true)]
	public ?string $discountType = null;

	/**
	 * @var float|string|null
	 */
	#[ORM\Column(type: Types::DECIMAL, precision: 10, scale: 2, nullable: true)]
	public $discountValue = null;

	/** @var Collection<ProductSpedition> */
	#[ORM\OneToMany(targetEntity: ProductSpedition::class, mappedBy: 'product', indexBy: 'spedition_id')]
	public Collection $speditions;

	/** @var Collection<ProductPayment> */
	#[ORM\OneToMany(targetEntity: ProductPayment::class, mappedBy: 'product', indexBy: 'payment_id')]
	public Collection $payments;

	/**
	 * In mm
	 */
	#[ORM\Column(type: Types::INTEGER, nullable: true, options: ['unsigned' => true])]
	public ?int $width = null;

	/**
	 * In mm
	 */
	#[ORM\Column(type: Types::INTEGER, nullable: true, options: ['unsigned' => true])]
	public ?int $height = null;

	/**
	 * In mm
	 */
	#[ORM\Column(type: Types::INTEGER, nullable: true, options: ['unsigned' => true])]
	public ?int $depth = null;

	/**
	 * In g
	 */
	#[ORM\Column(type: Types::INTEGER, nullable: true, options: ['unsigned' => true])]
	public ?int $weight = null;

	#[ORM\Column(type: Types::INTEGER, nullable: true)]
	public ?int $position = null;

	#[ORM\JoinColumn(name: 'country_of_origin', referencedColumnName: 'id', nullable: true, onDelete: 'SET NULL')]
	#[ORM\ManyToOne(targetEntity: Country::class)]
	public ?Country $countryOfOrigin = null;

	#[ORM\Column(type: Types::INTEGER, length: 10, nullable: true, options: ['unsigned' => true])]
	public ?int $hsCustomsCode = null;

	#[ORM\Column(type: Types::INTEGER, nullable: false, options: ['unsigned' => true, 'default' => 0])]
	public int $isDeleted = 0;

	#[ORM\Column(type: Types::STRING, length: 255, nullable: false, options: ['default' => 'pieces'])]
	public string $units = 'pieces';

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['unsigned' => true, 'default' => 0])]
	public int $disableListing = 0;

	#[ORM\JoinColumn(name: 'package_id', referencedColumnName: 'id', nullable: true, onDelete: 'SET NULL')]
	#[ORM\ManyToOne(targetEntity: Package::class)]
	public ?Package $package = null;

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['unsigned' => true, 'default' => 0])]
	public int $verifyAge = 0;

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['unsigned' => true, 'default' => 0])]
	public int $skipForReplenishmentPlanning = 0;

	#[ORM\Column(type: Types::SMALLINT, nullable: false, options: ['unsigned' => true, 'default' => 0])]
	public int $disableRegisterSale = 0;

	public function __construct()
	{
		$this->productTexts         = new ArrayCollection;
		$this->categoryProducts     = new ArrayCollection;
		$this->featureProducts      = new ArrayCollection;
		$this->suppliers            = new ArrayCollection;
		$this->productTags          = new ArrayCollection;
		$this->quantity             = 0;
		$this->inStock              = 0;
		$this->moreData             = [];
		$this->created              = new DateTime('now');
		$this->priceLevels          = new ArrayCollection;
		$this->extraFields          = new ArrayCollection;
		$this->unlimitedQuantity    = 0;
		$this->discountDisabled     = 0;
		$this->categoryGiftsAllowed = 1;
		$this->isPublished          = 1;
		$this->isVariant            = new ArrayCollection;
		$this->sites                = new ArrayCollection;
		$this->variants             = new ArrayCollection;
		$this->documents            = new ArrayCollection;
		$this->relatedProducts      = new ArrayCollection;
		$this->speditions           = new ArrayCollection;
		$this->payments             = new ArrayCollection;
		$this->prices               = new ArrayCollection;
		$this->priceLevelCountries  = new ArrayCollection;
		$this->dynamicFeatures      = new ArrayCollection;
		$this->position             = null;
	}

	public function __clone()
	{
		$this->id                  = null;
		$this->quantity            = 0;
		$this->gallery             = null;
		$this->isVariant           = new ArrayCollection;
		$this->productTexts        = new ArrayCollection;
		$this->categoryProducts    = new ArrayCollection;
		$this->featureProducts     = new ArrayCollection;
		$this->dynamicFeatures     = new ArrayCollection;
		$this->suppliers           = new ArrayCollection;
		$this->productTags         = new ArrayCollection;
		$this->priceLevels         = new ArrayCollection;
		$this->extraFields         = new ArrayCollection;
		$this->sites               = new ArrayCollection;
		$this->variants            = new ArrayCollection;
		$this->documents           = new ArrayCollection;
		$this->relatedProducts     = new ArrayCollection;
		$this->created             = new DateTime('now');
		$this->prices              = new ArrayCollection;
		$this->priceLevelCountries = new ArrayCollection;
		$this->isDeleted           = 0;
	}

	public function clonePrices(Product $product): array
	{
		$prices = [];
		foreach ($this->prices->toArray() as $row) {
			$price          = new ProductPrice($product, $row->getCountry(), $row->currency);
			$price->price   = $row->price;
			$price->vatRate = $row->vatRate;

			if (Config::load('product.allowRetailPrice')) {
				$price->retailPrice = $row->retailPrice;
			}

			$prices[] = $price;
		}

		return $prices;
	}

	public function clonePriceLevelCounties(Product $product): array
	{
		$prices = [];
		foreach ($this->priceLevelCountries->toArray() as $row) {
			/** @var ProductPriceLevelCountry $row */
			$price        = new ProductPriceLevelCountry($product, $row->getGroup(), $row->getCountry(), $row->currency);
			$price->price = $row->price;

			$prices[] = $price;
		}

		return $prices;
	}

	public function addProductText(string $lang): void
	{
		$this->productTexts->set($lang, new ProductTexts($this, $lang));
	}

	/**
	 * @param ProductTexts[] $texts
	 */
	public function setProductTexts(array $texts): self
	{
		$this->productTexts = new ArrayCollection($texts);

		return $this;
	}

	public function setProductText(ProductTexts $productTexts): void
	{
		$this->productTexts->set($productTexts->getLang(), $productTexts);
	}

	public function getText(?string $lang = null): ?ProductTexts
	{
		return $this->productTexts->get($lang ?? $this->locale);
	}

	/**
	 * @return Collection<string, ProductTexts>
	 */
	public function getTexts(): Collection { return $this->productTexts; }

	/**
	 * @return Collection<FeatureProduct>
	 */
	public function getFeatureProducts(): Collection
	{
		return $this->featureProducts;
	}

	/**
	 * @param FeatureProduct $featureProduct
	 */
	public function addFeatureProduct($featureProduct): void
	{
		$this->featureProducts->add($featureProduct);
	}

	/**
	 * @param FeatureProduct[] $features
	 */
	public function setFeaturesProduct(array $features): void
	{
		$this->featureProducts = new ArrayCollection($features);
	}

	public function addCategoryProduct(CategoryProduct $categoryProduct): self
	{
		if (!$this->categoryProducts->contains($categoryProduct)) {
			$this->categoryProducts->add($categoryProduct);
		}

		return $this;
	}

	public function removeCategoryProduct(CategoryProduct $categoryProduct): void
	{
		$this->categoryProducts->removeElement($categoryProduct);
	}

	/**
	 * @return Collection<CategoryProduct>
	 */
	public function getCategoryProducts(): Collection
	{
		return $this->categoryProducts;
	}

	public function removeAllCategoryProducts(): void
	{
		$this->categoryProducts->clear();
	}

	/**
	 * @param CategoryProduct[] $categories
	 */
	public function setCategories(array $categories): self
	{
		$this->categoryProducts = new ArrayCollection($categories);

		return $this;
	}

	public function removeProductTag(ProductTag $tag): self
	{
		$this->productTags->removeElement($tag);

		return $this;
	}

	public function addProductTag(ProductTag $productTag): void
	{
		$productTag->setProduct($this);
		$this->productTags->add($productTag);
	}

	/**
	 * @return Collection<ProductTag>
	 */
	public function getProductTags(): Collection
	{
		return $this->productTags;
	}

	public function setManufacturer(?Manufacturer $manufacturer = null): void
	{
		$this->manufacturer = $manufacturer;
	}

	public function getManufacturer(): ?Manufacturer
	{
		return $this->manufacturer;
	}

	/**
	 * @deprecated
	 */
	public function setDefaultCategory(Category $category): void
	{
		$this->idCategoryDefault = $category;
	}

	/**
	 * @deprecated
	 */
	public function getDefaultCategory(): ?Category
	{
		return $this->idCategoryDefault;
	}

	public function setVateRate(VatRate $vatRate): void
	{
		$this->vatRate = $vatRate;
	}

	public function getVateRate(): ?VatRate
	{
		return $this->vatRate;
	}

	public function setAvailability(?Availability $availability = null): self
	{
		$this->availability = $availability;

		return $this;
	}

	public function getAvailability(): ?Availability { return $this->availability; }

	/*************************************
	 * == Gallery
	 */

	public function getGallery(): ?Album { return $this->gallery; }

	public function setGallery(Album $album): self
	{
		$this->gallery = $album;

		return $this;
	}

	public function getImage(): ?Image
	{
		$img = null;
		if ($this->gallery) {
			if ($this->isVariant() && $this->isVariant()->defaultImage) {
				$img = $this->gallery->getImages()[$this->isVariant()->defaultImage] ?? null;
			}

			if (!$img) {
				$img = $this->gallery->getCoverImage();
			}
		}

		return $img;
	}

	/*************************************
	 * == ProductPriceLevel
	 */

	/**
	 * @return Collection<ProductPriceLevel>
	 */
	public function getPriceLevels(): Collection
	{
		return $this->priceLevels;
	}

	/*************************************
	 * == Supplier
	 */

	public function getSupplier(int $id): ?ProductSupplier
	{
		return $this->suppliers->get($id);
	}

	/**
	 * @return Collection<ProductSupplier>
	 */
	public function getSuppliers(): Collection
	{
		return $this->suppliers;
	}

	public function setSupplier(ProductSupplier $ps): void
	{
		if (!$this->suppliers->containsKey($ps->getSupplier()->getId())) {
			$this->suppliers->clear();
			$this->suppliers->add($ps);
		}
	}

	/*************************************
	 * == Dates
	 */

	public function getCreated(): \DateTime { return $this->created; }

	public function getModified(): \DateTime { return $this->modified; }

	public function validateCreated(): void
	{
		if ($this->created->format('y-m-d') <= 0) {
			$this->created = new DateTime;
		}
	}

	/*************************************
	 * == MoreData
	 */

	/**
	 * @param mixed $default
	 *
	 * @return mixed|null
	 */
	public function getMoreDataValue(string $key, $default = null) { return $this->moreData[$key] ?? $default; }

	/**
	 * @param mixed $value
	 */
	public function setMoreDataValue(string $key, $value): self
	{
		if (!is_array($this->moreData)) {
			$this->moreData = [];
		}

		$this->moreData[$key] = $value;

		return $this;
	}

	public function setMoreData(ArrayHash|array|null $data): self
	{
		$this->moreData = $data ? (array) $data : null;

		return $this;
	}

	/**
	 * @param string|int $key
	 */
	public function removeMoreData($key): Product
	{
		unset($this->moreData[$key]);

		return $this;
	}

	/*************************************
	 * == Booleans
	 */

	public function isCategoryGiftsAllowed(): int { return (int) $this->categoryGiftsAllowed; }

	/**
	 * @param string|int $allowed
	 */
	public function setCategoryGiftsAllowed($allowed): self
	{
		$this->categoryGiftsAllowed = (int) $allowed;

		return $this;
	}

	public function getIsVariantParent(): bool { return $this->isVariant() && $this->isVariant()->isDefault === 1; }

	public function getVariantParent(): ?Product
	{
		if ($this->getIsVariantParent()) {
			return $this;
		}

		foreach ($this->variants->toArray() as $v) {
			if ($v->isDefault === 1) {
				return $v->getProduct();
			}
		}

		return null;
	}

	/**
	 * @return ProductDocument[]
	 */
	public function getDocuments(): array
	{
		return $this->documents->toArray();
	}

	public function addDocument(ProductDocument $document): void
	{
		$this->documents->add($document);
	}

	public function addVideo(ProductVideo $video): void
	{
		$this->videos->add($video);
	}

	/**
	 * @return ProductVideo[]
	 */
	public function getVideos(): array
	{
		return $this->videos->toArray();
	}

	/**
	 * @return Collection<RelatedProduct>
	 */
	public function getRelatedProducts(): Collection { return $this->relatedProducts; }

	public function addRelatedProduct(RelatedProduct $relatedProduct): void
	{
		$contains = false;
		foreach ($this->relatedProducts->getKeys() as $keys) {
			if ($keys === $relatedProduct->getProduct()->getId()) {
				$contains = true;
				break;
			}
		}

		if ($this->getId() !== $relatedProduct->getProduct()->getId() && !$contains) {
			$this->relatedProducts->set($relatedProduct->getProduct()->getId(), $relatedProduct);
		}
	}

	public function isVariant(): ?ProductVariant
	{
		return $this->isVariant->first() ?: null;
	}

	/**
	 * @return Collection<ProductSpedition>
	 */
	public function getSpeditions(): Collection { return $this->speditions; }

	/**
	 * @return ProductPriceLevelCountry[][]
	 */
	public function getPriceLevelCountriesIndexedByGroup(): array
	{
		$result = [];

		foreach ($this->priceLevelCountries->toArray() as $row) {
			/** @var ProductPriceLevelCountry $row */
			$result[$row->getGroup()->getId()][$row->getCountry()->getId()] = $row;
		}

		return $result;
	}

}
