<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Entities\Country;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;

#[ORM\Table(name: 'eshop_catalog__product_price')]
#[ORM\Entity]
class ProductPrice
{
	#[ORM\JoinColumn(name: 'product', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\Id]
	#[ORM\ManyToOne(targetEntity: Product::class, inversedBy: 'prices')]
	protected Product $product;

	#[ORM\JoinColumn(name: 'country', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\Id]
	#[ORM\ManyToOne(targetEntity: Country::class)]
	protected Country $country;

	#[ORM\Column(name: 'currency_code', type: Types::STRING, length: 255, nullable: true)]
	public ?string $currency = null;

	/**
	 * @var double|string|null
	 */
	#[ORM\Column(type: Types::DECIMAL, precision: 10, scale: 2, nullable: true)]
	public $price;

	/**
	 * @var double|string|null
	 */
	#[ORM\Column(type: Types::DECIMAL, precision: 10, scale: 2, nullable: true)]
	public $retailPrice;

	#[ORM\JoinColumn(name: 'vat_rate', referencedColumnName: 'id', onDelete: 'SET NULL')]
	#[ORM\ManyToOne(targetEntity: VatRate::class)]
	public ?VatRate $vatRate = null;

	public function __construct(Product $product, Country $country, ?string $currency)
	{
		$this->product  = $product;
		$this->country  = $country;
		$this->currency = $currency;
	}

	public function getProduct(): Product { return $this->product; }

	public function getCountry(): Country { return $this->country; }

	public function toArray(): array
	{
		return [
			'country'     => $this->getCountry()->getId(),
			'product'     => $this->getProduct()->getId(),
			'currency'    => $this->currency,
			'price'       => $this->price,
			'retailPrice' => $this->retailPrice,
			'vatRate'     => $this->vatRate ? $this->vatRate->getId() : null,
		];
	}

}
