<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Navigation;

use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Dao\SiteMapUrl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use DateTimeInterface;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\AbstractQuery;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\Query\Parameter;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Event\RouteInFindAliasEvent;
use EshopOrders\FrontModule\Model\Customers;
use Navigations\AdminModule\Components\NavigationForm;
use Navigations\Model\Event\RouteInEvent;
use Navigations\Model\Event\RouteOutEvent;
use Navigations\Model\Event\RouteUpdateCacheDep;
use Navigations\Model\Providers\INavigationItem;
use Nette\Caching\Cache;
use Nette\DI\Container;
use Nette\Http\Request;
use Nette\Http\Url;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;
use Override;
use Pages\AdminModule\Model\TemplatePages;
use Users\Model\Security\User;

class Category extends BaseNavigation implements INavigationItem
{
	protected $title = 'category';

	protected $presenter = 'EshopCatalog:Front:Default';
	protected $action    = 'category';

	public const CACHE_CATEGORY = 'eshopCatalogCategory';

	protected ?int $formRootId = null;

	public function __construct(
		protected Categories    $categoriesService,
		protected Products      $productsService,
		protected TemplatePages $templatePagesService,
		protected Request       $request,
		protected Home          $home,
		protected Container     $container,
		protected User          $user,
	)
	{
	}

	#[Override]
	public function navigationFormCustomHandle(NavigationForm $component, array $data = []): void
	{
		$this->formRootId = $this->getRootCategory($data['site']);
		$component->loadComponent($data['componentId']);
		$component->redrawControl('component');
	}

	protected function getRootCategory(string $site): ?int
	{
		$result = $this->categoriesService->getEr()->createQueryBuilder('c')->select('c.id')
			->innerJoin('c.categoryTexts', 'ct', Join::WITH, 'ct.alias = :alias')
			->setParameter('alias', $site)
			->getQuery()->setMaxResults(1)->getScalarResult();

		return isset($result[0]['id']) ? (int) $result[0]['id'] : null;
	}

	/**
	 * @inheritDoc
	 */
	#[Override]
	public function getFormContainer(BaseForm $baseForm): BaseContainer
	{
		$container = new BaseContainer();
		$rootId    = $this->formRootId;

		/** @var ArrayHash $values */
		$values = $baseForm->getUntrustedValues();

		if (!$rootId) {
			$rootId = $this->getRootCategory(
				$baseForm->getCustomData('loadComponent')['site'] ?: $values->site
			);
		}

		$categories = [];
		if ($rootId) {
			$qb = $this->categoriesService->getEr()->createQueryBuilder('c')->select('c.id, c.lvl, ct.name')
				->andWhere('c.root = :root')
				->andWhere('c.lvl > 0')
				->join('c.categoryTexts', 'ct')
				->setParameters(new ArrayCollection([new Parameter('root', $rootId)]))
				->groupBy('c.id')
				->addOrderBy('c.lft');

			if (Config::load('category.publishedByLang')) {
				$qb->andWhere('ct.isPublished = 1');
			} else {
				$qb->andWhere('c.isPublished = 1');
			}

			foreach ($qb->getQuery()->getArrayResult() as $category) {
				$title = ' ' . $category['name'];
				for ($i = 1; $i < $category['lvl']; $i++) {
					$title = '---' . $title;
				}

				$categories[$category['id']] = $title;
			}
		}

		$site  = $baseForm->getCustomData('loadComponent')['site'];
		$pages = $this->templatePagesService->getOptionsForSelectGrouped();
		$pages = (array) ($pages[$site ?? $baseForm->getComponent('site')->getValue()] ?? $pages);

		$container->addSelect('category', $this->t('eshopCatalog.navigationBuilder.category'), $categories)
			->setTranslator(null);
		$container->addSelect('templatePage', $this->t('eshopCatalog.navigationBuilder.template'), $pages)
			->setTranslator(null);

		/** @var NavigationForm $parent */
		$parent = $baseForm->getParent();
		if ($parent->lookup(null, false)) {
			$componentId = $parent->getParameter('componentId');
			if ($componentId) {
				$container->getComponent('category')->setHtmlAttribute(
					'data-custom-link',
					$parent->link('componentCustom!', [$componentId])
				);
			}
		} else {
			$baseForm->onAnchor[] = static function() use ($container, $baseForm, $parent): void {
				$container->getComponent('category')->setHtmlAttribute(
					'data-custom-link',
					$parent->link('componentCustom!', [$baseForm->getValues()->componentType])
				);
			};
		}

		return $container;
	}

	#[Override]
	public function getTemplateFile(): ?string
	{
		return __DIR__ . '/Category.latte';
	}

	#[Override]
	public function formSuccess(BaseForm $form, ArrayHash $values)
	{

	}

	#[Override]
	public function updateCacheDep(array &$cache, $urlParams)
	{
		// Přidat klíče pouze pokud se nejedná o filtr, jinak se to zasekne a nepůjdou upravit kategorie
		if (!isset($urlParams['productsFilter-filter']) && !isset($urlParams['productsFilter-sort'])) {
			$cache[Cache::TAGS][] = 'category/' . $urlParams['id'];
			$cache[Cache::TAGS][] = 'categories';
		}

		$this->eventDispatcher->dispatch(
			new RouteUpdateCacheDep($cache, $urlParams),
			BaseNavigation::class . '::routeUpdateCacheDep'
		);

		parent::updateCacheDep($cache, $urlParams);
	}

	/*******************************************************************************************************************
	 * ===========================  Route
	 */

	#[Override]
	public function presenterActionCheck($presenter, $action)
	{
		return $this->presenter == $presenter && $this->action == $action;
	}

	/**
	 * @inheritDoc
	 */
	#[Override]
	public function routerIn($navigation, $urlParams = [])
	{
		$parts    = explode('/', (string) $urlParams['path']);
		$category = null;

		$return = [
			'presenter' => $this->presenter,
			'action'    => $this->action,
			'id'        => null,
		];

		if (count($parts) > 1) {
			$useBaseCategoryInUrl = (bool) Config::load('product.useBaseCategoryInUrl');
			$urlPath              = trim((string) $urlParams['path'], '/');

			foreach (array_reverse($parts) as $alias) {
				foreach ($this->categoriesService->findByAlias($alias) as $c) {
					if (!$c->link) {
						continue;
					}

					$url = new Url($c->link);

					$catUrlPath = trim(urldecode($url->getPath()), '/');

					if ($catUrlPath === $urlPath || $catUrlPath === $urlParams['locale'] . '/' . $urlPath) {
						$category = $c->getId();
						break 2;
					} else if ($useBaseCategoryInUrl) {
						$tmp = $parts;
						array_pop($tmp);
						if ($catUrlPath === implode('/', $tmp)) {
							$productNavs = $this->container->getService('navigations.navigations')
								->getPublishedByComponent('eshopCatalog.navigation.home');

							if ($productNavs) {
								$productNav = array_values($productNavs)[0];
								$isProduct  = $this->home->routerIn($productNav, ['path' => $urlParams['path']]);

								if ($isProduct) {
									$isProduct['activeNavigation'] = $productNav;

									return $isProduct;
								}
							}
						}
					}
				}

				$this->eventDispatcher->dispatch(
					new RouteInFindAliasEvent($alias, $urlParams, $return),
					BaseNavigation::class . '::routeInFindAlias'
				);
				if (isset($return['id'])) {
					if (!isset($return['activeNavigation'])) {
						$return['activeNavigation'] = $navigation;
					}

					$category = $return['id'];
					break;
				}
			}
		}

		$return['id'] = $category ?: $navigation->componentParams['category'];

		$this->eventDispatcher->dispatch(
			new RouteInEvent($navigation, $urlParams, $return),
			BaseNavigation::class . '::routeIn'
		);

		return $return;
	}

	#[Override]
	public function routerOut(DaoNavigationItem $navigation, &$urlParams): ?string
	{
		$cfUrl  = '';
		$cfUrls = [];

		$categoryId = $urlParams['id'] ?? $navigation->componentParams['category'];
		$path       = $this->categoriesService->getQueryPath((int) $categoryId);
		$tmp        = [];

		if (Config::load('allowCategoryFullUrlField', false)) {
			$cfUrls = $this->categoriesService->getCategoryFullUrl(null, $navigation->getLang());
			$cfUrl  = $cfUrls[$categoryId] ?? '';
		}

		if (!$cfUrl) {
			if ($path && !$path[0]) {
				array_shift($path);
			}

			if (\str_contains((string) $urlParams['id'], '|')) {
				$tmp[] = preg_replace('/\/' . $this->translator->translate('default.urlPart.page') . '\/(\d*)/', '', ltrim($this->request->getUrl()->getPath(), '/'));
			} else if (count($path) > 1) {
				$parentFound = false;
				foreach (array_reverse($path) as $k => $c) {
					if ($c['id'] == $navigation->componentParams['category']) {
						$fullUrl      = $cfUrls[$c['id']] ?? null;
						$navLinkParts = is_string($navigation->link) ? explode('/', $navigation->link) : [];

						if ($fullUrl) {
							$tmp[] = $fullUrl;
						} else if ($navigation->alias) {
							$tmp[] = $navigation->alias;
						} else if ($navLinkParts !== []) {
							$tmp[] = end($navLinkParts);
						}

						$parentFound = true;
						break;
					} else {
						$tmp[] = $c['alias'];
					}
				}
				if (!$parentFound) {
					return null;
				}
				$tmp = array_reverse($tmp);
			} else {
				if (isset($urlParams['id']) && $navigation->componentParams['category'] != $urlParams['id']) {
					return null;
				}

				$tmp[] = $navigation->alias;
			}

			$cfUrl = implode('/', $tmp);
			/** @phpstan-ignore-next-line */
			if ($cfUrl !== '' && $cfUrl !== '0') {
				$cfUrl = '/' . ltrim($cfUrl, '-');
			}
		}

		$this->eventDispatcher->dispatch(
			new RouteOutEvent($urlParams, $navigation, $cfUrl),
			BaseNavigation::class . '::routeOut'
		);

		return $cfUrl ? ('/' . ltrim((string) $cfUrl, '/')) : null;
	}

	/**
	 * @inheritDoc
	 */
	#[Override]
	public function getComponentChild(string $lang, $navigation = null, $siteMapData = false)
	{
		$arr = [];

		$category = $this->categoriesService->getEr()->createQueryBuilder('c')->select('c.lft, c.gt, IDENTITY(c.root) as root')
			->where('c.id = :id')
			->join('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
			->setParameters(new ArrayCollection([new Parameter('lang', $lang), new Parameter('id', $navigation->componentParams['category'])]))
			->getQuery()->getOneOrNullResult(AbstractQuery::HYDRATE_SCALAR);

		$qb = $this->categoriesService->getEr()->createQueryBuilder('c')
			->select('c.id, ct.name, ct.seo, c.modified')
			->andWhere('c.lft >= :lft')
			->andWhere('c.gt <= :gt')
			->andWhere('c.root = :root')
			->join('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
			->setParameters(new ArrayCollection([new Parameter('lang', $lang), new Parameter('lft', $category['lft']), new Parameter('gt', $category['gt']), new Parameter('root', $category['root'])]));

		if (Config::load('category.publishedByLang')) {
			$qb->andWhere('ct.isPublished = 1');
		} else {
			$qb->andWhere('c.isPublished = 1');
		}

		foreach ($qb->getQuery()->getScalarResult() as $row) {
			$id = $row['id'];

			$data = [
				'action'    => $this->action,
				'id'        => $id,
				'presenter' => $this->presenter,
				'locale'    => $lang,
			];

			if ($siteMapData) {
				$siteMapUrl = new SiteMapUrl();
				if ($row['modified']) {
					if ($row['modified'] instanceof DateTimeInterface) {
						$siteMapUrl->setModified($row['modified']);
					} else {
						$siteMapUrl->setModified(new DateTime($row['modified']));
					}
				}

				$seo = $row['seo'] ? unserialize($row['seo']) : null;
				if ($seo) {
					if (!isset($seo['addToSiteMap'])) {
						$seo['addToSiteMap'] = '1';
					}

					if (
						$seo['addToSiteMap'] === '0'
						|| (is_string($seo['robots']) && str_contains('noindex', $seo['robots']))
					) {
						continue;
					}

					$siteMapUrl->setSeo($seo);
				}

				$data['siteMapData']['default'] = $siteMapUrl;
			}
			$arr[] = $data;
		}

		return $arr;
	}

	public function checkAccess(DaoNavigationItem $nav): bool
	{
		$categoryId = (int) $nav->componentParams['category'];

		if ($categoryId && $this->container->hasService('eshopOrders.front.customers')) {
			/** @var Customers $customerService */
			$customerService = $this->container->getService('eshopOrders.front.customers');

			$customer = $this->user->isLoggedIn()
				/** @phpstan-ignore-next-line */
				? $customerService->getByUser($this->user->getIdentity())
				: null;
			$category = $this->categoriesService->get($categoryId);

			return $category && $this->categoriesService->checkCategoryRestrictionAccess($category->allowedCustomerGroups, $customer);
		}

		return true;
	}
}
