<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Navigation;

use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Dao\SiteMapUrl;
use Core\Model\Sites;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Doctrine\ORM\NonUniqueResultException;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Dao\Category;
use EshopCatalog\FrontModule\Model\ProductQuery;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Event\RouteInFindAliasEvent;
use EshopCatalog\Model\Helpers\ProductsHelper;
use EshopCatalog\Model\IProductsOldUrlResolver;
use Gallery\FrontModule\Model\Albums;
use Navigations\AdminModule\Components\NavigationForm;
use Navigations\Model\Event\RouteInEvent;
use Navigations\Model\Event\RouteOutEvent;
use Navigations\Model\Helper\NavigationsHelper;
use Navigations\Model\NavigationConfig;
use Navigations\Model\Providers\INavigationItem;
use Nette\Application\LinkGenerator;
use Nette\Application\UI\InvalidLinkException;
use Nette\Caching\Cache;
use Nette\DI\Container;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;
use Nette\Utils\Strings;
use Override;
use Pages\AdminModule\Model\TemplatePages;
use Throwable;

class Home extends BaseNavigation implements INavigationItem
{
	public const string COMPONENT_NAME    = 'eshopCatalog.navigation.home';
	public const string DEFAULT_PRESENTER = 'EshopCatalog:Front:Default';
	public const string ACTION_PRODUCT    = 'product';
	public const string ACTION_CATEGORY   = 'category';
	public const string CACHE_PRODUCT     = 'eshopCatalogProduct';

	protected $title  = 'home';
	protected $action = 'default';

	public ?LinkGenerator $linkGenerator       = null;
	protected int         $componentChildStart = 0;
	protected int         $componentChildLimit = 4000;
	protected ?string     $lastSite            = null;
	protected ?string     $lastLang            = null;

	protected array $checkedCategories = [];

	public function __construct(
		protected TemplatePages  $templatePagesService,
		protected ProductsFacade $productsFacade,
		protected ProductsHelper $productsHelper,
		protected Sites          $sites,
		protected Container      $container,
		protected Categories     $categories,
		protected Albums         $albums,
	)
	{
	}

	/**
	 * @inheritDoc
	 */
	#[Override]
	public function getFormContainer(BaseForm $baseForm): BaseContainer
	{
		$container = new BaseContainer;

		$site  = $baseForm->getCustomData('loadComponent')['site'];
		$pages = $this->templatePagesService->getOptionsForSelectGrouped();
		$pages = (array) ($pages[$site ?? $baseForm->getComponent('site')->getValue()] ?? $pages);

		$container->addSelect('templateHome', 'eshopCatalog.navigationBuilder.templateHome', $pages);
		$container->addSelect('templateProduct', 'eshopCatalog.navigationBuilder.templateDetail', $pages);
		$container->addSelect('templateCategory', 'eshopCatalog.navigationBuilder.templateCategory', $pages);

		/** @var NavigationForm $parent */
		$parent = $baseForm->getParent();
		if ($parent->lookup(null, false)) {
			$componentId = $parent->getParameter('componentId');
			if ($componentId) {
				$container->getComponent('templateHome')->setHtmlAttribute('data-custom-link', $parent->link('componentCustom!', [$componentId]));
			}
		} else {
			$baseForm->onAnchor[] = static function() use ($container, $baseForm, $parent): void {
				$container->getComponent('templateHome')->setHtmlAttribute('data-custom-link', $parent->link('componentCustom!', [$baseForm->getValues()->componentType]));
			};
		}

		return $container;
	}

	#[Override]
	public function formSuccess(BaseForm $form, ArrayHash $values)
	{

	}

	#[Override]
	public function updateCacheDep(array &$cache, $urlParams)
	{
		$cache[Cache::TAGS][] = self::CACHE_PRODUCT . '/' . $urlParams['id'];

		parent::updateCacheDep($cache, $urlParams);
	}

	/*******************************************************************************************************************
	 * ===========================  Route
	 */

	#[Override]
	public function presenterActionCheck($presenter, $action)
	{
		return self::DEFAULT_PRESENTER == $presenter && in_array($action, [$this->action, self::ACTION_PRODUCT,
				self::ACTION_CATEGORY]);
	}

	/**
	 * @param DaoNavigationItem $navigation
	 * @param array             $urlParams
	 *
	 * @return array|mixed
	 * @throws NonUniqueResultException
	 * @throws InvalidLinkException
	 * @throws Throwable
	 */
	#[Override]
	public function routerIn($navigation, $urlParams = [])
	{
		$returnDefault = [
			'presenter' => self::DEFAULT_PRESENTER,
			'action'    => $this->action,
			'id'        => null,
		];

		if (isset($urlParams['getHomepage'])) {
			return $returnDefault;
		}

		if (isset($urlParams['path']) && $navigation->alias != $urlParams['path']) {
			$params    = explode('/', (string) $urlParams['path']);
			$lastParam = NavigationsHelper::removeUrlSuffix(end($params));
			$id        = explode('-', $lastParam, 2);
			$resolver  = Config::load('useOldUrls') ? $this->productsHelper->getOldUrlResolver() : null;

			if ($resolver instanceof IProductsOldUrlResolver) {
				$return = $resolver->routeIn($navigation, $urlParams, $returnDefault);

				if ($return) {
					return $return;
				}
			}

			if (Config::load('product.removeIdFromUrl')) {
				$pathWithoutProduct = Strings::substring($urlParams['path'], 0, -Strings::length(end($params)));
				$pathWithoutProduct = Strings::trim($pathWithoutProduct, '/');

				$aliasId = null;
				foreach ($this->productsFacade->productsService->findIdByAlias($lastParam) as $row) {
					$categoryPrefix = $this->getCategoryPath((int) $row['category']);

					if (\str_ends_with($pathWithoutProduct, trim($categoryPrefix, '/'))) {
						$aliasId = $row['id'];
						break;
					}

					if ($aliasId === null) {
						$aliasId = $row['id'];
					}
				}

				if ($aliasId) {
					return [
						'presenter' => self::DEFAULT_PRESENTER,
						'action'    => self::ACTION_PRODUCT,
						'id'        => $aliasId,
					];
				}
			}

			if (is_numeric($id[0]) && $this->productsFacade->getProduct((int) $id[0])) {
				return [
					'presenter' => self::DEFAULT_PRESENTER,
					'action'    => self::ACTION_PRODUCT,
					'id'        => $id[0],
				];
			}

			$return = [];
			$this->eventDispatcher->dispatch(new RouteInFindAliasEvent($urlParams['path'], $urlParams, $return), BaseNavigation::class . '::routeInFindAlias');
			if ($return) {
				if (!isset($return['activeNavigation'])) {
					$return['activeNavigation'] = $navigation;
				}

				return $return;
			}
		}

		$returnEvent = [];
		$this->eventDispatcher->dispatch(new RouteInEvent(null, $urlParams, $returnEvent), Home::class . '::routeInStart');
		if ($returnEvent) {
			return $returnEvent;
		}

		if ($urlParams['path'] === $navigation->alias) {
			return $returnDefault;
		}

		if (isset($id) && is_numeric($id[0])) {
			// Pokud se nenalezne produkt, tak zkusime najit zneverejneny produkt
			ProductQuery::$ignoreProductPublish = true;
			ProductsFacade::$ignoreIsActive     = true;
			$product                            = $this->productsFacade->getProduct((int) $id[0]);
			ProductQuery::$ignoreProductPublish = false;
			ProductsFacade::$ignoreIsActive     = false;

			if ($product && !$product->isActive) {
				// Pokud je produkt nalezen, zkontrolujeme hlavni hlavni variantu a pripadne presmerujeme na ni
				if ($product->variantOf) {
					$baseVariant = $this->productsFacade->getProduct((int) $product->variantOf);

					if ($baseVariant && $baseVariant->isActive && $baseVariant->link) {
						header('Location: ' . $baseVariant->link, true, 302);
						exit;
					}
				}

				// Pokud neni varianta, zkontrolujeme kategorii a pripadne presmerujeme na ni
				if ($product->defaultCategory && $product->defaultCategory->link) {
					header('Location: ' . $product->defaultCategory->link, true, 302);
					exit;
				}
			}
		}

		return null;
	}

	#[Override]
	public function routerOut(DaoNavigationItem $navigation, &$urlParams): ?string
	{
		// Modifikace url pri pouziti virtualni kategorie
		if ($navigation->getParam('url')) {
			$cfUrl = $navigation->getParam('url');
		} else {
			$cfUrl = $navigation->isHomepage ? '/' : '/' . $navigation->alias;
		}

		if ($urlParams['action'] == self::ACTION_PRODUCT && isset($urlParams['id'])) {
			$resolver = Config::load('useOldUrls') ? $this->productsHelper->getOldUrlResolver() : null;

			if ($resolver instanceof IProductsOldUrlResolver) {
				$out = $resolver->routeOut($navigation, $urlParams);

				if ($out) {
					return $out;
				}
			}

			$useBaseCategoryInUrl = (bool) Config::load('product.useBaseCategoryInUrl');

			$product = $this->productsFacade->productsService->getForNavigation((int) $urlParams['id'], $useBaseCategoryInUrl, $urlParams['locale']);

			if ($product) {
				if ((string) $product['alias'] === '') {
					return null;
				}

				$return = ($navigation->isHomepage ? '' : '/' . $navigation->alias);

				if (!Config::load('product.removeIdFromUrl')) {
					$return .= '/' . $product['id'] . '-';
				}

				$return .= Strings::webalize($product['alias']);

				if ($useBaseCategoryInUrl && $product['category']) {
					$categoryPrefix = $this->getCategoryPath((int) $product['category']);

					if ($categoryPrefix !== '' && $categoryPrefix !== '0') {
						$return = '/' . $categoryPrefix . '/' . trim($return, '/');
					}
				}

				if (NavigationConfig::load('urlSuffix')) {
					$return .= NavigationConfig::loadScalar('urlSuffix');
				}

				return $return;
			}
		} else {
			$this->eventDispatcher->dispatch(new RouteOutEvent($urlParams, $navigation, $cfUrl), BaseNavigation::class . '::routeOut');
		}

		return '/' . ltrim((string) $cfUrl, '/');
	}

	/**
	 * @inheritDoc
	 */
	#[Override]
	public function getComponentChild(string $lang, $navigation = null, $siteMapData = false)
	{
		$i           = 0;
		$arr         = [];
		$currentSite = $this->sites->getCurrentSite();

		if ($this->lastSite !== $currentSite->getIdent() || $lang !== $this->lastLang) {
			$this->componentChildStart = 0;
		}

		$categories = $this->categories->getCategories($this->categories->getRootIdForSite($currentSite->getIdent()));

		foreach ((new ProductQuery($lang))
			         ->withTexts()
			         ->inSite($currentSite->getIdent())
			         ->hasPrice()
			         ->getQueryBuilder($this->productsFacade->productsService->getEr())
			         ->select('p.id, pt.name, p.modified, pt.seo, p.modified, IDENTITY(p.gallery) as galleryId, IDENTITY(sites.category) as catId')
			         ->andWhere('sites.isActive = 1')
			         ->andWhere('p.isDeleted = 0')
			         ->andWhere('p.disableListing = 0')
			         ->setFirstResult($this->componentChildStart)->setMaxResults($this->componentChildLimit)
			         ->getQuery()->getScalarResult() as $row) {
			$i++;

			if (!$row['name'] || !$row['catId']) {
				continue;
			}

			if (!isset($categories[$row['catId']]) || !$this->checkCategoryAccess($categories[$row['catId']])) {
				continue;
			}


			$id = $row['id'];

			$data = [
				'presenter' => self::DEFAULT_PRESENTER,
				'action'    => self::ACTION_PRODUCT,
				'id'        => $id,
				'locale'    => $lang,
			];

			if ($siteMapData) {
				$siteMapUrl = new SiteMapUrl();
				if ($row['modified']) {
					$siteMapUrl->setModified(DateTime::from($row['modified']));
				}

				$seo = $row['seo'] ? @unserialize($row['seo']) : null;

				if (!$seo) {
					$seo = [];
					foreach (['addToSiteMap', 'siteMapChangeFreq', 'siteMapPriority', 'robots'] as $col) {
						$seo[$col] = $navigation->getSeo($col);
					}
				}

				if (is_array($seo)) {
					if (!isset($seo['addToSiteMap'])) {
						$seo['addToSiteMap'] = '1';
					}
					if ($seo['addToSiteMap'] === '0' || (is_string($seo['robots']) && str_contains('noindex', $seo['robots']))) {
						continue;
					}

					$siteMapUrl->setSeo($seo);
				}

				if ($row['galleryId']) {
					$gallery = $this->albums->getAlbums([$row['galleryId']], 'eshopCatalog%siteIdent%PlaceholderImage')[$row['galleryId']] ?? null;

					if ($gallery) {
						$cover = $gallery->getCover();
						if ($cover) {
							$siteMapUrl->addImage($cover);
						}

						foreach ($gallery->getImages() as $img) {
							if ($cover && $img->id === $cover->id) {
								continue;
							}

							$siteMapUrl->addImage($img);
						}
					}
				}

				$data['siteMapData']['default'] = $siteMapUrl;
			}

			$arr[] = $data;
		}

		$this->productsFacade->productsService->em->clear();
		$this->productsFacade->clearTemp();

		$this->componentChildStart += $this->componentChildLimit;
		$this->lastSite            = $currentSite->getIdent();
		$this->lastLang            = $lang;

		if ($i >= $this->componentChildLimit) {
			$arr['continue'] = true;
		}

		return $arr;
	}

	protected function getCategoryPath(int $catId): string
	{
		$path = $this->categories->getQueryPath($catId);
		array_shift($path);

		return implode('/', array_map(static fn($v) => $v['alias'], $path));
	}

	protected function checkCategoryAccess(Category $category): bool
	{
		if (!isset($this->checkedCategories[$category->getId()])) {
			$this->checkedCategories[$category->getId()] = $this->categories->checkCategoryRestrictionAccess($category->allowedCustomerGroups, null);
		}

		return $this->checkedCategories[$category->getId()];
	}
}
