<?php declare(strict_types = 1);

namespace EshopCatalog\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping\ClassMetadataInfo;
use EshopCatalog\AdminModule\Model\Helpers\PricesHelper;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductInSite;
use EshopCatalog\Model\Entities\ProductPrice;
use EshopCatalog\Model\Entities\ProductPriceHistory;
use EshopCatalog\Model\Entities\ProductPriceLevel;
use EshopCatalog\Model\Entities\ProductTag;
use EshopCatalog\Model\Entities\ProductTexts;
use EshopCatalog\Model\Entities\ProductVariant;
use EshopGifts\DI\EshopGiftsExtension;
use EshopGifts\Model\Entities\ProductGift;
use EshopOrders\Model\Entities\GroupCustomers;
use Nette\Caching\Cache;

class Products
{
	protected array        $cBaseCatsForProducts = [];
	protected static array $processed            = [];
	protected static array $postUpdateDone       = [];
	protected static array $parentVariants       = [];

	public function __construct(
		protected EntityManagerDecorator $em,
		protected PricesHelper           $pricesHelper,
		protected CacheService           $cacheService,
	)
	{
	}

	public function getBaseCats(int $productId): array
	{
		if (!isset($this->cBaseCatsForProducts[$productId])) {
			$this->cBaseCatsForProducts[$productId] = [];

			$q = $this->em
				->createQuery("SELECT IDENTITY(pis.category) as cat FROM " . ProductInSite::class . " pis WHERE pis.product = :prod AND pis.isActive = :active");
			foreach ($q->execute([
				'prod'   => $productId,
				'active' => 1,
			]) as $row) {
				$this->cBaseCatsForProducts[$productId][] = $row['cat'];
			}
		}

		return $this->cBaseCatsForProducts[$productId];
	}

	public function prepareUpdateVariant(Product $product): void
	{
		if ($product->isVariant()->isDefault === 1) {
			$variants = $product->variants instanceof Collection ? $product->variants->toArray() : [];
			if (!$variants) {
				$metadata = $this->em->getClassMetadata($product->isVariant()::class);
				$table    = $metadata->getTableName();

				$varIds = [];
				foreach ($this->em->getConnection()
					         ->fetchAllAssociative("SELECT product_id FROM $table WHERE variant_id = ?", [$product->isVariant()->getVariantId()]) as $row) {
					$varIds[] = $row['product_id'];
				}

				$variants = $this->em->getRepository(ProductVariant::class)->createQueryBuilder('pv')
					->where('pv.product IN (:ids)')->setParameter('ids', $varIds)
					->getQuery()->getResult();
			}
			foreach ($variants as $va) {
				/** @var ProductVariant $va */
				$vaProd = $va->getProduct();

				if ($vaProd->getId() === $product->getId() || $va->isDefault === 1) {
					continue;
				}

				$this->updateVariant($va, $product);
			}
		} else if (!in_array($product->getId(), self::$postUpdateDone)) {
			self::$postUpdateDone[] = $product->getId();
			$va                     = $product->isVariant();
			if ($va instanceof ProductVariant) {
				if (!isset(self::$parentVariants[$va->getVariantId()])) {
					self::$parentVariants[$va->getVariantId()] = $this->em->getRepository(Product::class)
						->createQueryBuilder('p')
						->innerJoin('p.isVariant', 'va')
						->where('va.variantId = :variantId')
						->setParameter('variantId', $va->getVariantId())
						->andWhere('va.isDefault = 1')->getQuery()
						->getOneOrNullResult();
				}
				$parent = self::$parentVariants[$va->getVariantId()];

				if ($parent) {
					$this->updateVariant($va, $parent);
				}
			}
		}
	}

	public function updateVariant(ProductVariant $va, Product $parent): void
	{
		$em     = $this->em;
		$vaProd = $va->getProduct();

		if (in_array($vaProd->getId(), self::$processed)) {
			return;
		}

		self::$processed[] = $vaProd->getId();

		/** @var ProductTexts[] $vaProdTexts */
		$vaProdTexts = $vaProd->getTexts()->toArray();

		$va->createdDefault = $parent->getCreated();

		foreach ($parent->getTexts()->toArray() as $lang => $text) {
			if (!isset($vaProdTexts[$lang])) {
				$vaProdTexts[$lang] = new ProductTexts($vaProd, $lang);
				$vaProdTexts[$lang]->setName($text->name);
				$vaProdTexts[$lang]->setName2($text->name2);
				$vaProdTexts[$lang]->shortDescription = $text->shortDescription;
				$vaProdTexts[$lang]->description      = $text->description;
			} else {
				/** @var ProductTexts $text */
				if ($va->useName === 0) {
					$vaProdTexts[$lang]->setName($text->name);
				}
				if ($va->useName2 === 0) {
					$vaProdTexts[$lang]->setName2($text->name2);
				}
				if ($va->useShortDescription === 0) {
					$vaProdTexts[$lang]->shortDescription = $text->shortDescription;
				}
				if ($va->useDescription === 0) {
					$vaProdTexts[$lang]->description = $text->description;
				}
			}

			$em->persist($vaProdTexts[$lang]);
		}

		if ($va->usePriceLevels === 0) {
			$vaPriceLevels = $vaProd->getPriceLevels()->toArray();
			foreach ($parent->getPriceLevels()->toArray() as $id => $v) {
				/** @var ProductPriceLevel $v */
				/** @var GroupCustomers $groupCustomer */
				$groupCustomer = $em->getReference(GroupCustomers::class, $id);
				$g             = $vaPriceLevels[$id] ?? new ProductPriceLevel($vaProd, $groupCustomer);

				$g->price = $v->price;

				$em->persist($g);
			}

			$priceLevelCountries = [];
			foreach ($parent->getPriceLevelCountriesIndexedByGroup() as $groupId => $countries) {
				foreach ($countries as $countryId => $row) {
					$priceLevelCountries[$groupId][$countryId] = $row->toArray();
				}
			}

			$this->pricesHelper->savePriceLevelCountriesToProduct($vaProd, $priceLevelCountries);
		}

		$productGiftClass = ProductGift::class;
		if (!$va->useGifts && class_exists(EshopGiftsExtension::class) && class_exists($productGiftClass)) {
			$vaGifts   = [];
			$prodGifts = [];
			foreach ($em->getRepository($productGiftClass)->createQueryBuilder('g')
				         ->where('g.product = :id')->setParameter('id', $vaProd->getId())
				         ->getQuery()->getResult() as $row) {
				$vaGifts[$row->getId()] = $row;
			}
			foreach ($em->getRepository($productGiftClass)->createQueryBuilder('g')
				         ->where('g.product = :id')->setParameter('id', $parent->getId())
				         ->getQuery()->getResult() as $row) {
				$prodGifts[$row->getId()] = $row;
			}

			foreach ($prodGifts as $k => $v) {
				$g = $vaGifts[$k] ?? new $productGiftClass($v->getGift(), $vaProd);

				$g->dateFrom = $v->dateFrom;
				$g->dateTo   = $v->dateTo;
				$g->setActive($v->isActive());

				$em->persist($g);
			}
		}

		if ($va->useLabels === 0) {
			/** @var ProductTag[] $vaTags */
			$vaTags    = $vaProd->getProductTags()->toArray();
			$forRemove = $vaProd->getProductTags()->toArray();
			/** @var ProductTag[] $prodTags */
			$prodTags = [];
			foreach ($this->em->getRepository(ProductTag::class)->createQueryBuilder('pt')
				         ->where('pt.product = :product')->setParameter('product', $parent->getId())
				         ->getQuery()->getResult() as $tagRow) {
				$prodTags[$tagRow->getTag()->getId()] = $tagRow;
			}

			foreach ($prodTags as $k => $v) {
				$g            = $vaTags[$k] ?? new ProductTag($vaProd, $v->getTag());
				$g->validFrom = $v->validFrom;
				$g->validTo   = $v->validTo;
				unset($forRemove[$k]);

				$em->persist($g);
				if (!isset($vaTags[$k])) {
					$vaProd->addProductTag($g);
				}
			}

			foreach ($forRemove as $v) {
				$vaProd->removeProductTag($v);
				$em->remove($v);
			}
		}

		if (Config::load('enableCountryPrices')) {
			$prices = [];
			foreach ($parent->prices->toArray() as $countryId => $row) {
				/** @var ProductPrice $row */
				$prices[$countryId] = $row->toArray();
				if ($va->usePrice !== 0) {
					$prices[$countryId] = [];
					continue;
				}

				if (Config::load('product.allowRetailPrice') && $va->useRetailPrice) {
					unset($prices[$countryId]['retailPrice']);
				}
			}

			$this->pricesHelper->savePricesToProduct($vaProd, $prices);
		}

		if ($va->usePrice === 0) {
			if (Config::load('enablePriceHistory') && $vaProd->price !== $parent->price) {
				$priceHistory = new ProductPriceHistory(
					$vaProd,
					(float) $vaProd->price,
					(float) $parent->price,
					'variantUpdate'
				);
				$this->em->persist($priceHistory);
			}

			$vaProd->price = $parent->price;
		}

		if (Config::load('product.allowRetailPrice') && !$va->useRetailPrice) {
			$vaProd->retailPrice = $parent->retailPrice;
		} else {
			$vaProd->retailPrice = null;
		}

		if ($va->useUnlimitedQuantity === 0) {
			$vaProd->unlimitedQuantity = $parent->unlimitedQuantity;
		}
		if ($va->useDiscountDisabled === 0) {
			$vaProd->discountDisabled = $parent->discountDisabled;
		}
		if ($va->useCategoryGiftsAllowed === 0) {
			$vaProd->setCategoryGiftsAllowed($parent->isCategoryGiftsAllowed());
		}

		if ($va->useOrderGiftsAllowed === 0) {
			$vaProd->orderGiftsAllowed = $parent->orderGiftsAllowed;
		}

		if ($va->useImportPrice === 0) {
			$vaProd->setMoreDataValue('stopImportPrice', $parent->getMoreDataValue('stopImportPrice'));
		}
		$vaProd->setMoreDataValue('stopImportQuantity', $parent->getMoreDataValue('stopImportQuantity'));

		$vaProd->setManufacturer($parent->getManufacturer());
		$vaProd->setVateRate($parent->getVateRate());

		$em->persist($vaProd);
		$em->flush();

		$em->persist($va);
		$em->flush();

		$this->cacheService->productCache->clean([
			Cache::Tags => ['variants'],
		]);
	}

}
