<?php declare(strict_types = 1);

namespace EshopCatalog\Model\TemplateTextType;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Sites;
use Core\Model\TemplateReader\Providers\ITemplateTextType;
use Core\Model\TemplateReader\Providers\TemplateTextType;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\QueryBuilder;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\FrontModule\Model\ProductQuery;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\FrontModule\Model\Tags;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Helpers\ProductsHelper;
use Nette\Caching\Cache;
use Override;

class ProductsWithTag extends TemplateTextType implements ITemplateTextType
{
	protected $type = 'eshopCatalogProductsWithTag';

	public function __construct(
		protected ProductsFacade         $productsFacade,
		protected EntityManagerDecorator $em,
		protected Translator             $translator,
		protected Sites                  $sitesService,
		protected Tags                   $tagsService,
		protected CacheService           $cacheService,
	)
	{
	}

	/**
	 * @inheritDoc
	 */
	#[Override]
	public function loadContainer($formContainer, $params = [], $texts = []): void
	{
		$container = $formContainer->addContainer($params['name']);

		if (!isset($params['tag'])) {
			$opts = ['' => ''];
			foreach ($this->tagsService->getAll() as $tag) {
				$opts[$tag->type] = $tag->name;
			}
			$container->addSelect('tag', 'eshopCatalog.templateTextFields.tagId', $opts);
		}

		if (!isset($params['limit'])) {
			$container->addText('limit', 'eshopCatalog.templateTextFields.limit')
				->setHtmlType('number')
				->setValue(6);
		}

		if (!isset($params['onlyInStock'])) {
			$container->addBool('onlyInStock', 'eshopCatalog.templateTextFields.onlyInStock');
		}

		if (!isset($params['onlyCanAddToCart'])) {
			$container->addBool('onlyCanAddToCart', 'eshopCatalog.templateTextFields.onlyCanAddToCart');
		}

		$form = $formContainer->getForm(false);
		if ($form) {
			$form->onSuccess[] = function(): void {
				$this->cacheService->defaultCache->clean([Cache::TAGS => ['productsByTag']]);
			};
		}
	}

	/**
	 * @return \EshopCatalog\FrontModule\Model\Dao\Product[]
	 */
	#[Override]
	public function render(array $params)
	{
		$default                    = $this->getDefault();
		$params['tag']              = $params['tag'] ?: $default['tag'] ?? null;
		$params['limit']            = $params['limit'] ?: $default['limit'] ?? null;
		$params['onlyInStock']      = $params['onlyInStock'] ?: $default['onlyInStock'] ?? false;
		$params['onlyCanAddToCart'] = $params['onlyCanAddToCart'] ?: $default['onlyCanAddToCart'] ?? false;

		if (!isset($params['toVar'])) {
			return [];
		}
		if (!isset($params['tag'])) {
			return [];
		}
		$params['tag'] = (string) $params['tag'];

		$data = $this->cacheService->defaultCache->load(
			'byTag_' . $params['tag'] . '_' . $this->sitesService->getCurrentSite()->getIdent() . '_' . $this->translator->getLocale(),
			function(&$dep) use ($params) {
				$dep = [
					Cache::TAGS   => ['productsByTag'],
					Cache::EXPIRE => '1 hour',
				];
				$tag = $this->tagsService->getAll()[$params['tag']] ?? null;
				if (!$tag) {
					return [];
				}

				$query = (new ProductQuery($this->translator->getLocale()))
					->withTexts(false)
					->withTag($tag->id)
					->hasPrice()
					->inSite($this->sitesService->getCurrentSite()->getIdent())
					->selectIds();

				$query = $query->getQueryBuilder($this->em->getRepository(Product::class));
				$query->setMaxResults(($params['limit'] ?: 20) * 6);

				/** @var QueryBuilder $query */
				if ($params['onlyInStock']) {
					$query->innerJoin('p.availability', 'av', Join::WITH, 'av.ident = :avIdent')
						->setParameter('avIdent', Availability::IN_STOCK);
				}

				if ($params['onlyCanAddToCart']) {
					$query->innerJoin('p.availability', 'av2', Join::WITH, 'av2.canAddToCart = 1');
				}

				return $query->getQuery()
					->getScalarResult();
			}
		);

		$ids = array_map(static fn($r): mixed => $r['id'], $data);
		shuffle($ids);

		$products = $this->productsFacade->getProducts($ids);
		if (Config::load('groupVariantsInList', false)) {
			$variants = [];
			foreach ($products as $product) {
				foreach ($product->variants as $variant) {
					$variants[$variant->getId()] = [
						'prod'      => $variant->getId(),
						'variantId' => $variant->variantId,
						'isDefault' => $variant->variantOf === $product->getId() ? 1 : 0,
					];
				}
			}

			$filteredProducts = ProductsHelper::filterGroupVariantsInList($ids, $variants);

			$products = array_intersect_key($products, array_flip($filteredProducts));
		}

		return array_slice($products, 0, (int) $params['limit'], true);
	}

}
