<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Categories;

use Core\Components\Flashes\Flashes;
use Core\Model\Helpers\Arrays;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\AdminModule\Model\AvailabilityService;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\Category;
use EshopCatalog\Model\Entities\CategoryTexts;
use Nette\Application\UI\Multiplier;
use Nette\Caching\Cache;
use Nette\Utils\Html;

class CategoriesGrid extends BaseControl
{
	protected string $siteIdent;

	protected Categories           $categoriesService;
	protected ICategoryFormFactory $categoryFormFactory;
	protected IRelatedFormFactory  $relatedFormFactory;
	protected CacheService         $cacheService;
	protected AvailabilityService  $availabilityService;

	protected ?array $cData                         = null;
	protected ?array $cDefaultCategoryProductsCount = null;
	protected ?array $cOtherCategoryProductsCount   = null;

	public function __construct(
		string               $siteIdent,
		Categories           $categories,
		ICategoryFormFactory $categoryFormFactory,
		IRelatedFormFactory  $relatedFormFactory,
		CacheService         $cacheService,
		AvailabilityService  $availabilityService
	)
	{
		$this->siteIdent           = $siteIdent;
		$this->categoriesService   = $categories;
		$this->categoryFormFactory = $categoryFormFactory;
		$this->relatedFormFactory  = $relatedFormFactory;
		$this->cacheService        = $cacheService;
		$this->availabilityService = $availabilityService;
	}

	public function render(): void
	{
		$this->template->categories = $this->getData()['tree'];
		$this->template->siteIdent  = $this->siteIdent;
		$this->template->render($this->getTemplateFile());
	}

	protected function canUpdateGrid(): bool { return $this->categoriesService->getSessionSection()->categoriesGridUpdateEnabled ?? true; }

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete(int $id): void
	{
		$presenter = $this->presenter;
		if ($this->categoriesService->removeCategory($id)) {
			$presenter->flashMessage('eshopCatalog.defaultGrid.removed', Flashes::FLASH_SUCCESS);
		} else {
			$presenter->flashMessage('eshopCatalog.defaultGrid.removeFailed', Flashes::FLASH_DANGER);
		}

		$this->cacheStorage->clean([Cache::TAGS => ['eshopNavigation']]);
		$this->cacheService->categoryCache->clean([Cache::TAGS => [\EshopCatalog\FrontModule\Model\Categories::CACHE_NAMESPACE]]);

		if ($presenter->isAjax()) {
			$this->redrawControl('list');
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	public function handleEdit(int $id): void
	{
		$category = $this->categoriesService->get($id);

		if ($category) {
			$this['categoryForm']->categoryId = $id;
			$this->template->modalTitle       = $this->t('eshopCatalog.title.editCategory') . ' ' . $category->getCategoryText()->getName();
			$this->template->modal            = 'categoryForm';
			$this->redrawControl('modal');
		}
	}

	public function handleRelated(int $id): void
	{
		$category = $this->categoriesService->get($id);

		if ($category) {
			$this->template->modalTitle = $this->t('eshopCatalog.title.editCategory') . ' ' . $category->getCategoryText()->getName();
			$this->template->modal      = 'relatedForm';
			$this->template->id         = $id;
			$this->redrawControl('modal');
		}
	}

	public function handleChangePosition(): void
	{
		$presenter = $this->presenter;
		$request   = $presenter->getHttpRequest();

		$id   = (int) $request->getPost('id');
		$move = (int) $request->getPost('move');

		if ($id) {
			$er  = $this->categoriesService->getEr();
			$cat = $er->find($id);

			if ($cat) {
				if ($move < 0) {
					$er->moveUp($cat, abs($move));
				} else if ($move > 0) {
					$er->moveDown($cat, $move);
				}

				$presenter->flashMessageSuccess('default.positionChanged');
			}
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		$this->em->flush();
		if ($id) {
			$this->categoriesService->cleanCacheDeep($id);
		}

		$this->cacheStorage->clean([Cache::TAGS => ['eshopNavigation']]);
		$this->cacheService->categoryCache->clean([Cache::TAGS => [\EshopCatalog\FrontModule\Model\Categories::CACHE_NAMESPACE]]);

		$presenter->redrawControl('flashes');
	}

	public function handleClearProductsNotDefaultNested(string $siteIdent, int $id): void
	{
		if (Config::load('category.allowClearProductsNotDefault') && $this->categoriesService->clearProductsNotDefaultCategoryNested($siteIdent, [$id])) {
			$this->presenter->flashMessageSuccess('default.saved');
		} else {
			$this->presenter->flashMessageDanger('default.error');
		}

		$this->presenter->redirect('this');
	}

	public function handleInvertPublish(int $id, ?string $lang = null): void
	{
		$presenter = $this->presenter;
		if (!$presenter) {
			return;
		}

		if ($this->categoriesService->invertPublish($id, $lang)) {
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.publishChanged');
		} else {
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.publishChangeFailed');
		}

		if ($this->canUpdateGrid()) {
			//			$this->redrawControl('publish' . $id);
			$this->redrawControl('list');
		}
		$presenter->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentCategoryForm(): CategoryForm
	{
		$control = $this->categoryFormFactory->create();
		if ($this->getParameter('id')) {
			$control->setCategory((int) $this->getParameter('id'));
		}

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			if ($this->canUpdateGrid()) {
				$this->redrawControl('list');
			}

			$this->presenter->redrawControl('flashes');
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			if ($this->canUpdateGrid()) {
				$this->redrawControl('list');
			}

			$this->presenter->payload->hideModal = true;
			$this->presenter->redrawControl('flashes');
		};
		$control['form']->getComponent('saveControl')->closeModalOnCancel();

		return $control;
	}

	protected function createComponentRelatedForm(): Multiplier
	{
		return new Multiplier(function($id): RelatedForm {
			return $this->relatedFormFactory->create((int) $id);
		});
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function getData(): array
	{
		if ($this->cData !== null) {
			return $this->cData;
		}

		$rootId = $this->categoriesService->getEr()->createQueryBuilder('c')
			->select('c.id')
			->innerJoin(CategoryTexts::class, 'ct', Join::WITH, 'c.id = ct.id')
			->where('c.lvl = :lvl')
			->andWhere('ct.alias = :alias')
			->setParameters([
				'lvl'   => 0,
				'alias' => $this->siteIdent,
			])->getQuery()->setMaxResults(1)->getArrayResult()[0]['id'] ?? null;

		if (!$rootId) {
			$newCategory     = new Category;
			$newCategoryText = new CategoryTexts($newCategory, $this->translator->getLocale());
			$newCategoryText->setName($this->siteIdent);

			$this->em->persist($newCategory)
				->persist($newCategoryText)
				->flush();

			$rootId = $newCategory->getId();
		}

		// Kategorie
		$qb   = $this->categoriesService->getEr()->createQueryBuilder('c', 'c.id')
			->select("c.id, c.isPublished, IDENTITY(c.parent) as parent, GROUP_CONCAT(ct.lang, ';', ct.name, ';', ct.isPublished, ';', ct.alias SEPARATOR '|') as texts")
			->leftJoin('c.categoryTexts', 'ct')
			->andWhere('c.root = :root')
			->setParameters([
				'root' => $rootId,
			])
			->groupBy('c.id')
			->addOrderBy('c.lft');
		$data = $qb->getQuery()->getArrayResult();

		foreach ($data as $k => &$row) {
			$texts = [];
			foreach (explode('|', $row['texts']) as $v) {
				$tmp = explode(';', $v);

				$texts[$tmp[0]] = [
					'name'        => $tmp[1],
					'alias'       => $tmp[3],
					'isPublished' => Config::load('category.publishedByLang') ? (int) $tmp[2] : (int) $row['isPublished'],
				];
			}

			$row['texts'] = $texts;

			// Počet produktů podle výchozí kategorie
			$tmp1 = $this->getProductsCountByDefaultCategory((int) $k);
			$tmp2 = $this->getProductsCountByOtherCategories((int) $k);

			$row['productsCount'] = [
				'inStock' => count($tmp1['inStock'] ?? []) + count($tmp2['inStock'] ?? []),
				'soldOut' => count($tmp1['soldOut'] ?? []) + count($tmp2['soldOut'] ?? []),
			];
		}

		$this->addPublishHtml($data);

		$this->cData = [
			'data' => $data,
			'tree' => Arrays::buildTree($data, 'parent', 'id', $rootId),
		];

		return $this->cData;
	}

	protected function getProductsCountByDefaultCategory(int $categoryId): array
	{
		$soldOut   = $this->availabilityService->getAllByIdent()[Availability::SOLD_OUT] ?? null;
		$soldOutId = $soldOut ? $soldOut->getId() : null;

		if ($this->cDefaultCategoryProductsCount === null) {
			$this->cDefaultCategoryProductsCount = [];

			foreach ($this->em->getConnection()->executeQuery("SELECT pis.product_id as id, pis.category_id as cat, p.id_availability as av FROM eshop_catalog__product_in_site pis
					INNER JOIN eshop_catalog__product p ON pis.product_id = p.id AND p.id_availability IS NOT NULL
					WHERE pis.site = ? AND pis.category_id IS NOT NULL", [$this->siteIdent])->iterateAssociative() as $row) {
				$isSoldOut = $row['av'] === $soldOutId;

				$this->cDefaultCategoryProductsCount[(int) $row['cat']][$isSoldOut ? 'soldOut' : 'inStock'][] = $row['id'];
			}
		}

		return $this->cDefaultCategoryProductsCount[$categoryId] ?: [];
	}

	protected function getProductsCountByOtherCategories(int $categoryId): array
	{
		$soldOut   = $this->availabilityService->getAllByIdent()[Availability::SOLD_OUT] ?? null;
		$soldOutId = $soldOut ? $soldOut->getId() : null;

		if ($this->cOtherCategoryProductsCount === null) {
			$this->cOtherCategoryProductsCount = [];

			foreach ($this->em->getConnection()->executeQuery("SELECT cp.id_product as id, cp.id_category as cat FROM eshop_catalog__category_product cp
					INNER JOIN eshop_catalog__product p ON p.id = cp.id_product AND p.id_availability IS NOT NULL")->iterateAssociative() as $row) {
				$isSoldOut = $row['av'] === $soldOutId;

				$this->cOtherCategoryProductsCount[(int) $row['cat']][$isSoldOut ? 'soldOut' : 'inStock'][] = $row['id'];
			}
		}

		return $this->cOtherCategoryProductsCount[$categoryId] ?: [];
	}

	protected function addPublishHtml(array &$data): void
	{
		foreach ($data as &$row) {
			$wrap = Html::el('div class=grid-langs');
			/** @var array<int, array{
			 *     class: string,
			 *     href: string,
			 *     text: ?string,
			 *     ico: ?Html,
			 * }> $links
			 */
			$links = [];

			if (Config::load('category.publishedByLang')) {
				foreach ($this->langsService->getLangs(false) as $lang) {
					$text = $row['texts'][$lang->getTag()];
					if ($text) {
						$links[] = [
							'class' => 'ajax btn btn-xs ' . ($text['isPublished'] ? 'btn-success' : 'btn-danger'),
							'href'  => $this->link('invertPublish!', [$row['id'], $lang->getTag()]),
							'text'  => $lang->getShortTitle(),
						];
					}
				}
			} else {
				$wrap->setAttribute('style', 'justify-content: center;');
				$text    = $row['texts'][$this->translator->getLocale()];
				$links[] = [
					'class' => 'ajax btn btn-xs ' . ($text['isPublished'] ? 'btn-success' : 'btn-danger'),
					'href'  => $this->link('invertPublish!', [$row['id']]),
					'ico'   => Html::el('i', [
						'class' => 'fa ' . ($text['isPublished'] ? 'fa-check' : 'fa-times'),
					]),
				];
			}

			foreach ($links as $link) {
				$el = Html::el('div', [
					'class' => 'grid-langs__lang',
					'style' => count($links) === 1 ? 'width: auto;' : '',
				]);

				$a = Html::el('a', [
					'class' => $link['class'],
					'href'  => $link['href'],
				]);

				if (isset($link['text'])) {
					$a->setText($link['text']);
				} else if (isset($link['ico'])) {
					$a->addHtml($link['ico']);
				}

				$el->addHtml($a);

				$wrap->addHtml($el);
			}

			$row['publishHtml'] = $wrap;
		}
	}

}
