<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Features;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\DataGrid\DataSource\DoctrineDataSource;
use EshopCatalog\AdminModule\Model\Features;
use EshopCatalog\AdminModule\Model\FeatureValues;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureValue;
use Nette\Application\AbortException;
use Nette\Application\BadRequestException;
use Nette\Utils\Html;

class FeatureValuesGrid extends BaseControl
{
	public ?Feature                $feature = null;
	protected Features             $featuresService;
	protected FeatureValues        $featureValuesServices;
	protected IProductsGridFactory $productsGridFactory;

	public function __construct(
		Features             $features,
		FeatureValues        $featureValues,
		IProductsGridFactory $productsGridFactory
	)
	{
		$this->featuresService       = $features;
		$this->featureValuesServices = $featureValues;
		$this->productsGridFactory   = $productsGridFactory;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	/**
	 * @param string|int $id
	 *
	 * @throws AbortException
	 */
	public function handleDelete($id): void
	{
		$presenter = $this->presenter;
		if ($this->featureValuesServices->remove($id)) {
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.removed');
		} else {
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.removeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	public function handleRepairPositions(): void
	{
		$this->featureValuesServices->repairPositions($this->feature->getId());
		$this->redirect('this');
	}

	public function handleMergeAllValues(): void
	{
		$this->featureValuesServices->mergeAll($this->feature->getId());
		$this->redirect('this');
	}

	public function handleSortAlphabeticallyFeatureValues(): void
	{
		$this->featureValuesServices->sortAlphabetically($this->feature->getId());
		$this->redirect('this');
	}

	public function handlePublish(array $id): void
	{
		$presenter = $this->presenter;

		if ($this->featureValuesServices->setPublishMultiple($id, 1)) {
			$presenter->flashMessageSuccess('default.published');
		} else {
			$presenter->flashMessageDanger('default.error');
		}

		$this['grid']->reload();
		$presenter->redrawControl('flashes');
	}

	public function handleUnPublish(array $id): void
	{
		$presenter = $this->presenter;

		if ($this->featureValuesServices->setPublishMultiple($id, 0)) {
			$presenter->flashMessageSuccess('default.unPublished');
		} else {
			$presenter->flashMessageDanger('default.error');
		}

		$this['grid']->reload();
		$presenter->redrawControl('flashes');
	}

	public function handleProducts(int $id): void
	{
		$this['productsGrid']->featureValueId = $id;

		$featureValue = $this->featureValuesServices->get($id);

		$this->template->modalTitle = $this->t('eshopCatalog.featuresGrid.productsOfFeatureValue') . ' ' . ($featureValue ? $featureValue->getFeatureValueText()->name : '');
		$this->template->modal      = 'productsGrid';
		$this->redrawControl('modal');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setDefaultPerPage(50);
		$grid->setRememberState();

		if (!$this->feature) {
			return $grid;
		}

		$qb = $this->featureValuesServices->getEr()->createQueryBuilder('fv')->addSelect('fvt')
			->join('fv.featureValueTexts', 'fvt', 'WITH', 'fvt.lang = :lang')
			->setParameter('lang', $this->translator->getLocale())
			->andWhere('fv.feature = :id')->setParameter('id', $this->feature->getId())
			->orderBy('fv.position');

		$productsInFeatureValue = [];
		$dataSource             = new DoctrineDataSource($qb, 'fv.id');

		$grid->setDataSource($dataSource);

		$dataSource->onDataLoaded[] = function($items) use (&$productsInFeatureValue) {
			/** @var FeatureValue[] $items */
			$ids = [];

			foreach ($items as $item) {
				$ids[] = $item->getId();
			}

			if ($ids) {
				foreach ($this->em->getConnection()->fetchAllAssociative("SELECT fvp.id_product, fvp.id_feature_value 
					FROM eshop_catalog__feature_product fvp
					WHERE fvp.id_feature_value IN (" . implode(',', $ids) . ")") as $row) {
					$productsInFeatureValue[$row['id_feature_value']][] = $row['id_product'];
				}
			}
		};


		$grid->setSortable();
		$grid->setSortableHandler('featureValuesGrid:gridSortableRow!');

		// Columns
		$grid->addColumnText('name', 'eshopCatalog.defaultGrid.value')->setRenderer(function($row) {
			return Html::el('a', ['href' => $this->presenter->link('VariantsFeatures:editFeatureValue', $row->getId())])
				->setText($row->getFeatureValueText($this->translator->getLocale())->name);
		});
		$grid->addColumnText('productsCount', 'eshopCatalog.featuresGrid.productsCount')->setRenderer(function($row) use (&$productsInFeatureValue) {
			return Html::el('a', [
				'href'  => $this->link('products!', $row->getId()),
				'class' => 'ajax',
			])
				->setText(number_format(count($productsInFeatureValue[$row->getId()] ?? []), 0, ',', ' '));
		})
			->setAlign('right');
		$grid->addColumnStatus('isPublished', 'default.isPublished')->setAlign('center')
			->addOption(1, 'default.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'default.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		// Filters
		$grid->addFilterText('name', '', 'fvt.name');
		$grid->addFilterSelect('isPublished', '', [
			''  => '',
			'1' => $this->t('default.yes'),
			'0' => $this->t('default.no'),
		]);

		$grid->getColumn('productsCount')->getElementPrototype('th')->addClass('w1nw');

		// Actions
		$grid->addAction('edit', '', 'VariantsFeatures:editFeatureValue')->setIcon('edit')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')->addClass('ajax')->setConfirm('default.reallyDelete');

		$grid->addGroupAction('default.publish')->onSelect[]   = [$this, 'handlePublish'];
		$grid->addGroupAction('default.unPublish')->onSelect[] = [$this, 'handleUnPublish'];

		return $grid;
	}

	protected function createComponentProductsGrid(): ProductsGrid
	{
		return $this->productsGridFactory->create();
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridPublishChange(string $id, string $newStatus): void
	{
		$presenter = $this->presenter;

		if ($this->featureValuesServices->setPublish((int) $id, (int) $newStatus)) {
			$presenter->flashMessageSuccess('default.publishChanged');
		} else {
			$presenter->flashMessageDanger('default.publishChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem((int) $id);
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	public function handleGridSortableRow(): void
	{
		$presenter = $this->presenter;
		$request   = $presenter->getHttpRequest();
		$id        = $request->getPost('id');
		$position  = $request->getPost('position');

		if ($id != null && $position != null && $this->featureValuesServices->setPosition($id, $position)) {
			$presenter->flashMessageSuccess('default.positionChanged');
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		if ($presenter->isAjax()) {
			$presenter->redrawControl('flashes');
			$this['grid']->reload();
		} else {
			$this->redirect('this');
		}
	}

	/*******************************************************************************************************************
	 * ==================  GET / SET
	 */

	/**
	 * @param string|int $id
	 *
	 * @throws BadRequestException
	 */
	public function setFeature($id): void
	{
		$this->feature = $this->featuresService->get($id);

		if (!$this->feature) {
			$this->presenter->error();
		}
	}

}
