<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Products;

use Contributte\Translation\Translator;
use Core\AdminModule\Model\Sites;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Entities\Site;
use Core\Model\UI\Form\BaseContainer;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\AdminModule\Model\CategoryProducts;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\Model\Entities\CategoryProduct;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductInSite;
use EshopCatalog\Model\Helpers\VariantsHelper;
use Nette\Utils\ArrayHash;

class CategoryContainer
{
	protected EntityManagerDecorator $em;
	protected Translator             $translator;
	protected Categories             $categoryServices;
	protected Products               $productsService;
	protected CategoryProducts       $categoryProductService;
	protected Sites                  $sitesService;
	protected VariantsHelper         $variantsHelper;

	public function __construct(
		EntityManagerDecorator $em,
		Categories             $categories,
		CategoryProducts       $categoryProducts,
		Products               $products,
		Translator             $translator,
		Sites                  $sites,
		VariantsHelper         $variantsHelper
	)
	{
		$this->em                     = $em;
		$this->categoryServices       = $categories;
		$this->translator             = $translator;
		$this->productsService        = $products;
		$this->categoryProductService = $categoryProducts;
		$this->sitesService           = $sites;
		$this->variantsHelper         = $variantsHelper;
	}

	public function getContainers(): BaseContainer
	{
		$categoriesContainer = new BaseContainer;
		$sites               = $this->sitesService->getAll();
		foreach ($sites as $site) {
			$categoryContainer = new BaseContainer;

			if (count($sites) > 1) {
				$categoryContainer->addBool('isActive', 'eshopCatalog.productForm.activeInEshop')->setDefaultValue(0);
			} else {
				$categoryContainer->addHidden('isActive', '1');
			}

			$trees = $this->categoryServices->getTreeForSelect($site->getIdent());

			$categoryContainer->addCustomData('trees', $trees);
			$categoryContainer->addCustomData('template', __DIR__ . '/CategoryContainer.latte');

			$categoryContainer->addRadioList('defaultCategory', 'eshopCatalog.productForm.defaultCategory',
				$this->categoryServices->getOptionsForSelect($site->getIdent()))
				->setHtmlAttribute('data-disable-toggle', 'true');
			$categoryContainer->addCheckboxList('category', 'eshopCatalog.productForm.categories',
				$this->categoryServices->getOptionsForSelect($site->getIdent()))
				->setHtmlAttribute('data-disable-toggle', 'true');

			$categoriesContainer->addComponent($categoryContainer, $site->getIdent());
		}

		return $categoriesContainer;
	}

	public function getContainer(string $siteIdent): BaseContainer
	{
		$container = new BaseContainer;

		$trees = $this->categoryServices->getTreeForSelect($siteIdent);

		$container->addCustomData('trees', $trees);
		$container->addCustomData('template', __DIR__ . '/CategoryContainer.latte');

		$container->addRadioList('defaultCategory', 'eshopCatalog.productForm.defaultCategory', $this->categoryServices->getOptionsForSelect($siteIdent))
			->setHtmlAttribute('data-disable-toggle', 'true');
		$container->addCheckboxList('category', 'eshopCatalog.productForm.categories', $this->categoryServices->getOptionsForSelect($siteIdent))
			->setHtmlAttribute('data-disable-toggle', 'true');

		return $container;
	}

	/**
	 * @param array|ArrayHash $values
	 */
	public function getFormData($values): array
	{
		return (array) $values;
	}

	/**
	 * @param BaseContainer $container
	 * @param array         $siteValues
	 * @param array         $allCategories
	 *
	 * @return void
	 */
	public function setDefaultsMultipleSite(BaseContainer $container, array $siteValues, array $allCategories): void
	{
		foreach ($siteValues as $siteIdent => $v) {
			/** @var ProductInSite $v */
			$formCat = $container->getComponent($siteIdent, false);

			if (!$formCat) {
				continue;
			}

			$formCat->setDefaults([
				'isActive'        => $v->isActive(),
				'defaultCategory' => $v->category ? $v->category->getId() : null,
				'category'        => [],
			]);
		}

		/** @var BaseContainer $component */
		foreach ($container->getComponents() as $component) {
			$component->setDefaults([
				'category' => $allCategories,
			]);
		}
	}

	public function saveDataMultipleSite(array $productsId, array $data, string $editType = 'default'): void
	{
		$allCategories = [];
		/** @var Product[] $products */
		$products = array_map(function($id) { return $this->productsService->getReference($id); }, $productsId);
		// Prepare variants
		$this->variantsHelper->getProductVariants($productsId);
		$siteIdents = $this->sitesService->getIdents(true);

		if ($editType === 'default') {
			foreach ($data as $siteIdent => $v) {
				/** @var Site|null $site */
				$site = isset($siteIdents[$siteIdent])
					? $this->em->getReference(Site::class, $siteIdent)
					: null;

				if (!$site) {
					continue;
				}

				$defaultCategory = $v->defaultCategory ? $this->categoryServices->getReference($v->defaultCategory) : null;

				foreach ($products as $product) {
					$productSite = $product->sites->get($siteIdent) ?? null;

					if (!$productSite) {
						$productSite = new ProductInSite($product, $site);
					}

					$productSite->setActive((int) (count($data) <= 1 ? 1 : $v->isActive));

					if ($defaultCategory) {
						$productSite->category = $defaultCategory;

						foreach ($v->category as $k1 => $v1) {
							if ($v->defaultCategory != $v1) {
								$allCategories[] = $v1;
							}
						}
					} else {
						$productSite->category = null;
					}

					$this->em->persist($productSite);
				}
			}

			$allCategories = array_unique($allCategories);

			$currentProductCategories = [];
			foreach ($this->em->getRepository(CategoryProduct::class)->createQueryBuilder('cp')
				         ->select('IDENTITY(cp.product) as product, IDENTITY(cp.category) as category')
				         ->where('cp.product IN (:ids)')
				         ->setParameter('ids', $productsId)
				         ->getQuery()->getScalarResult() as $row) {
				$currentProductCategories[$row['product']][] = $row['category'];
			}

			foreach ($products as $product) {
				$productCategories = isset($currentProductCategories[$product->getId()]) ? $currentProductCategories[$product->getId()] : [];

				// Smazat
				$remove = array_diff($productCategories, $allCategories);
				if (!empty($remove)) {
					$removeProds = array_merge([$product->getId()], $this->variantsHelper->getVariantsOfProduct($product->getId()));

					foreach ($removeProds as $rProd) {
						$this->em->createQueryBuilder()->delete(CategoryProduct::class, 'cp')
							->where('cp.product = :prod')
							->andWhere('cp.category IN (:categories)')
							->setParameters([
								'prod'       => $rProd,
								'categories' => $remove,
							])->getQuery()->execute();
					}
				}

				// Přidat
				foreach (array_diff($allCategories, $productCategories) as $id) {
					foreach (array_merge([$product->getId()], $this->variantsHelper->getVariantsOfProduct($product->getId())) as $prodId) {
						$this->em->getConnection()->executeStatement("INSERT IGNORE INTO eshop_catalog__category_product (id_product, id_category) VALUES (?, ?)",
							[$prodId, $id]);
					}
				}

				$this->em->persist($product);
			}
		} else {
			$currentOtherCategories = [];
			foreach ($this->em->getRepository(CategoryProduct::class)->createQueryBuilder('cp')
				         ->select('IDENTITY(cp.product) as product, IDENTITY(cp.category) as category')
				         ->where('cp.product IN (:ids)')
				         ->setParameter('ids', $productsId)
				         ->getQuery()->getScalarResult() as $row) {
				$currentOtherCategories[$row['product']][$row['category']] = $row['category'];
			}

			$categoriesToEdit = [];
			foreach ($data as $site => $rows) {
				$categoriesToEdit = array_merge($categoriesToEdit, $rows['category']);
			}

			foreach ($productsId as $productId) {
				if ($editType === 'replaceOtherCategories') {
					// add
					foreach (array_diff($categoriesToEdit, $currentOtherCategories[$productId]) as $cat) {
						foreach (array_merge([$productId], $this->variantsHelper->getVariantsOfProduct($productId)) as $prodId) {
							$this->em->getConnection()->executeStatement("INSERT IGNORE INTO eshop_catalog__category_product (id_product, id_category) VALUES (?, ?)",
								[$prodId, $cat]);
						}
					}

					// remove
					$remove = array_diff($currentOtherCategories[$productId], $categoriesToEdit);
					if ($remove) {
						foreach (array_merge([$productId], $this->variantsHelper->getVariantsOfProduct($productId)) as $prodId) {
							$this->em->createQueryBuilder()->delete(CategoryProduct::class, 'cp')
								->where('cp.product = :prod')
								->andWhere('cp.category IN (:categories)')
								->setParameters([
									'prod'       => $prodId,
									'categories' => $remove,
								])->getQuery()->execute();
						}
					}
				} else if ($editType === 'addOtherCategories') {
					// add
					foreach (array_diff($categoriesToEdit, $currentOtherCategories[$productId]) as $cat) {
						foreach (array_merge([$productId], $this->variantsHelper->getVariantsOfProduct((int) $productId)) as $prodId) {
							$this->em->getConnection()->executeStatement("INSERT IGNORE INTO eshop_catalog__category_product (id_product, id_category) VALUES (?, ?)",
								[$prodId, $cat]);
						}
					}
				}
			}
		}
	}

}
