<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Products;

use Core\FrontModule\Model\SeoContainer;
use Core\Model\Entities\ExtraField;
use Core\Model\Event\ComponentTemplateEvent;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\FormValidateEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\Helpers\Strings;
use Core\Model\Parameters;
use Core\Model\UI\AbstractPresenter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\BootstrapRenderer;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\AdminModule\Model\CategoryProducts;
use EshopCatalog\AdminModule\Model\FeatureProducts;
use EshopCatalog\AdminModule\Model\Features;
use EshopCatalog\AdminModule\Model\FormContainers\DocumentContainer;
use EshopCatalog\AdminModule\Model\FormContainers\ProductChangePlanContainer;
use EshopCatalog\AdminModule\Model\FormContainers\ProductPackageContainer;
use EshopCatalog\AdminModule\Model\FormContainers\ProductPaymentContainer;
use EshopCatalog\AdminModule\Model\FormContainers\ProductPriceLevelCountriesContainer;
use EshopCatalog\AdminModule\Model\FormContainers\ProductPricesContainer;
use EshopCatalog\AdminModule\Model\FormContainers\ProductSpeditionContainer;
use EshopCatalog\AdminModule\Model\FormContainers\ProductSupplierContainer;
use EshopCatalog\AdminModule\Model\FormContainers\RelatedProductContainer;
use EshopCatalog\AdminModule\Model\FormContainers\VideoContainer;
use EshopCatalog\AdminModule\Model\Helpers\ProductFormHelper;
use EshopCatalog\AdminModule\Model\Manufacturers;
use EshopCatalog\AdminModule\Model\ProductDocuments;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\AdminModule\Model\ProductTags;
use EshopCatalog\AdminModule\Model\ProductVideos;
use EshopCatalog\AdminModule\Model\Suppliers;
use EshopCatalog\AdminModule\Model\Tags as AdminTags;
use EshopCatalog\AdminModule\Model\VatRates;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\AvailabilityService;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\DynamicFeatureProduct;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureProduct;
use EshopCatalog\Model\Entities\FeatureValue;
use EshopCatalog\Model\Entities\FeatureValueTexts;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductListener;
use EshopCatalog\Model\Entities\ProductPriceHistory;
use EshopCatalog\Model\Entities\ProductPriceLevel;
use EshopCatalog\Model\Entities\ProductSupplier;
use EshopCatalog\Model\Entities\ProductTag;
use EshopCatalog\Model\Entities\ProductTexts;
use EshopCatalog\Model\Entities\ProductVariant;
use EshopCatalog\Model\Entities\ProductVariantText;
use EshopCatalog\Model\Entities\Supplier;
use EshopCatalog\Model\Products as BaseProductsService;
use Exception;
use Gallery\AdminModule\Components\Image\IImagesZoneFactory;
use Gallery\AdminModule\Components\Image\ImagesZone;
use Gallery\FrontModule\Model\Albums;
use Gallery\Model\Entities\Album;
use Gallery\Model\Entities\Image;
use Import\DI\ImportExtension;
use Nette\Application\IPresenter;
use Nette\Caching\Cache;
use Nette\Caching\Storage;
use Nette\Http\Request;
use Nette\Utils\ArrayHash;
use Nette\Utils\Arrays;
use Nette\Utils\Html;
use Nette\Utils\Json;
use Nette\Utils\Validators;
use Throwable;

class ProductForm extends BaseControl
{
	public ?Product                               $product = null;
	protected Products                            $productServices;
	protected FeatureProducts                     $featureProductServices;
	protected CategoryProducts                    $categoryProductServices;
	protected Categories                          $categoryServices;
	protected CategoryContainer                   $categoryContainerService;
	protected Manufacturers                       $manufacturerServices;
	protected Suppliers                           $supplierServices;
	protected VatRates                            $vatRateServices;
	protected Features                            $featureServices;
	protected SeoContainer                        $seoContainerService;
	protected IImagesZoneFactory                  $imagesZoneFactory;
	protected Request                             $httpRequest;
	protected TagsContainer                       $tagsContainerService;
	protected ProductTags                         $productTagsServices;
	protected ProductFormHelper                   $helper;
	protected AvailabilityService                 $availabilityService;
	protected CacheService                        $cacheService;
	protected DocumentContainer                   $documentContainer;
	protected VideoContainer                      $videoContainer;
	protected RelatedProductContainer             $relatedProductContainer;
	protected ProductDocuments                    $productDocuments;
	protected ProductVideos                       $productVideos;
	protected ProductSupplierContainer            $productSupplierContainer;
	protected ProductSpeditionContainer           $speditionContainer;
	protected ProductPaymentContainer             $paymentContainer;
	protected ProductPackageContainer             $packageContainer;
	protected Storage                             $storage;
	protected ProductPricesContainer              $productPricesContainer;
	protected ProductPriceLevelCountriesContainer $productPriceLevelCountriesContainer;
	protected BaseProductsService                 $baseProductsService;
	protected AdminTags                           $tags;
	protected IPriceHistoryGridFactory            $priceHistoryGridFactory;
	protected ProductChangePlanContainer          $productsChangePlanContainer;

	protected ?array $suppliersToAdd = null;
	private ?array   $cFeatures      = null;
	private ?array   $cFeatureValues = null;

	public function __construct(
		Products                            $products,
		Features                            $features,
		FeatureProducts                     $featureProducts,
		Categories                          $categories,
		CategoryProducts                    $categoryProducts,
		Manufacturers                       $manufacturers,
		Suppliers                           $suppliers,
		VatRates                            $vatRates,
		SeoContainer                        $seoContainer,
		IImagesZoneFactory                  $imagesZoneFactory,
		Request                             $request,
		CategoryContainer                   $categoryContainer,
		TagsContainer                       $tagsContainer,
		ProductTags                         $productTags,
		ProductFormHelper                   $helper,
		DocumentContainer                   $documentContainer,
		ProductDocuments                    $productDocuments,
		RelatedProductContainer             $relatedProductContainer,
		ProductSpeditionContainer           $speditionContainer,
		ProductPaymentContainer             $paymentContainer,
		AvailabilityService                 $availabilityService,
		ProductSupplierContainer            $productSupplierContainer,
		Storage                             $storage,
		CacheService                        $cacheService,
		VideoContainer                      $videoContainer,
		ProductVideos                       $productVideos,
		ProductPricesContainer              $productPricesContainer,
		ProductPriceLevelCountriesContainer $productPriceLevelCountriesContainer,
		BaseProductsService                 $baseProductsService,
		ProductPackageContainer             $packageContainer,
		AdminTags                           $tags,
		IPriceHistoryGridFactory            $priceHistoryGridFactory,
		ProductChangePlanContainer          $productsChangePlanContainer
	)
	{
		$this->productServices                     = $products;
		$this->featureServices                     = $features;
		$this->featureProductServices              = $featureProducts;
		$this->categoryContainerService            = $categoryContainer;
		$this->categoryServices                    = $categories;
		$this->categoryProductServices             = $categoryProducts;
		$this->manufacturerServices                = $manufacturers;
		$this->supplierServices                    = $suppliers;
		$this->seoContainerService                 = $seoContainer;
		$this->vatRateServices                     = $vatRates;
		$this->imagesZoneFactory                   = $imagesZoneFactory;
		$this->httpRequest                         = $request;
		$this->tagsContainerService                = $tagsContainer;
		$this->productTagsServices                 = $productTags;
		$this->helper                              = $helper;
		$this->documentContainer                   = $documentContainer;
		$this->videoContainer                      = $videoContainer;
		$this->relatedProductContainer             = $relatedProductContainer;
		$this->productDocuments                    = $productDocuments;
		$this->productVideos                       = $productVideos;
		$this->speditionContainer                  = $speditionContainer;
		$this->paymentContainer                    = $paymentContainer;
		$this->availabilityService                 = $availabilityService;
		$this->productSupplierContainer            = $productSupplierContainer;
		$this->storage                             = $storage;
		$this->cacheService                        = $cacheService;
		$this->productPricesContainer              = $productPricesContainer;
		$this->productPriceLevelCountriesContainer = $productPriceLevelCountriesContainer;
		$this->baseProductsService                 = $baseProductsService;
		$this->packageContainer                    = $packageContainer;
		$this->tags                                = $tags;
		$this->priceHistoryGridFactory             = $priceHistoryGridFactory;
		$this->productsChangePlanContainer         = $productsChangePlanContainer;
	}

	public function render(): void
	{
		$this->template->thisForm         = $this['form'];
		$this->template->product          = $this->product;
		$this->template->groupCustomers   = $this->helper->getGroupCustomers();
		$this->template->enablePriceLevel = $this->helper->config->get('enablePriceLevels', false);
		$this->template->sites            = $this->helper->sitesService->getAll();

		$features                      = $this->getFeatures();
		$this->template->features      = $features;
		$this->template->featureValues = $this->getFeatureValues();

		$defaults = [];
		if ($this->product) {
			foreach ($this->product->getFeatureProducts()->toArray() as $row) {
				$defaults[] = [
					'feature' => $row->getFeature()->getId(),
					'value'   => $row->getFeatureValue()->getId(),
				];
			}

			foreach ($this->product->dynamicFeatures as $row) {
				$defaults[] = [
					'feature' => $row->getFeature()->getId(),
					'value'   => $row->value,
				];
			}
		}
		$this->template->defaultFeatures = $defaults;

		$this->template->featuresJson = Json::encode([
			'features' => $features,
			'values'   => $this->getFeatureValues(),
		]);

		$this->template->suppliersToAdd = $this->suppliersToAdd;

		$this->eventDispatcher->dispatch(new ComponentTemplateEvent($this->template, $this), ProductForm::class . '::render');
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ========================  HANDLE
	 */

	public function handleSetParentGallery(): void
	{
		$this->productServices->setParentGallery($this->product);
		$this->presenter->redirectUrl($this->link('this'));
	}

	public function handlePriceHistory(): void
	{
		if (!Config::load('product.allowShowPriceHistory')) {
			return;
		}

		$this->template->modalTitle = $this->t('eshopCatalog.title.priceHistory');
		$this->template->modal      = 'priceHistory';
		$this->redrawControl('modal');
	}

	public function handleShowSuppliersToAdd(): void
	{
		$this->suppliersToAdd = [];

		$ids = $this->product
			? array_map(static fn(ProductSupplier $ps) => $ps->getSupplier()->getId(), $this->product->getSuppliers()->toArray())
			: [];
		$qb  = $this->em->getRepository(Supplier::class)->createQueryBuilder('s');
		$qb->select('s.id, s.name')
			->orderBy('s.name');

		if ($ids) {
			$qb->andWhere($qb->expr()->notIn('s.id', $ids));
		}

		$container = new BaseContainer;
		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$this->suppliersToAdd[$row['id']] = $row['name'];
			$c                                = $this->productSupplierContainer->getContainer();
			$c->addCustomData('name', $row['name']);
			$container->addComponent($c, 'supp_' . $row['id']);
		}
		$this['form']->addComponent($container, 'suppliersContainer');

		$this->redrawControl('suppliers-to-add-snippet');
	}

	/**
	 * @param string|int $documentId
	 */
	public function handleRemoveDocument($documentId): void
	{
		foreach ($this->product->getDocuments() as $key => $doc) {
			if ($doc->getId() === (int) $documentId) {
				$this->productDocuments->remove((int) $documentId);
			}
		}
	}

	/**
	 * @param string|int $videoId
	 */
	public function handleRemoveVideo($videoId): void
	{
		foreach ($this->product->getVideos() as $v) {
			if ($v->getId() === (int) $videoId) {
				$this->productVideos->remove((int) $videoId);
			}
		}
	}

	public function handleAddFeatureValue(string $id, string $val): void
	{
		set_time_limit(120);

		if (!$val) {
			$this->presenter->sendJson([]);
		}

		$exist = $this->em->getRepository(FeatureValueTexts::class)->createQueryBuilder('fvt')
			->select('IDENTITY(fvt.id) as id, fvt.name as name')
			->innerJoin('fvt.id', 'fv', Join::WITH, 'fv.feature = :feature')
			->where('fvt.name = :v')
			->setParameters([
				'feature' => $id,
				'v'       => $val,
			])
			->getQuery()->getResult()[0] ?? null;

		if (!$exist) {
			$featureValue = new FeatureValue();
			$featureValue->setFeature($this->featureServices->getReference((int) $id));
			$featureValue->isPublished = 1;
			$this->em->persist($featureValue);

			foreach ($this->langsService->getLangs(false) as $lang) {
				$featureValueText = new FeatureValueTexts($featureValue, $lang->getTag());
				$featureValueText->setName($val);
				$this->em->persist($featureValueText);
			}

			$this->em->flush();
			$exist = [
				'id'      => $featureValue->getId(),
				'name'    => $val,
				'created' => true,
			];

			$this->cacheService->clearFeatures([$id]);
			$this->cacheService->clearFeaturesValues();
		}

		$this->presenter->sendJson([
			'data' => $exist,
		]);
	}

	public function handleCreateVariant(int $id): void
	{
		$variant = $this->productServices->createVariant($id);

		$this->presenter->flashMessageSuccess('eshopCatalog.product.variantCreated');
		$this->presenter->redirect('edit', $variant->getId());
	}

	public function handleRemoveFromVariants(): void
	{
		if ($this->productServices->removeFromVariants($this->product)) {
			$this->presenter->flashMessageSuccess('default.saved');
		} else {
			$this->presenter->flashMessageDanger('default.error');
		}
		$this->redirect('this');
	}

	public function handleSetAsVariantFor(): void
	{
		if ($this->productServices->setAsVariantFor(
			$this->product->getId(),
			(int) $this->presenter->getHttpRequest()->getPost('productId'))
		) {
			$this->presenter->flashMessageSuccess('default.saved');
			$this->redirect('this');
		} else {
			$this->presenter->flashMessageDanger('eshopCatalog.productForm.setAsVariantForError');
			$this->presenter->redrawControl('flashes');
		}
	}

	public function handleSetAsMainVariant(int $id): void
	{
		if ($this->productServices->setAsMainVariant($id)) {
			$this->presenter->flashMessageSuccess('default.saved');
			$this->presenter->redirect('edit', $id);
		} else {
			$this->presenter->flashMessageDanger('default.errorl');
			$this->presenter->redrawControl('flashes');
		}
	}

	public function handleDuplicateProduct(int $id): void
	{
		$newProductId = $this->productServices->duplicateProduct($id);
		if ($newProductId) {
			$this->presenter->flashMessageSuccess('eshopCatalog.product.duplicated');
			$this->presenter->redirect('edit', $newProductId);
		} else {
			$this->presenter->flashMessageDanger('eshopCatalog.product.duplicateFailed');
			$this->presenter->redrawControl('flashes');
		}
	}

	/*******************************************************************************************************************
	 * ========================  COMPONENT
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$wareHouseExist = Config::wareHouseExist();

		$this->monitor(IPresenter::class, function() use ($form) {
			if ($this->product) {
				$el = Html::el()
					->addHtml(Html::el('a', [
						'href'  => $this->link('createVariant!', $this->product->getId()),
						'class' => 'btn ajax',
					])->setText($this->t('eshopCatalog.product.createVariant')))
					->addHtml(Html::el('a', [
						'href'  => $this->link('duplicateProduct!', $this->product->getId()),
						'class' => 'btn',
					])->setText($this->t('eshopCatalog.product.duplicateProduct')));
				$form->addHeadLinkEl($el);

				if ($this->product->isPublished || Config::load('product.publishedByLang')) {
					foreach ($this->product->sites as $site) {
						if ($site->isActive()) {
							$firstDomain = $site->getSite()->texts->first() ?: null;

							if ($firstDomain) {
								$link = Html::el('a', [
									'href'   => '//' . $firstDomain->domain . '/eshopcatalog/product/' . $this->product->getId(),
									'target' => '_blank',
									'class'  => 'btn btn-success',
								])
									->addHtml(Html::el('i', [
										'class' => 'fa fa-eye',
									]))
									->addHtml(Html::el('span')->setText($this->t('eshopCatalog.product.showProduct')));

								$form->addHeadLinkEl($link);
							}
						}
					}
				}
			}
		});

		$form->addBool('isPublished', 'eshopCatalog.productForm.isPublished')
			->setDefaultValue(1);

		if (Config::load('product.publishedByLang')) {
			$form->getComponent('isPublished')->setIsMultilanguage(true);
		}

		$form->addText('name', 'eshopCatalog.productForm.name')->setMaxLength(255)->setIsMultilanguage();
		if (Config::load('enableProductName2')) {
			$form->addText('name2', 'eshopCatalog.productForm.name2')->setMaxLength(255)
				->setIsMultilanguage()->setNullable();
		}

		if (Config::load('product.allowModifyAlias')) {
			$form->addText('alias', 'eshopCatalog.product.alias')
				->setIsMultilanguage();
		}

		$form->addText('ean', 'eshopCatalog.productForm.ean')->setMaxLength(13);
		$form->addText('code1', 'eshopCatalog.productForm.code1')->setMaxLength(60);
		$form->addText('code2', 'eshopCatalog.productForm.code2')->setMaxLength(60);

		if (Config::load('product.allowAssort')) {
			$form->addBool('isAssort', 'eshopCatalog.productForm.isAssort')->setDefaultValue(0);
		}

		if (Config::load('product.allowNote')) {
			$form->addBool('addNote', 'eshopCatalog.product.allowAddNote')->setDefaultValue(0);
			$form->addText('noteTitle', 'eshopCatalog.product.noteTitle')->setIsMultilanguage();
		}

		if (Config::load('product.allowVerifyAge')) {
			$form->addBool('verifyAge', 'eshopCatalog.product.verifyAge')->setDefaultValue(0);
		}

		// Slevy
		if (self::allowSaleDiscountCouponsAsProducts()) {
			$form->addBool('isDiscount', 'eshopCatalog.productForm.isDiscount')->setDefaultValue(0);
			$form->addSelect('discountType', 'eshopCatalog.productForm.discountType.caption', [
				'fix'     => $this->t('eshopCatalog.productForm.discountType.items.fix'),
				'percent' => $this->t('eshopCatalog.productForm.discountType.items.percent'),
			]);
			$form->addText('discountValue', 'eshopCatalog.productForm.discountValue')
				->setNullable()
				->addCondition($form::FILLED)
				->addRule($form::NUMERIC);
			if (Parameters::load('eshopSales.autoGenerateDiscountCoupon.allowPhysicalShipment', false)) {
				$form->addSelect('discountShipmentMethod', 'eshopCatalog.productForm.discountShipmentMethod.caption', [
					'email'    => $this->t('eshopCatalog.productForm.discountShipmentMethod.email'),
					'physical' => $this->t('eshopCatalog.productForm.discountShipmentMethod.physical'),
				]);
			}
		}

		$form->addText('position', 'eshopCatalog.productForm.position')
			->setNullable()
			->addCondition($form::FILLED)
			->addRule($form::INTEGER);

		$form->addSelect('condition', 'eshopCatalog.productForm.condition', [
			'new'         => 'eshopCatalog.productConditions.new',
			'refurbished' => 'eshopCatalog.productConditions.refurbished',
			'used'        => 'eshopCatalog.productConditions.used',
			'openBox'     => 'eshopCatalog.productConditions.openBox',
		]);

		if (Config::load('product.allowConditionDesc')) {
			$form->addTextArea('conditionDescription', 'eshopCatalog.product.conditionDescription')
				->setIsMultilanguage()
				->setHtmlAttribute('rows', 8);
		}

		if (Config::load('product.allowCountryOfOrigin')) {
			$form->addSelect('countryOfOrigin', 'eshopCatalog.product.countryOfOrigin', ['' => ''] + $this->helper->countries->getAllNameColumn());
		}

		if (Config::load('product.allowHSCustomsCode')) {
			$form->addText('hsCustomsCode', 'eshopCatalog.product.hsCustomsCode')
				->setHtmlType('number')
				->setHtmlAttribute('maxlength', 10)
				->setHtmlAttribute('oninput', 'javascript: if (this.value.length > this.maxLength) this.value = this.value.slice(0, this.maxLength);');
		}

		if (class_exists('\EshopProductionWarehouse\DI\EshopProductionWarehouseExtension')) {
			$units = [];
			foreach (Product::UNITS as $v) {
				$units[$v] = sprintf('eshopCatalog.productForm.units.items.%s', $v);
			}
			$form->addSelect('units', 'eshopCatalog.productForm.units.caption', $units)
				->setRequired();
		}
		$form->addInteger('quantity', 'eshopCatalog.productForm.quantity')
			->setDescription('eshopCatalog.productForm.quantityDescription')
			->setDefaultValue(0);
		if (!Config::load('product.allowEditQuantity')) {
			if (!$wareHouseExist) {
				$form->getComponent('quantity')->setDisabled();
			} else if ($this->product && class_exists('\EshopProductionWarehouse\DI\EshopProductionWarehouseExtension') && !Parameters::load('eshopProductionWarehouse.enableEditQuantity')) {
				$form->getComponent('quantity')->setDisabled();
			}
		}

		$form->addInteger('minimumAmount', 'eshopCatalog.productForm.minimumAmount')
			->setDefaultValue(1);
		if (Config::load('product.allowMaximumQuantityPerOrder')) {
			$form->addInteger('maximumAmount', 'eshopCatalog.productForm.maximumAmount');
		}
		if (Config::load('pseudoWarehouse')) {
			$form->addBool('unlimitedQuantity', 'eshopCatalog.productForm.unlimitedQuantity')->setDefaultValue(0);
		}
		if (class_exists('EshopGifts\DI\EshopGiftsExtension')) {
			$form->addBool('categoryGiftsAllowed', 'eshopCatalog.productForm.categoryGiftsAllowed')->setDefaultValue(1);
			$form->addBool('orderGiftsAllowed', 'eshopCatalog.productForm.orderGiftsAllowed')->setDefaultValue(1);
		}
		$form->addSelect('availability', $this->t('eshopCatalog.productForm.availability'), $this->helper->availabilityService->getOptionsForSelect())
			->setTranslator(null);
		$form->addSelect('availabilityAfterSoldOut', $this->t('eshopCatalog.productForm.availabilityAfterSoldOut'), [null => ''] + $this->helper->availabilityService->getOptionsForSelect())
			->setTranslator(null);

		if (Config::load('product.allowPreorder')) {
			$form->addText('preorderText', 'eshopCatalog.productForm.preorderText')
				->setIsMultilanguage();
		}

		$form->addText('price', 'eshopCatalog.productForm.price' . (Config::load('product.priceIsWithoutVat', false) ? 'WithoutVat' : ''))
			->setHtmlType('number')
			->setHtmlAttribute('step', .01)
			->setHtmlAttribute('min', 0)
			->setRequired();

		if (Config::load('product.allowRetailPrice') && !Config::load('enablePriceHistory')) {
			$form->addText('retailPrice', 'eshopCatalog.productForm.retailPrice' . (Config::load('product.priceIsWithoutVat', false) ? 'WithoutVat' : ''))
				->setHtmlType('number')
				->setHtmlAttribute('step', .01)
				->setHtmlAttribute('min', 0)
				->setNullable();
		}

		// DPH
		$vatRates = $this->vatRateServices->getOptionsForSelect();
		$form->addSelect('vatRate', 'eshopCatalog.productForm.vatRate', $vatRates)
			->setRequired();

		if (Config::load('enablePurchasePrice')) {
			$form->addText('purchasePrice', 'eshopCatalog.productForm.purchasePrice')
				->setHtmlType('number')
				->setHtmlAttribute('step', .01)
				->setHtmlAttribute('min', 0)
				->setNullable();
		}

		$shortDescMaxLength = Config::load('product.shortDescriptionMaxLength', null);
		$form->addEditor('shortDesc', 'eshopCatalog.productForm.shortDesc')
			->setToolbar('Text')
			->setDisableAutoP()
			->setHeight(300)
			->setIsMultilanguage();

		if ($shortDescMaxLength) {
			$form->getComponent('shortDesc')
				->setHeight((int) min(300, round($shortDescMaxLength / 2.5)))
				->setMaxLength($shortDescMaxLength);
		}

		$form->addEditor('desc', 'eshopCatalog.productForm.desc')->setHeight(300)->setIsMultilanguage();

		$form->addText('recyclingFee', 'eshopCatalog.productForm.recyclingFee')
			->setHtmlType('number')
			->setHtmlAttribute('step', .01)
			->setHtmlAttribute('min', 0);

		$form->addBool('disableRegisterSale', 'eshopCatalog.product.disableRegisterSale')
			->setDefaultValue(0);

		if (class_exists(ImportExtension::class)) {
			$form->addBool('stopImportPrice', 'eshopCatalog.productForm.stopImportPrice')->setDefaultValue(1);
		}

		if (Config::load('product.allowStopImportQuantity')) {
			$form->addBool('stopImportQuantity', 'eshopCatalog.productForm.stopImportQuantity')->setDefaultValue(0);
		}

		if (Config::load('product.loadQuantityOnStock') && class_exists('EshopStock\DI\EshopStockExtension')) {
			$form->addText('quantityOnStock', 'eshopCatalog.productForm.quantityOnStock')
				->setDefaultValue(0)
				->setHtmlAttribute('readonly')
				->setOmitted();
		}

		$form->addBool('discountDisabled', 'eshopCatalog.productForm.discountDisabled')->setDefaultValue(1);

		if (Config::load('product.allowDisablePickUpSpedition')) {
			$form->addBool('disablePickUpSpedition', 'eshopCatalog.product.disablePickUpSpedition')->setDefaultValue(0);
		}
		if (Config::load('product.allowMarkAsOversize')) {
			$form->addBool('isOversize', 'eshopCatalog.product.isOversize')->setDefaultValue(0);
		}
		if (Config::load('product.allowDisableStackInCart')) {
			$form->addBool('disableStackInCart', 'eshopCatalog.product.disableStackInCart')->setDefaultValue(0);
		}

		$form->addSelect('manufacturer', $this->t('eshopCatalog.productForm.manufacturer'), ['' => ''] + $this->manufacturerServices->getOptionsForSelect())
			->setTranslator(null);

		if (Config::load('productReplenishmentPlanner.enable')) {
			$form->addBool('skipForReplenishmentPlanning', 'eshopCatalog.productForm.skipForReplenishmentPlanning')
				->setDefaultValue(0);
		}

		//Suppliers - Dodavatele
		// Aktuálně pouze zobrazení z importu

		//PriceLevel
		if ($this->helper->config->get('enablePriceLevels', false)) {
			foreach ($this->helper->getGroupCustomers() as $groupCustomers) {
				$nameSuffix = '';
				if (Config::load('product.priceIsWithoutVat', false)) {
					$nameSuffix = ' ' . $this->t('eshopCatalog.withoutVat');
				}
				$form->addText('pricelevel' . $groupCustomers->getId(), $groupCustomers->name . $nameSuffix)
					->setHtmlType('number')
					->setHtmlAttribute('step', .01)
					->setHtmlAttribute('min', 0)
					->setNullable();
			}
		}

		// Ceny podle zeme
		if (Config::load('enableCountryPrices')) {
			$form->addComponent($this->productPricesContainer->getContainer(), 'prices');
			$form->addComponent($this->productPriceLevelCountriesContainer->getContainer(), 'priceLevelCountries');
		}

		// Kategorie a Eshopy
		$form->addComponent($this->categoryContainerService->getContainers(), 'categories');
		if (Config::load('product.setDefaultTrueVisibleInEshop') && !$this->product) {
			foreach ($form->getComponent('categories')->getComponents() as $catSiteComponent) {
				$catSiteComponent->setDefaults([
					'isActive' => 1,
				]);
			}
		}

		//Features - Vlastnosti
		$form->addContainer('features');

		//Variants
		$form->addComponent($this->seoContainerService->getContainer(true), 'seo');

		$form->addComponent($this->tagsContainerService->getContainer(), 'tags');

		// Doprava
		if (Config::load('product.allowModifySpedition', false)) {
			$form->addComponent($this->speditionContainer->getContainer(), 'speditions');
		}

		// Platba
		if (Config::load('product.allowModifyPayment', false)) {
			$form->addComponent($this->paymentContainer->getContainer(), 'payments');
		}

		// Package
		if (Config::load('product.allowPackage', false)) {
			$form->addComponent($this->packageContainer->getContainer($form, 'package'), 'package');
		}

		// Plan zmen
		if ($this->product && Config::load('product.allowPlan', false)) {
			$form->addComponent($this->productsChangePlanContainer->getContainer([$this->product->getId()]), 'changePlan');
		}

		// Rozměry
		$form->addInteger('width', $this->t('eshopCatalog.productForm.width'))
			->setHtmlType('number')
			->setHtmlAttribute('min', 0)
			->setNullable();
		$form->addInteger('height', $this->t('eshopCatalog.productForm.height'))
			->setHtmlType('number')
			->setHtmlAttribute('min', 0)
			->setNullable();
		$form->addInteger('depth', $this->t('eshopCatalog.productForm.depth'))
			->setHtmlType('number')
			->setHtmlAttribute('min', 0)
			->setNullable();
		$form->addInteger('weight', $this->t('eshopCatalog.productForm.weight'))
			->setHtmlType('number')
			->setHtmlAttribute('min', 0)
			->setNullable();

		$form->addHidden('preparedAlbumId');

		$event                 = new CreateFormEvent($form, $this->getPresenterIfExists() ? $this->template : null);
		$event->data['entity'] = $this->product;
		$event->control        = $this;
		$this->eventDispatcher->dispatch($event, ProductForm::class . '::createForm');

		$form->addSaveCancelControl();

		// Nastaveni pro varianty
		if ($this->product && $this->product->isVariant()) {
			if ($this->product->getIsVariantParent()) {
				$form->addText('variantName', 'eshopCatalog.product.variantName')
					->setMaxLength(255)
					->setIsMultilanguage()
					->setNullable();
			} else {
				$variantUse    = $form->addContainer('variantUse');
				$parentProduct = $this->product->getVariantParent();

				foreach (get_object_vars($this->product->isVariant()) as $k => $v) {
					if (in_array($k, ['defaultImage', 'createdDefault'])) {
						continue;
					}
					$variantUse->addBool($k, 'eshopCatalog.product.useForVariant')->setDefaultValue($v ? 1 : 0);
				}

				$form->getComponent('manufacturer')->setDisabled()
					->setDefaultValue($parentProduct->getManufacturer() ? $parentProduct->getManufacturer()->getId() : null);
				$form->getComponent('vatRate')->setDisabled()
					->setDefaultValue($parentProduct->getVateRate() ? $parentProduct->getVateRate()->getId() : null);
			}
		}
		// Dokumenty
		$form->addComponent($this->documentContainer->getContainer(), 'documents');

		// Videa
		$form->addComponent($this->videoContainer->getContainer(), 'videos');

		// Souvisejici produkty
		$form->addComponent($this->relatedProductContainer->getContainer(), 'relatedProducts');

		$this->monitor(IPresenter::class, function(AbstractPresenter $presenter) use ($form) {
			$form->getComponent('relatedProducts')['product']->getControlPrototype()->setAttribute('data-link', $presenter->link(':EshopCatalog:Cron:Products:loadAll', [
				'excluded' => $this->product ? [$this->product->getId()] : [],
			]));

			$this->template->renderUseVariant = static fn() => '';
			if ($this->product) {
				// vykresleni zapnuti hodnot pro varianty
				$this->template->isVariant        = $this->product->isVariant() && !$this->product->getIsVariantParent();
				$this->template->renderUseVariant = static function(string $c) use ($form) {
					if ($form->getComponent('variantUse', false) && in_array($c, ['vatRate'])) {
						return Html::el('div', ['style' => 'height: 2.8rem;']);
					}

					/** @var BootstrapRenderer $renderer */
					$renderer = $form->getRenderer();

					$c = 'use' . ucfirst($c);
					if (isset($form->getComponent('variantUse', false)[$c])) {
						return Html::el('div', [
							'class'             => 'variant-use-bool ' . ($form->getComponent('variantUse')[$c]->getValue() ? 'active' : ''),
							'data-variant-bool' => $c,
						])->addHtml($renderer->renderPair($form->getComponent('variantUse')[$c]));
					}

					return '';
				};
			}
		});

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		$nameFilled = false;
		foreach ($values->name as $v) {
			if ($v) {
				$nameFilled = true;
				break;
			}
		}

		if (!$nameFilled) {
			$form->addError('eshopCatalog.productForm.nameMissing');
		}

		// Duplikace aliasu
		if (Config::load('allowValidateDuplicateTextsProduct')) {
			$skipCheck = Config::load('disableValidateDuplicateTextForLanguage') ?? [];
			foreach (['name', 'alias'] as $col) {
				if ($col === 'alias' && !Config::load('product.allowModifyAlias')) {
					continue;
				}

				$data = $values->{$col};

				if (!$data) {
					continue;
				}

				$qb = $this->productServices->em->getRepository(ProductTexts::class)->createQueryBuilder('pt')
					->select('IDENTITY(pt.id) as id, pt.lang, pt.' . $col);

				if (Config::load('product.publishedByLang')) {
					$qb->andWhere('pt.isPublished = 1');
				} else {
					$qb->innerJoin('pt.id', 'p', Join::WITH, 'p.isPublished = 1');
				}

				$orWhere = [];
				foreach ($this->langsService->getLangs(false) as $langKey => $lang) {
					if (!isset($data[$langKey]) || in_array($langKey, $skipCheck, true)) {
						continue;
					}

					$tmp = ["pt.lang = '{$langKey}'"];

					if ($col === 'alias') {
						$tmp[] = "pt.{$col} = '" . Strings::webalize($data[$langKey]) . "'";
					} else {
						$tmp[] = "pt.{$col} = " . $qb->expr()->literal($data[$langKey]);
					}

					$orWhere[] = '(' . implode(' AND ', $tmp) . ')';
				}

				if (!empty($orWhere)) {
					$qb->andWhere(implode(' OR ', $orWhere));
				}

				if ($this->product && $this->product->variants) {
					$variantProductIds = [];
					$defaultVariant    = null;
					foreach ($this->product->variants as $v) {
						if (!$v->useName) {
							$variantProductIds[$v->product->getId()] = $v->product->getId();
						}

						if ($v->isDefault) {
							$defaultVariant = $v;
						}
					}

					if ($values->variantUse && $values->variantUse->useName) {
						unset($variantProductIds[$defaultVariant->product->getId()]);
					} else {
						$variantProductIds[$defaultVariant->product->getId()] = $defaultVariant->product->getId();
					}

					if (!empty($variantProductIds)) {
						$qb->andWhere('pt.id NOT IN (' . implode(',', array_values($variantProductIds)) . ')');
					}
				}

				foreach ($qb->getQuery()->getArrayResult() as $row) {
					if (
						!$form->getComponent($col)->getErrors()
						&& $row[$col] && (
							!$this->product
							|| ($this->product->getId() !== $row['id'])
						)
					) {
						$form->getComponent($col)->addError($this->t('eshopCatalog.productForm.duplicateField', [
							'lang'  => Strings::upper($row['lang']),
							'field' => $this->translator->translate('eshopCatalog.productForm.' . $col),
						]));
					}
				}
			}
		}

		$event                   = new FormValidateEvent($form, $values,
			$this->getPresenterIfExists() ? $this->template : null, $this->getPresenterIfExists() ? $this->presenter : null);
		$event->custom['entity'] = $this->product;
		if ($this->getPresenterIfExists()) {
			$event->presenter = $this->presenter;
		}
		$this->eventDispatcher->dispatch($event, ProductForm::class . '::formValidate');

		if ($form->hasErrors()) {
			$this->redrawControl('formErrors');
		}
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$beforeFormSuccessEvent          = new FormSuccessEvent($form, $values, $this->template, $this->presenter);
		$beforeFormSuccessEvent->control = $this;
		$this->eventDispatcher->dispatch($beforeFormSuccessEvent, self::class . '::beforeFormSuccess');

		$this->em->beginTransaction();
		try {
			ProductListener::$updateVariant = false;
			$langValues                     = $form->convertMultilangValuesToArray();
			if ($this->product) {
				$product = $this->product;
			} else {
				$product = new Product;

				$this->em->persist($product);
				$this->em->flush();
			}

			// Ulozeni hodnot pouziti pro variantu
			if ($product->isVariant() && !$product->getIsVariantParent()) {
				foreach ($values->variantUse as $k => $v) {
					if (Arrays::contains(['product', 'variantName', 'texts'], $k)) {
						continue;
					}

					$product->isVariant()->$k = $v;
				}

				$this->em->persist($product->isVariant());
				$this->em->flush($product->isVariant());
			}

			/** @var ProductVariant|null $va */
			$va = $product->isVariant();

			if (!$form->getComponent('quantity')->isDisabled()) {
				$product->quantity = $values->quantity;
				$product->inStock  = $values->quantity > 0 ? 1 : 0;
			}

			$product->minimumAmount = $values->minimumAmount;

			if (Config::load('enablePriceHistory') && $product->price !== $values->price) {
				$priceHistory = new ProductPriceHistory(
					$product,
					(float) $product->price,
					(float) $values->price,
					'productForm',
				);

				$this->em->persist($priceHistory);
			}

			$product->price = $values->price;

			if (Config::load('product.allowRetailPrice') && !Config::load('enablePriceHistory')) {
				$product->retailPrice = $values->retailPrice ?: null;
			}

			$product->purchasePrice = $values->purchasePrice ?: null;
			$product->recyclingFee  = $values->recyclingFee ?: null;
			$product->isPublished   = Config::load('product.publishedByLang')
				? (int) array_values($values->isPublished)[0]
				: $values->isPublished;

			if (self::allowSaleDiscountCouponsAsProducts()) {
				$product->isDiscount    = (int) $values->isDiscount;
				$product->discountType  = $values->discountType;
				$product->discountValue = $values->discountValue;
				if (property_exists($values, 'discountShipmentMethod') && $product->isDiscount) {
					$product->setMoreDataValue('discountShipmentMethod', $values->discountShipmentMethod);
				} else {
					$product->removeMoreData('discountShipmentMethod');
				}
			} else {
				$product->isDiscount    = 0;
				$product->discountType  = null;
				$product->discountValue = null;
			}

			$product->position         = Validators::isNone($values->position) ? null : ((int) $values->position);
			$product->ean              = $values->ean;
			$product->code1            = $values->code1;
			$product->code2            = $values->code2;
			$product->discountDisabled = $values->discountDisabled == 1 ? 0 : 1;
			$product->isAssort         = Config::load('product.allowAssort')
				? ($values->isAssort ? 1 : 0)
				: 0;
			$product->condition        = $values->condition;
			$product->setAvailability($this->helper->availabilityService->getReference($values->availability));
			$product->availabilityAfterSoldOut = $values->availabilityAfterSoldOut ? $this->helper->availabilityService->getReference($values->availabilityAfterSoldOut) : null;

			if ($values->offsetExists('skipForReplenishmentPlanning')) {
				$product->skipForReplenishmentPlanning = (int) $values->skipForReplenishmentPlanning;
			}

			if ($form->getComponent('maximumAmount', false)) {
				$product->maximumAmount = is_numeric($values->maximumAmount) && $values->maximumAmount > 0 ? (int) $values->maximumAmount : null;
			}

			$product->width  = $values->width;
			$product->height = $values->height;
			$product->depth  = $values->depth;
			$product->weight = $values->weight;

			$product->disableRegisterSale = (int) $values->disableRegisterSale;

			if (property_exists($values, 'disablePickUpSpedition')) {
				$product->disablePickUpSpedition = $values->disablePickUpSpedition ? 1 : 0;
			}

			if (property_exists($values, 'isOversize')) {
				$product->isOversize = $values->isOversize ? 1 : 0;
			}

			if (property_exists($values, 'disableStackInCart')) {
				$product->disableStackInCart = $values->disableStackInCart ? 1 : 0;
			}

			if (isset($values['units'])) {
				$product->units = $values['units'];
			}

			if (Config::load('product.allowCountryOfOrigin')) {
				$product->countryOfOrigin = $values->countryOfOrigin
					? $this->helper->countries->getReference($values->countryOfOrigin)
					: null;
			}

			if (Config::load('product.allowHSCustomsCode')) {
				$product->hsCustomsCode = $values->hsCustomsCode ? (int) $values->hsCustomsCode : null;
			}

			if (Config::load('pseudoWarehouse')) {
				$product->unlimitedQuantity = $values->unlimitedQuantity;
				$avs                        = $this->helper->availabilityService->getAllByIdent();

				$tmp = [$avs[Availability::IN_STOCK]->getId()];
				if (isset($avs[Availability::PREORDER])) {
					$tmp[] = $avs[Availability::PREORDER]->getId();
				}

				if ($product->unlimitedQuantity && !in_array($values['availability'], $tmp)) {
					$product->setAvailability($this->helper->availabilityService->getAllByIdent()['inStock']);
				}
			}

			if (class_exists('EshopGifts\DI\EshopGiftsExtension')) {
				$product->setCategoryGiftsAllowed($values->categoryGiftsAllowed);
				$product->orderGiftsAllowed = (int) $values->orderGiftsAllowed;
			}

			$this->em->persist($product);
			$this->em->flush();

			if ($values->preparedAlbumId) {
				$product->setGallery($this->em->getRepository(Album::class)->getReference($values->preparedAlbumId));
			}

			//vyrobce
			if (!$product->isVariant() || $product->getIsVariantParent()) {
				$product->setManufacturer($values->manufacturer ? $this->manufacturerServices->get($values->manufacturer) : null);
			}

			//vatRate
			if ($values->vatRate) {
				$vatRate = $this->vatRateServices->get($values->vatRate);
				$product->setVateRate($vatRate);
			}

			// Texty
			/** @var ProductTexts[] $productTexts */
			$productTexts = $product->getTexts()->toArray();
			$seoData      = $this->seoContainerService->getFormData($values->seo, true);
			$activeByLang = Config::load('product.publishedByLang');

			// Extra fields
			$ef = $product->getExtraFieldsByKey();

			if (Config::load('product.allowVerifyAge')) {
				$product->verifyAge = (int) $values->verifyAge;
			}

			if (Config::load('product.allowNote')) {
				$addNote = $ef['addNote'] ?? null;

				if (!$addNote) {
					$addNote = new ExtraField('addNote', (string) $values->addNote);
					$product->fillExtraField($addNote);
				}

				$addNote->value = (string) $values->addNote;
				$this->em->persist($addNote);

				foreach ($langValues as $l => $v) {
					/** @phpstan-ignore-next-line */
					$noteTitle = $ef['noteTitle'][$l] ?? null;

					if (!$noteTitle) {
						$noteTitle       = new ExtraField('noteTitle', $v['noteTitle']);
						$noteTitle->lang = $l;
						$product->fillExtraField($noteTitle);
					}

					$noteTitle->value = $v['noteTitle'];
					$this->em->persist($noteTitle);
				}
			}

			foreach ($langValues as $l => $v) {
				if (!isset($productTexts[$l])) {
					$productTexts[$l] = new ProductTexts($product, $l);
				}

				$alias = $v['name'];
				if (Config::load('product.allowModifyAlias') && $v['alias']) {
					$alias = $v['alias'];
				}

				$productTexts[$l]->setName($v['name'])
					->setAlias(Strings::webalize((string) $alias))
					->setDescription($v['desc'])
					->setShortDescription($v['shortDesc'])
					->setSeo($seoData[$l] ?? []);

				if (Config::load('enableProductName2')) {
					$productTexts[$l]->setName2($v['name2']);
				}

				if (Config::load('product.allowPreorder')) {
					$productTexts[$l]->preorderText = $v['preorderText'] ?: null;
				}

				if (Config::load('product.allowConditionDesc')) {
					$productTexts[$l]->conditionDescription = $v['conditionDescription'] ?: null;
				}

				if (!$activeByLang) {
					$productTexts[$l]->isPublished = $product->isPublished;
				} else {
					$productTexts[$l]->isPublished = $v['isPublished'] && (int) $v['isPublished'] === 1
						? 1
						: 0;
				}

				$this->em->persist($productTexts[$l]);
			}

			$product->setProductTexts($productTexts);

			$this->em->persist($product);
			$this->em->flush();

			//Vlastnosti
			$currentFeatures = [];
			$formFeatures    = [];
			foreach ($product->getFeatureProducts() as $fp) {
				if (isset($currentFeatures[(int) $fp->getFeatureValue()->getId()])) {
					$this->product->getFeatureProducts()->removeElement($fp);
					$this->em->remove($fp);
					continue;
				}
				$currentFeatures[(int) $fp->getFeatureValue()->getId()] = $fp;
			}

			foreach ($form->getHttpData()['features'] as $v) {
				$formFeatures[$v] = $v;
			}

			$newFeatures = array_diff_key($formFeatures, $currentFeatures);
			$oldFeatures = array_diff_key($currentFeatures, $formFeatures);

			foreach ($newFeatures as $value) {
				/** @var FeatureValue $fv */
				$fv = $this->em->getReference(FeatureValue::class, $value);

				$fp = new FeatureProduct(
					$product,
					$fv,
				);

				unset($formFeatures[$value]);
				$this->em->persist($fp);
			}

			foreach ($oldFeatures as $fp) {
				$this->product->getFeatureProducts()->removeElement($fp);
				$this->em->remove($fp);
			}

			/** @var DynamicFeatureProduct[][] $currentDynamicFeatures */
			$currentDynamicFeatures = [];
			$formDynamicFeatures    = $form->getHttpData()['dynamicFeatures'] ?? [];
			$toRemove               = [];
			foreach ($product->dynamicFeatures as $df) {
				if (isset($currentDynamicFeatures[$df->getFeature()->getId()][(string) $df->value])) {
					$toRemove[] = $df->getId();
					continue;
				}

				$currentDynamicFeatures[$df->getFeature()->getId()][(string) $df->value] = $df;
			}

			foreach ($formDynamicFeatures as $k => $v) {
				foreach ($v as $v2) {
					$dfp        = $currentDynamicFeatures[$k][(string) $v2] ?? new DynamicFeatureProduct($product, $this->featureServices->getReference($k), (string) $v);
					$dfp->value = (string) $v2;
					$this->em->persist($dfp);
				}
			}

			foreach ($currentDynamicFeatures as $featureId => $vals) {
				foreach ($vals as $k => $v) {
					if (!in_array($k, $formDynamicFeatures[$featureId] ?? [])) {
						$toRemove[] = $v->getId();
					}
				}
			}

			if (!empty($toRemove)) {
				$this->em->createQueryBuilder()->delete(DynamicFeatureProduct::class, 'dfp')
					->where('dfp.id IN (' . implode(',', $toRemove) . ')')
					->getQuery()->execute();
			}

			//PriceLevels
			if ($this->helper->config->get('enablePriceLevels', false) && (!$va || $va->usePriceLevels || $va->isDefault)) {
				foreach ($this->helper->getGroupCustomers() as $gc) {
					$remove = true;
					if (isset($values['pricelevel' . $gc->getId()])) {
						$v = $values['pricelevel' . $gc->getId()];
						if ($v != '' && $v !== null) {
							$remove     = false;
							$ppl        = $product->getPriceLevels()->get($gc->getId()) ?: new ProductPriceLevel($product, $gc);
							$ppl->price = $v;
							$this->em->persist($ppl);
							if (!$product->getPriceLevels()->containsKey($gc->getId())) {
								$product->getPriceLevels()->add($ppl);
							}
						}
					}

					if ($remove) {
						$ppl = $product->getPriceLevels()->get($gc->getId());
						if ($ppl) {
							$product->getPriceLevels()->remove($gc->getId());
							$this->em->getConnection()->delete('eshop_catalog__product_price_level', [
								'product_id' => $product->getId(),
								'group_id'   => $gc->getId(),
							]);
						}
					}
				}
			}

			// Prices
			if (Config::load('enableCountryPrices')) {
				if ($va && !$va->isDefault) {
					foreach ($values->prices as &$row) {
						if (!$va->usePrice) {
							unset($row['price']);
						}

						if (Config::load('product.allowRetailPrice') && !Config::load('enablePriceHistory') && !$va->useRetailPrice) {
							unset($row['retailPrice']);
						}
					}
				}

				$this->productPricesContainer->saveData($product, (array) $values->prices);

				if (!$va || $va->isDefault || $va->usePriceLevels) {
					$this->productPriceLevelCountriesContainer->saveData($product, (array) $values->priceLevelCountries);
				}
			}

			// Kategorie
			if (!$va || $va->isDefault) {
				$this->categoryContainerService->saveDataMultipleSite([$product->getId()], (array) $values->categories);
			}

			// Dodavatele
			if ($values->suppliers) {
				$suppliers = $product->getSuppliers();

				foreach ($values->suppliers as $k => $v) {
					$suppId = substr($k, 1);

					if (isset($suppliers[$suppId])) {
						$suppliers[$suppId]->isActive = (int) $v;
						$this->em->persist($suppliers[$suppId]);
					}
				}
			}

			//Tags
			$tagsForRemove = [];
			foreach ($product->getProductTags()->toArray() as $tag) {
				$tagsForRemove[$tag->getTag()->type] = $tag;
			}

			foreach ($values->tags as $tagType => $vals) {
				if (!$vals->active) {
					continue;
				}

				if (!isset($tagsForRemove[$tagType])) {
					$productTag = new ProductTag($product, $this->tags->getReference($vals['id']));
					$product->addProductTag($productTag);
				} else {
					$productTag = $tagsForRemove[$tagType];
				}

				unset($tagsForRemove[$tagType]);
				$productTag->validFrom = $vals['dateFrom'];
				$productTag->validTo   = $vals['dateTo'];

				$this->em->persist($productTag);
			}

			foreach ($tagsForRemove as $tagType => $tag) {
				$product->removeProductTag($tag);
				$this->em->remove($tag);
			}

			if (property_exists($values, 'stopImportPrice')) {
				$values->stopImportPrice = $values->stopImportPrice == 1 ? 0 : 1;

				if ($values->stopImportPrice) {
					$product->setMoreDataValue('stopImportPrice', $values->stopImportPrice);
				} else {
					$product->removeMoreData('stopImportPrice');
				}
			}

			if (property_exists($values, 'stopImportQuantity')) {
				if ($values->stopImportQuantity) {
					$product->setMoreDataValue('stopImportQuantity', $values->stopImportQuantity);
				} else {
					$product->removeMoreData('stopImportQuantity');
				}
			}

			// dokumenty
			$this->documentContainer->saveData($values['documents'], $product);

			// videa
			$this->videoContainer->saveData($values['videos'], $product);

			// souvisejici produkty
			if (Config::load('allowRelatedProducts')) {
				$this->relatedProductContainer->saveData($form->getHttpData(), $product);
			}

			// Doprava
			if (Config::load('product.allowModifySpedition')) {
				$this->speditionContainer->saveData($values->speditions, $product);
			}

			// Platba
			if (Config::load('product.allowModifyPayment', false)) {
				$this->paymentContainer->saveData($values->payments, [$product]);
			}

			// Package
			if (Config::load('product.allowPackage')) {
				$this->packageContainer->saveData($form->getHttpData()['package'] ?: [], $product);
			}

			// Pridani dodavatelů
			$suppliersContainer = $form->getHttpData()['suppliersContainer'];
			$suppliersToAdd     = isset($suppliersContainer) && is_array($suppliersContainer) && count($suppliersContainer) > 0 ? $suppliersContainer : [];
			foreach ($suppliersToAdd as $k => $row) {
				if (isset($row['selected']) || $row['selected'] === 'on') {
					$supplier = $this->supplierServices->get((int) explode('_', $k)[1]);

					/** @var Availability $availability */
					$availability = $this->availabilityService->getEr()->find((int) $row['availability']);

					if (!$supplier) {
						continue;
					}

					$this->productSupplierContainer->saveData($row, $supplier, $product, $availability);
				}
			}

			$this->em->persist($product);

			$event                   = new FormSuccessEvent($form, $values, $this->template, $this->presenter);
			$event->custom['entity'] = $product;
			$event->presenter        = $this->presenter;
			$event->control          = $this;
			$this->eventDispatcher->dispatch($event, ProductForm::class . '::formSuccess');

			$this->availabilityService->updateAvailabilityByQuantity($product);
			$this->em->flush();
			$this->em->commit();

			$this->baseProductsService->prepareUpdateVariant($product);

			// Ulozeni dat pro hlavni variant ua nastaveni aktegorii variantam
			if ($product->isVariant()) {
				if ($product->getIsVariantParent()) {
					$vaTexts = $product->isVariant()->texts->toArray();
					foreach ($langValues as $lang => $v) {
						$vaText       = isset($vaTexts[$lang]) ? $vaTexts[$lang] : new ProductVariantText($product->isVariant(), $lang);
						$vaText->name = $v['variantName'];

						$this->em->persist($vaText);
					}

					foreach (array_diff_key($vaTexts, $langValues) as $v) {
						$this->em->remove($v);
					}

					$parent = $product;
					$this->em->persist($product->isVariant());
				} else {
					$parent = $product->getVariantParent();
				}

				if ($parent->getId()) {
					$this->productServices->updateVariantCategories($parent, $parent->isVariant()->getVariantId());
				}

				$this->em->flush();
			}

			$event                   = new FormSuccessEvent($form, $values, $this->template, $this->presenter);
			$event->custom['entity'] = $product;
			$event->presenter        = $this->presenter;
			$event->control          = $this;
			$this->eventDispatcher->dispatch($event, ProductForm::class . '::formSuccessAfterSave');

			$form->addCustomData('productId', $product->getId());
			$this->presenter->flashMessageSuccess('eshopCatalog.productForm.productSaved');
			$this->presenter->redrawControl('flashes');

			if (Config::load('useOldUrls')) {
				$oldIdsCache = new Cache($this->storage, 'oldZuriel');
				foreach ($this->translator->getAvailableLocales() as $l) {
					$oldIdsCache->remove('prodIds_' . $l);
				}
			}

			$this->productServices->clearProductCache($product->getId());

			if ($product->getGallery()) {
				$albumCache = new Cache($this->storage, Albums::CACHE_NAMESPACE);
				$albumCache->clean([Cache::TAGS => ['album/' . $product->getGallery()->getId()]]);
			}

			$this->cacheService->clearTagsCache();

			if (Config::load('product.allowPackage') && $product->package) {
				$this->packageContainer->clearCache([$product->package->getId()]);
			}
		} catch (Exception $e) {
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->rollback();
			}

			$form->addError($e->getMessage());
			$this->presenter->redrawControl('flashes');
			$this->redrawControl('formErrors');

			return false;
		}

		return true;
	}

	protected function createComponentImagesZone(): ImagesZone
	{
		$control = $this->imagesZoneFactory->create();

		if ($this->product && $this->product->isVariant()) {
			$control->onBeforeUpdate[] = function($id, $v) {
				$variant = $this->product->isVariant();

				$variant->defaultImage = (int) $id;
				$this->em->persist($variant);
			};

			$control->onPrepareDefaultImage[] = function(Image $image, &$arr) {
				$arr['variantImage'] = $image->getId() === $this->product->isVariant()->defaultImage ? 1 : 0;

				return $arr;
			};

			$control->addCustomFieldLatteBefore(__DIR__ . '/ProductFormImageZoneCustomFields.latte');
		}

		if ($this->product && $this->product->getGallery()) {
			$control->setAlbum($this->product->getGallery()->getId());
		}

		$control->onEmpty[] = function(ImagesZone $control) {
			$dataRaw = $this->httpRequest->getPost('formData');

			if (!$dataRaw) {
				return;
			}

			if (!$this->product || !$this->product->getGallery()) {
				$album = new Album(UPLOADS_PATH . '/products');
				$this->em->persist($album)->flush();
				$control->setAlbum($album->getId());
			}

			if ($this->product && isset($album)) {
				$this->product->setGallery($album);
				$this->em->persist($this->product)->flush();
			}
		};

		return $control;
	}

	/*******************************************************************************************************************
	 * ========================  GET / SET
	 */

	/**
	 * @param int|string $id
	 *
	 * @throws Throwable
	 */
	public function setProduct($id): void
	{
		$this->product = $this->productServices->get($id);

		if ($this->product) {
			$prod       = $this->product;
			$parentProd = null;
			$variant    = null;

			if ($prod->isVariant()) {
				$variants = $this->em->getRepository(ProductVariant::class)->createQueryBuilder('pv')
					->where('pv.variantId = :variantId')->setParameter('variantId', $prod->isVariant()->getVariantId())
					->getQuery()->getResult();

				$prod->variants = new ArrayCollection($variants);
				$this->product  = $prod;

				// kontrola hlavniho produktu
				if (!$prod->getVariantParent()) {
					$prod->isVariant()->isDefault = 1;
					$this->em->persist($prod->isVariant());
					$this->em->flush($prod->isVariant());
					$this->presenter->redirect('this');
				}
			}

			if ($prod->isVariant() && !$prod->getIsVariantParent()) {
				$parentProd = $prod->getVariantParent();
				$variant    = $prod->isVariant();
			}

			$form = $this['form'];
			$d    = [
				'isPublished'                  => $prod->isPublished,
				'ean'                          => $prod->ean,
				'quantity'                     => $prod->quantity,
				'minimumAmount'                => $prod->minimumAmount,
				'recyclingFee'                 => $prod->recyclingFee,
				'price'                        => $prod->price ?: $parentProd->price,
				'retailPrice'                  => Config::load('product.allowRetailPrice') && !Config::load('enablePriceHistory') && $variant && !$variant->useRetailPrice
					? $parentProd->retailPrice
					: $prod->retailPrice,
				'purchasePrice'                => $variant && !$variant->usePurchasePrice ? $parentProd->purchasePrice : $prod->purchasePrice,
				'code1'                        => $prod->code1,
				'code2'                        => $prod->code2,
				'discountDisabled'             => ($variant && !$variant->useDiscountDisabled ? $parentProd->discountDisabled : $prod->discountDisabled) == 1 ? 0 : 1,
				'condition'                    => $prod->condition,
				'width'                        => $prod->width,
				'height'                       => $prod->height,
				'depth'                        => $prod->depth,
				'weight'                       => $prod->weight,
				'position'                     => $prod->position,
				'skipForReplenishmentPlanning' => $prod->skipForReplenishmentPlanning,
				'disableRegisterSale'          => $prod->disableRegisterSale,
			];

			if ($form->getComponent('maximumAmount', false)) {
				$d['maximumAmount'] = $prod->maximumAmount;
			}

			if ($form->getComponent('units', false)) {
				$d['units'] = $prod->units;
			}

			if (isset($this['form']['disablePickUpSpedition'])) {
				$d['disablePickUpSpedition'] = $prod->disablePickUpSpedition;
			}

			if (isset($this['form']['isOversize'])) {
				$d['isOversize'] = $prod->isOversize;
			}
			if (isset($this['form']['disableStackInCart'])) {
				$d['disableStackInCart'] = $prod->disableStackInCart;
			}
			if (isset($this['form']['isDiscount'])) {
				$d['isDiscount']    = $prod->isDiscount;
				$d['discountType']  = $prod->discountType;
				$d['discountValue'] = $prod->discountValue;

				if ($form->getComponent('discountShipmentMethod', false)) {
					$discountShipmentMethod = $prod->getMoreDataValue('discountShipmentMethod');
					$form->getComponent('discountShipmentMethod')->setValue($discountShipmentMethod);
				}
			}

			if (Config::load('product.allowCountryOfOrigin')) {
				$d['countryOfOrigin'] = $prod->countryOfOrigin ? $prod->countryOfOrigin->getId() : null;
			}

			if (Config::load('product.allowHSCustomsCode')) {
				$d['hsCustomsCode'] = $prod->hsCustomsCode;
			}

			if ($prod->getAvailability() && array_key_exists($prod->getAvailability()->getId(), $form->getComponent('availability')->getItems())) {
				$form->getComponent('availability')->setDefaultValue($prod->getAvailability()->getId());
			}

			if ($prod->availabilityAfterSoldOut && array_key_exists($prod->availabilityAfterSoldOut->getId(), $form->getComponent('availabilityAfterSoldOut')
					->getItems())) {
				$form->getComponent('availabilityAfterSoldOut')
					->setDefaultValue($prod->availabilityAfterSoldOut->getId());
			}

			if (Config::load('product.allowAssort')) {
				$d['isAssort'] = $prod->isAssort ? 1 : 0;
			}

			if (Config::load('pseudoWarehouse')) {
				$form->getComponent('unlimitedQuantity')
					->setDefaultValue($variant && !$variant->useUnlimitedQuantity ? $parentProd->unlimitedQuantity : $prod->unlimitedQuantity);
			}
			if (class_exists('EshopGifts\DI\EshopGiftsExtension')) {
				$form->getComponent('categoryGiftsAllowed')->setDefaultValue($variant && !$variant->useCategoryGiftsAllowed ? $parentProd->isCategoryGiftsAllowed() : $prod->isCategoryGiftsAllowed());
				$form->getComponent('orderGiftsAllowed')->setDefaultValue($variant && !$variant->useOrderGiftsAllowed ? $parentProd->orderGiftsAllowed : $prod->orderGiftsAllowed);
			}

			if (class_exists(ImportExtension::class)) {
				$stopImportPrice = ($variant && !$variant->useImportPrice ? $parentProd : $prod)->getMoreDataValue('stopImportPrice') == 1 ? 0 : 1;
				$form->getComponent('stopImportPrice')->setValue($stopImportPrice);
			}

			if (Config::load('product.allowStopImportQuantity')) {
				$stopImportQuantity = ($variant ? $parentProd : $prod)->getMoreDataValue('stopImportQuantity');
				$form->getComponent('stopImportQuantity')->setValue($stopImportQuantity);
			}

			if (Config::load('product.loadQuantityOnStock') && class_exists('EshopStock\DI\EshopStockExtension')) {
				$qb = $this->em->getRepository('EshopStock\Model\Entities\SupplyProduct')->createQueryBuilder('sp')
					->select('COUNT(sp)')
					->join('sp.product', 'p')
					->where('p.id = :id AND sp.order IS NULL AND sp.writtenOffDate IS NULL')
					->setParameter('id', (int) $id);
				$form->getComponent('quantityOnStock')->setValue($qb->getQuery()->getSingleResult());
			}

			// Manufacturer
			if ($prod->getManufacturer() && array_key_exists($prod->getManufacturer()->getId(), $form->getComponent('manufacturer')->getItems())) {
				$form->getComponent('manufacturer')->setDefaultValue($prod->getManufacturer()->getId());
			}

			//Suppliers
			if ($prod->getSuppliers()->count()) {
				$formSupp = $form->addContainer('suppliers');

				foreach ($prod->getSuppliers()->toArray() as $sup) {
					/** @var ProductSupplier $sup */
					$formSupp->addBool('s' . $sup->getSupplier()->getId(), '')
						->setDefaultValue($sup->isActive);
				}
			}

			$this->onAnchor[] = function() use ($prod) {
				$this->template->suppliers = [];
				foreach ($prod->getSuppliers()->toArray() as $productSupplier) {
					/** @var ProductSupplier $productSupplier */
					$this->template->suppliers[$productSupplier->getSupplier()->getId()] = [
						'name'       => $productSupplier->getSupplier()->name,
						'allowSale'  => $productSupplier->getSupplier()->allowSale,
						'code'       => $productSupplier->code,
						'quantity'   => $productSupplier->quantity,
						'supplierId' => $productSupplier->getSupplier()->getId(),
					];
				}
			};

			if (Config::load('product.allowVerifyAge')) {
				$d['verifyAge'] = $prod->verifyAge;
			}

			// Extra Fields
			$ef = $prod->getExtraFieldsValues();
			if (Config::load('product.allowNote')) {
				$d['addNote']   = $ef['addNote'] ?? false;
				$d['noteTitle'] = $ef['noteTitle'] ?? null;
			}

			$form->setDefaults($d);

			// Texty
			$mTexts  = [];
			$seoData = [];

			/** @var ProductTexts[] $prodTexts */
			$prodTexts   = $prod->getTexts()->toArray();
			$parentTexts = [];
			if ($variant) {
				$parentTexts = $this->em->getRepository(ProductTexts::class)->createQueryBuilder('pt', 'pt.lang')
					->where('pt.id = :id')->setParameter('id', $parentProd->getId())
					->getQuery()->getResult();
			}

			foreach ($prodTexts as $lang => $texts) {
				foreach (['name' => 'name', 'shortDesc' => 'shortDescription', 'desc' => 'description'] as $k => $v) {
					$mTexts[$k][$lang] = $variant && !$variant->{'use' . ucfirst($v)} ? $parentTexts[$lang]->$v : $texts->$v;
				}

				if (Config::load('product.allowModifyAlias')) {
					$mTexts['alias'][$lang] = $texts->alias ? Strings::webalize($texts->alias) : Strings::webalize($texts->name);
				}

				if (Config::load('enableProductName2')) {
					$mTexts['name2'][$lang] = $variant && !$variant->useName2 ? $parentTexts[$lang]->name2 : $texts->name2;
				}

				if (Config::load('product.allowPreorder')) {
					$mTexts['preorderText'][$lang] = $texts->preorderText;
				}

				if (Config::load('product.allowConditionDesc')) {
					$mTexts['conditionDescription'][$lang] = $texts->conditionDescription;
				}

				foreach ($texts->getSeo() as $k => $v) {
					$seoData[$k][$lang] = $v;
				}

				if (Config::load('product.publishedByLang')) {
					$mTexts['isPublished'][$lang] = $texts->isPublished;
				}
			}

			if ($prod->isVariant() && $prod->getIsVariantParent()) {
				foreach ($prod->isVariant()->texts as $k => $v) {
					$mTexts['variantName'][$k] = $v->name;
				}
			}

			$form->setDefaults($mTexts);
			$this->seoContainerService->setDefaults($form->getComponent('seo'), (array) $seoData);

			if ($prod->getVateRate() && array_key_exists($prod->getVateRate()->getId(), $form->getComponent('vatRate')->getItems())) {
				$form->getComponent('vatRate')->setDefaultValue($prod->getVateRate()->getId());
			}

			// set Prices
			if (Config::load('enableCountryPrices')) {
				$this->productPricesContainer->setDefaults($form->getComponent('prices'), $prod);
				$this->productPriceLevelCountriesContainer->setDefaults($form->getComponent('priceLevelCountries'), $prod);
			}

			// set kategorie
			$allCategoriesId = array_map(static fn($c) => $c->getCategory()->getId(), $prod->getCategoryProducts()->toArray());
			$this->categoryContainerService->setDefaultsMultipleSite($form->getComponent('categories'), $prod->sites->toArray(), $allCategoriesId);

			// set tags
			$tags = [];
			foreach ($prod->getProductTags()->toArray() as $t) {
				$tags[$t->getTag()->type] = [
					'active'   => 1,
					'dateFrom' => $t->validFrom,
					'dateTo'   => $t->validTo,
					'id'       => $t->getTag()->getId(),
				];
			}
			$this->tagsContainerService->setDefaults($form->getComponent('tags'), $tags);

			// priceLevel
			$prodPriceLevels = $prod->getPriceLevels()->toArray();
			if ($variant && !$variant->usePriceLevels) {
				$prodPriceLevels = [];
				foreach ($this->em->getRepository(ProductPriceLevel::class)->createQueryBuilder('pl')
					         ->where('pl.productId = :id')->setParameter('id', $parentProd->getId())
					         ->getQuery()->getResult() as $row) {
					$prodPriceLevels[$row->getGroupId()->getId()] = $row;
				}
			}
			if ($this->helper->config->get('enablePriceLevels', false)) {
				foreach ($this->helper->getGroupCustomers() as $gc) {
					if ($form->getComponent('pricelevel' . $gc->getId(), false) && isset($prodPriceLevels[$gc->getId()])) {
						$form->getComponent('pricelevel' . $gc->getId())->setDefaultValue($prodPriceLevels[$gc->getId()]->price);
					}
				}
			}

			$this->monitor(IPresenter::class, function() use ($prod, $form) {
				$documentsData = [];
				foreach ($this->productDocuments->getByProduct($prod->getId()) as $document) {
					$documentsData[] = [
						'id'         => $document->getId(),
						'fileName'   => $document->name,
						'filePath'   => $document->file,
						'lang'       => $document->lang,
						'removeLink' => $this->link('removeDocument', $document->getId()),
					];
				}
				$this->documentContainer->setDefaults($form->getComponent('documents'), $documentsData);

				$videos = [];
				foreach ($this->productVideos->getByProduct($prod->getId()) as $video) {
					$arr = [
						'id'         => $video->getId(),
						'title'      => $video->title,
						'lang'       => $video->lang,
						'thumbnail'  => $video->thumbnail,
						'removeLink' => $this->link('removeVideo', $video->getId()),
					];

					if ($video->url && Strings::startsWith($video->url, 'http')) {
						$arr['url'] = $video->url;
					} else if (Strings::startsWith($video->url, '<')) {
						$arr['iframe'] = $video->url;
					} else if ($video->url) {
						$arr['file'] = $video->url;
					}

					$videos[] = $arr;
				}
				$this->videoContainer->setDefaults($form->getComponent('videos'), $videos);

				if (Config::load('allowRelatedProducts')) {
					$this->relatedProductContainer->setDefaults($form->getComponent('relatedProducts'), $prod->getRelatedProducts()
						->toArray());
				}

				// Doprava
				if (Config::load('product.allowModifySpedition', false)) {
					$this->speditionContainer->setDefaults(
						$form->getComponent('speditions'),
						$prod->getSpeditions()->toArray(),
						$prod
					);
				}

				// Platba
				if (Config::load('product.allowModifyPayment', false)) {
					$this->paymentContainer->setDefaults(
						$form->getComponent('payments'),
						$prod->payments->toArray(),
						$prod
					);
				}

				// Package
				if (Config::load('product.allowPackage', false) && $prod->package) {
					$this->packageContainer->setDefaults($form->getComponent('package'), $prod->package);
				}
			});

			$this->eventDispatcher->dispatch(new SetFormDataEvent($form, $this->product), ProductForm::class . '::setProduct');
		} else {
			$this->presenter->error();
		}
	}

	protected function getFeatures(): array
	{
		if ($this->cFeatures === null) {

			$this->cFeatures = [];
			foreach ($this->featureServices->getEr()
				         ->createQueryBuilder('f')
				         ->select('f.id, ft.name, f.type, f.unit, f.decimals')
				         ->join('f.featureTexts', 'ft', 'WITH', 'ft.lang = :lang')
				         ->setParameter('lang', $this->translator->getLocale())
				         ->orderBy('f.position')
				         ->getQuery()
				         ->getResult() as $v) {
				$v['name'] = trim((string) $v['name']);

				if ($v['type'] === Feature::TYPE_RANGE) {
					$v['step'] = Feature::getStep((int) $v['decimals']);
				}

				$this->cFeatures[$v['id']] = $v;
			}
		}

		return $this->cFeatures;
	}

	protected function getFeatureValues(?int $id = null): array
	{
		if ($this->cFeatureValues === null) {
			$this->cFeatureValues = [];

			foreach ($this->featureServices->em->getRepository(FeatureValue::class)
				         ->createQueryBuilder('fv')
				         ->select('fv.id, fvt.name, IDENTITY(fv.feature) as featureId')
				         ->join('fv.featureValueTexts', 'fvt', 'WITH', 'fvt.lang = :lang')
				         ->setParameter('lang', $this->translator->getLocale())
				         ->andWhere('fv.isPublished = 1')
				         ->orderBy('fv.position')
				         ->getQuery()
				         ->getResult() as $v) {
				$this->cFeatureValues[$v['featureId']][] = [
					'id'   => $v['id'],
					'name' => trim($v['name']),
				];
			}
		}

		return $id ? $this->cFeatureValues[$id] ?? [] : $this->cFeatureValues;
	}

	public static function allowSaleDiscountCouponsAsProducts(): bool
	{
		return Parameters::load('eshopSales.autoGenerateDiscountCoupon.enable', false) && class_exists('EshopSales\DI\EshopSalesExtension');
	}

	public function createComponentPriceHistoryGrid(): PriceHistoryGrid
	{
		return $this->priceHistoryGridFactory->create($this->product->getId());
	}
}
