<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Products;

use Core\AdminModule\Model\Sites;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\AdminModule\Model\CategoryProducts;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\Model\AvailabilityService;
use EshopCatalog\Model\Entities\CategoryProduct;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductInSite;
use Exception;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;

class ProductsCategoryForm extends BaseControl
{
	public array                  $products = [];
	private ?string               $ids      = '';
	protected Products            $productServices;
	protected CategoryProducts    $categoryProductServices;
	protected Categories          $categoryServices;
	protected CategoryContainer   $categoryContainerService;
	protected Sites               $sitesService;
	protected AvailabilityService $availabilities;

	public function __construct(
		Products            $products,
		Categories          $categories,
		CategoryProducts    $categoryProducts,
		CategoryContainer   $categoryContainer,
		Sites               $sites,
		AvailabilityService $availabilities
	)
	{
		$this->productServices          = $products;
		$this->categoryContainerService = $categoryContainer;
		$this->categoryServices         = $categories;
		$this->categoryProductServices  = $categoryProducts;
		$this->sitesService             = $sites;
		$this->availabilities           = $availabilities;
	}

	public function render(): void
	{
		$this->template->sites = $this->sitesService->getAll();
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$form->addHidden('produtcIds', $this->ids);
		$form->addComponent($this->categoryContainerService->getContainers(), 'categories');

		$form->addSelect('editType', 'eshopCatalog.productsCategoryForm.editType', [
			'default'                => 'eshopCatalog.productsCategoryForm.type.default',
			'replaceOtherCategories' => 'eshopCatalog.productsCategoryForm.type.replaceOtherCategories',
			'addOtherCategories'     => 'eshopCatalog.productsCategoryForm.type.addOtherCategories',
		]);

		$form->addSaveCancelControl();
		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$this->em->beginTransaction();
		try {
			$ids = explode('-', $this->ids);
			$this->categoryContainerService->saveDataMultipleSite($ids, (array) $values->categories, $values->editType);
			$this->em->flush();
			$this->em->commit();

			$this->em->clear();

			foreach (array_chunk($ids, 10) as $chunk) {
				foreach ($this->em->getRepository(Product::class)->findBy(['id' => $chunk]) as $v) {
					/** @var Product $v */
					if (!$v->getAvailability()) {
						$this->availabilities->updateAvailabilityByQuantity($v);

						$this->em->persist($v);
					}
				}
			}

			$this->em->flush();

			foreach ($ids as $id) {
				$this->productServices->clearProductCache((int) $id);
			}

			$form->addCustomData('productIds', $values->produtcIds);
			$this->presenter->flashMessageSuccess('eshopCatalog.productForm.productSaved');
		} catch (Exception $e) {
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->rollback();
			}
			$form->addError($e->getMessage());

			return false;
		}

		return true;
	}

	public function setProducts(?string $ids): void
	{
		if (!$ids) {
			throw new InvalidArgumentException;
		}

		$this->ids = $ids;
		$idsArray  = explode('-', $ids);

		$productSites = [];
		foreach ($this->em->getRepository(ProductInSite::class)->createQueryBuilder('ps')
			         ->where('ps.product = :id')
			         ->setParameter('id', end($idsArray))
			         ->getQuery()->getResult() as $row) {
			$productSites[$row->getSite()->getIdent()] = $row;
		}

		$categories = [];
		foreach ($this->em->getRepository(CategoryProduct::class)->createQueryBuilder('cp')
			         ->select('IDENTITY(cp.category) as cat')
			         ->where('cp.product IN (:ids)')
			         ->setParameter('ids', $idsArray)
			         ->getQuery()->getScalarResult() as $row) {
			$categories[] = $row['cat'];
		}

		$categories = array_unique($categories);
		$this->categoryContainerService->setDefaultsMultipleSite($this['form']->getComponent('categories'), $productSites, $categories);
	}

}
