<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components;

use Core\AdminModule\Components\DataForm;
use Core\AdminModule\Model\Sites;
use Core\Model\Countries;
use Core\Model\Helpers\Strings;
use EshopCatalog\AdminModule\Model\AvailabilityService;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Availability;
use EshopOrders\AdminModule\Model\Payments;
use Users\Model\Roles;

class SettingsForm extends DataForm
{
	/** @var Sites|null @inject */
	public ?Sites $sitesService = null;

	/** @var Roles|null @inject */
	public ?Roles $roles = null;

	/** @var Payments|null @inject */
	public ?Payments $payments = null;

	/** @var AvailabilityService|null @inject */
	public ?AvailabilityService $availabilityService = null;

	/** @var Countries|null @inject */
	public ?Countries $countries = null;

	public function getDataForInputs(): array
	{
		$data = parent::getDataForInputs();

		foreach ($this->sitesService->getOptionsForSelect() as $site) {
			$data[] = [
				'name'      => $site . 'RegisterUserSale',
				'title'     => [
					0 => 'eshopCatalog.settings.registerUserSale',
				],
				'type'      => 'number',
				'step'      => .01,
				'min'       => 0,
				'default'   => 0,
				'section'   => $site,
				'group'     => 'base',
				'groupName' => 'eshopCatalog.settings.baseSettings',
			];
			$data[] = [
				'name'    => $site . 'SortingInformationPage',
				'title'   => [
					0 => 'eshopCatalog.settings.sortingInformationPage',
				],
				'type'    => 'navByGroup',
				'showIf'  => Config::load('enableSortingInformationPage', false),
				'site'    => $site,
				'section' => $site,
				'group'   => 'base',
			];
			$data[] = [
				'name'    => $site . 'PlaceholderImage',
				'title'   => [
					0 => 'eshopCatalog.settings.placeholderImage',
				],
				'type'    => 'image',
				'site'    => $site,
				'section' => $site,
				'group'   => 'base',
			];

			$data[] = [
				'name'      => $site . 'DeliveryPage',
				'title'     => [
					0 => 'eshopCatalog.settings.delivery.deliveryPage',
				],
				'type'      => 'navByGroup',
				'site'      => $site,
				'section'   => $site,
				'group'     => 'delivery',
				'groupName' => 'eshopCatalog.settings.delivery.title',
			];

			if (Config::load('enableDeliveryTo', false)) {
				$data[] = [
					'name'      => $site . 'DeliveryExpeditionToDesc',
					'title'     => [
						0 => 'eshopCatalog.settings.delivery.expeditionToDesc',
					],
					'type'      => 'textarea',
					'multiLang' => true,
					'section'   => $site,
					'group'     => 'delivery',
				];

				$data[] = [
					'name'    => $site . 'DeliveryExpeditionTo',
					'title'   => [
						0 => 'eshopCatalog.settings.delivery.expeditionToTime',
					],
					'type'    => 'time',
					'section' => $site,
					'group'   => 'delivery',
				];

				$baseDaysText         = $this->translator->translate('eshopCatalog.settings.delivery.expeditionDeliveryBaseDays');
				$baseDaysTextSupplier = $this->translator->translate('eshopCatalog.settings.delivery.expeditionDeliveryBaseDaysSupplier');

				foreach ($this->em->getConnection()->executeQuery("SELECT country_id FROM eshop_orders__payment_spedition_country GROUP BY country_id")->iterateAssociative() as $row) {
					$countryId = Strings::upper($row['country_id']);

					$data[] = [
						'name'    => $site . 'DeliveryExpeditionDeliveryBaseDays' . $countryId,
						'title'   => $baseDaysText . ' - ' . $countryId,
						'type'    => 'number',
						'section' => $site,
						'group'   => 'delivery',
					];

					if (class_exists('EshopStock\DI\EshopStockExtension')) {
						$data[] = [
							'name'    => $site . 'DeliveryExpeditionDeliveryBaseDaysSupplier' . $countryId,
							'title'   => $baseDaysTextSupplier . ' - ' . $countryId,
							'type'    => 'number',
							'section' => $site,
							'group'   => 'delivery',
						];
					}
				}
			}

			$data[] = [
				'name'      => $site . 'TermsAndConditionsNavId',
				'title'     => [
					0 => 'eshopCatalog.settings.termsAndConditionsNavId',
				],
				'type'      => 'text',
				'section'   => $site,
				'group'     => 'termsAndConditions',
				'groupName' => 'eshopCatalog.settings.termsAndConditions',
			];
			$data[] = [
				'name'    => $site . 'GdprNavId',
				'title'   => [
					0 => 'eshopCatalog.settings.gdprNavId',
				],
				'type'    => 'text',
				'section' => $site,
				'group'   => 'termsAndConditions',
			];

			$data[] = [
				'name'      => $site . 'EnableReturn',
				'title'     => [
					0 => 'eshopCatalog.settings.enableReturn',
				],
				'type'      => 'bool',
				'section'   => $site,
				'group'     => 'returnPolicy',
				'groupName' => 'eshopCatalog.settings.returnPolicy',
			];
			$data[] = [
				'name'    => $site . 'ReturnPolicy',
				'title'   => [
					0 => 'eshopCatalog.settings.returnPolicy',
				],
				'type'    => 'link',
				'section' => $site,
				'group'   => 'returnPolicy',
			];
			$data[] = [
				'name'    => $site . 'ReturnApplicableCountry',
				'title'   => [
					0 => 'eshopCatalog.settings.returnApplicableCountry',
				],
				'type'    => 'selectBox',
				'items'   => $this->countries->getAllNameColumn(),
				'section' => $site,
				'group'   => 'returnPolicy',
			];
			$data[] = [
				'name'        => $site . 'ReturnDays',
				'title'       => [
					0 => 'eshopCatalog.settings.returnDays',
				],
				'type'        => 'text',
				'description' => 'eshopCatalog.settings.returnDaysDescription',
				'section'     => $site,
				'group'       => 'returnPolicy',
			];
			$data[] = [
				'name'    => $site . 'ReturnMethod',
				'title'   => [
					0 => 'eshopCatalog.settings.returnMethod',
				],
				'type'    => 'selectBox',
				'items'   => [
					'ReturnByMail'  => 'eshopCatalog.settings.returnList.returnByMail',
					'ReturnInStore' => 'eshopCatalog.settings.returnList.returnInStore',
					'ReturnAtKiosk' => 'eshopCatalog.settings.returnList.returnAtKiosk',
				],
				'section' => $site,
				'group'   => 'returnPolicy',
			];
			$data[] = [
				'name'    => $site . 'ReturnFees',
				'title'   => [
					0 => 'eshopCatalog.settings.returnFees',
				],
				'type'    => 'selectBox',
				'items'   => [
					'FreeReturn'                       => 'eshopCatalog.settings.returnFeesList.freeReturn',
					'OriginalShippingFees'             => 'eshopCatalog.settings.returnFeesList.originalShippingFees',
					'RestockingFees'                   => 'eshopCatalog.settings.returnFeesList.restockingFees',
					'ReturnFeesCustomerResponsibility' => 'eshopCatalog.settings.returnFeesList.returnFeesCustomerResponsibility',
					'ReturnShippingFees'               => 'eshopCatalog.settings.returnFeesList.returnShippingFees',
				],
				'section' => $site,
				'group'   => 'returnPolicy',
			];

			if (class_exists('EshopProductsComparison\DI\EshopProductsComparisonExtension')) {
				$data[] = [
					'name'      => $site . 'XmlFeedZboziExtendedReturn',
					'title'     => [
						0 => 'eshopCatalog.settings.xmlFeedZbozi.extendedReturn',
					],
					'type'      => 'bool',
					'section'   => $site,
					'group'     => 'xmlFeedZbozi',
					'groupName' => 'eshopCatalog.settings.xmlFeedZbozi.title',
				];
				$data[] = [
					'name'    => $site . 'XmlFeedZboziExtendedReturnDateTo',
					'title'   => [
						0 => 'eshopCatalog.settings.xmlFeedZbozi.extendedReturnDateTo',
					],
					'type'    => 'date',
					'section' => $site,
					'group'   => 'xmlFeedZbozi',
				];

				for ($i = 1; $i <= 5; $i++) {
					$data[] = [
						'name'      => $site . 'XmlFeedHeurekaSpecialService' . $i,
						'title'     => [
							0 => 'eshopCatalog.settings.xmlFeedHeureka.specialService',
						],
						'multiLang' => true,
						'type'      => 'text',
						'section'   => $site,
						'group'     => 'xmlFeedHeureka',
						'groupName' => 'eshopCatalog.settings.xmlFeedHeureka.title',
					];
					$data[] = [
						'name'    => $site . 'XmlFeedHeurekaSpecialServiceDateTo' . $i,
						'title'   => [
							0 => 'eshopCatalog.settings.xmlFeedHeureka.specialServiceDateTo',
						],
						'type'    => 'date',
						'section' => $site,
						'group'   => 'xmlFeedHeureka',
					];
				}
			}
		}

		$data[] = [
			'name'  => 'disableRegisterUserSaleForRole',
			'title' => [
				0 => 'eshopCatalog.settings.disableRegisterUserSaleForRole',
			],
			'items' => $this->roles->getForSelectOptions(),
			'type'  => 'checkboxList',
		];

		if (isset($this->availabilityService->getAllByIdent()[Availability::PREORDER])) {
			$data[] = [
				'name'  => 'allowedPaymentsForPreorder',
				'title' => [
					0 => 'eshopCatalog.settings.allowedPaymentsForPreorder',
				],
				'items' => $this->payments->getForSelectOption(),
				'type'  => 'checkboxList',
			];
		}

		$dataFromConfig = Config::load('settingsForm');
		if ($dataFromConfig) {
			$data = array_merge($data, $dataFromConfig);
		}

		return $data;
	}

}
