<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model;

use Core\Model\Helpers\BaseEntityService;
use Doctrine\DBAL\Connection;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\ProductReplenishmentPlan;
use Nette\InvalidArgumentException;
use Nette\Utils\DateTime;
use Nette\Utils\Floats;
use Nette\Utils\Json;
use Nette\Utils\Strings;

/**
 * @method ProductReplenishmentPlan|null getReference($id)
 * @method ProductReplenishmentPlan[] getAll()
 */
class ProductReplenishmentPlanner extends BaseEntityService
{
	protected $entityClass = ProductReplenishmentPlan::class;

	protected DateTime $now;
	protected array $calcCache = [];

	public function __construct()
	{
		// skript se spousti o pulnoci, vypocet se kona tedy k predeslemu dni
		$this->now = (new DateTime)->modify('-1 day')->setTime(23, 59, 59);
	}

	/**
	 * @return array{num: int, unit: string, unitName: string, singleUnitName: string}
	 */
	protected static function parseTimeDuration(string $value): array
	{
		$return = ['num' => 0, 'unit' => 'd', 'unitName' => 'days', 'singleUnitName' => 'day'];

		// 3m => [3m, 3, m]
		$parts = Strings::match(Strings::lower($value), '/^(\d+)\s*([mdy])$/i');

		if ($parts) {
			$return['num']  = (int) $parts[1];
			$return['unit'] = $parts[2];

			switch ($return['unit']) {
				case 'd':
					$return['singleUnitName'] = 'day';
					$return['unitName']       = 'days';
					break;
				case 'm':
					$return['singleUnitName'] = 'month';
					$return['unitName']       = 'months';
					break;
				case 'y':
					$return['singleUnitName'] = 'year';
					$return['unitName']       = 'years';
					break;
			}
		}

		return $return;
	}

	protected static function timeDurationToDays(int $num, string $unit): int
	{
		switch ($unit) {
			case 'd':
				return $num;
			case 'm':
				return $num * 30;
			case 'y':
				return $num * 365;
		}

		throw new InvalidArgumentException('Unit unknown');
	}

	/**
	 * @return array{id: int, name: string, leadTimeDays: int}[]
	 */
	protected function getManufacturers(): array
	{
		$sites = Config::load('productReplenishmentPlanner.sites');
		$conn  = $this->em->getConnection();

		$params = [];
		$types  = [];
		$sql    = "SELECT DISTINCT m.id, m.name, m.lead_time_days FROM eshop_catalog__product_in_site pis 
    		JOIN eshop_catalog__product p ON (pis.product_id = p.id)
    		JOIN eshop_catalog__manufacturer m ON p.id_manufacturer = m.id
    		WHERE p.is_published = 1 AND p.skip_for_replenishment_planning = 0 AND pis.is_active = 1 AND m.lead_time_days IS NOT NULL AND m.lead_time_days <> ''";

		if ($sites) {
			$sql             .= ' AND pis.site IN (:sites)';
			$params['sites'] = $sites;
			$types['sites']  = Connection::PARAM_STR_ARRAY;
		}

		$mans = [];
		foreach ($conn->executeQuery($sql, $params, $types)->iterateAssociative() as $v) {
			$id        = (int) $v['id'];
			$mans[$id] = [
				'id'           => $id,
				'name'         => $v['name'],
				'leadTimeDays' => (int) $v['lead_time_days'],
			];
		}

		return $mans;
	}

	protected function computeSeasonCoefByManufacturer(): array
	{
		$sites          = Config::load('productReplenishmentPlanner.sites');
		$conn           = $this->em->getConnection();
		$manufacturers  = $this->getManufacturers();
		$avgSalesPeriod = self::parseTimeDuration(Config::load('productReplenishmentPlanner.avgSalesPeriod'));

		$getDailyAvgQuantityForPeriod = function(DateTime $from, DateTime $to, array $m) use ($sites, $conn) {
			$daysInterval  = $from->modifyClone()->setTime(0, 0)->diff($to->modifyClone()
																		  ->setTime(23, 59, 59))->days + 1;
			$dateFrom      = $from->format('Y-m-d 00:00:00');
			$dateTo        = $to->format('Y-m-d 23:59:59');
			$params        = [
				'manId'    => $m['id'],
				'dateFrom' => $dateFrom,
				'dateTo'   => $dateTo,
			];
			$types         = [];
			$siteCondition = '';
			if ($sites) {
				$siteCondition   = 'AND o.site_id IN (:sites)';
				$params['sites'] = $sites;
				$types['sites']  = Connection::PARAM_STR_ARRAY;
			}

			$sql = "SELECT SUM(oi.quantity) sum_quantity FROM eshop_orders__order o
				JOIN eshop_orders__order_item oi ON (o.id = oi.order_id {$siteCondition} AND o.is_corrective_tax_document = 0)
				JOIN eshop_orders__order_status os ON (o.id = os.order_id AND os.status_id = 'created' AND (os.created BETWEEN :dateFrom AND :dateTo))
				JOIN eshop_catalog__product p ON (oi.product_id = p.id AND p.is_published = 1 AND p.skip_for_replenishment_planning = 0)
				JOIN eshop_catalog__manufacturer m ON (p.id_manufacturer = m.id AND m.id = :manId)";

			return ((float) ($conn->executeQuery($sql, $params, $types)->fetchOne() ?? 0.0)) / $daysInterval;
		};

		foreach ($manufacturers as $k => $m) {
			$manufacturers[$k]['seasonCoef'] = 1.0;

			$futureAvg = $getDailyAvgQuantityForPeriod(
				$this->now->modifyClone('+ 1 day - 1 year'),
				$this->now->modifyClone("+ {$m['leadTimeDays']} days - 1 year"),
				$m
			);
			$pastAvg   = $getDailyAvgQuantityForPeriod(
				$this->now->modifyClone("- {$avgSalesPeriod['num']} {$avgSalesPeriod['unitName']} - 1 year"),
				$this->now->modifyClone('- 1 year'),
				$m
			);

			if (!Floats::isZero($futureAvg) && !Floats::isZero($pastAvg)) {
				$coef                            = $futureAvg / $pastAvg;
				$manufacturers[$k]['seasonCoef'] = $coef;
			} else {
				$manufacturers[$k]['seasonCoef'] = null;
			}
		}

		return $manufacturers;
	}

	protected function getSalesStatistics(int $limit, int $offset): array
	{
		$sites          = Config::load('productReplenishmentPlanner.sites');
		$conn           = $this->em->getConnection();
		$avgSalesPeriod = Config::load('productReplenishmentPlanner.avgSalesPeriod');
		$statisticsList = Config::load('productReplenishmentPlanner.statistics');
		$params         = [];
		$types          = [];

		if (!in_array('0y_' . $avgSalesPeriod, $statisticsList)) {
			$statisticsList[] = '0y_' . $avgSalesPeriod;
		}

		$clauseSelect = ['p.id', 'p.quantity', 'p.id_manufacturer'];

		foreach ($statisticsList as $item) {
			$periodParts                  = explode('_', $item);
			$stepBackData                 = self::parseTimeDuration($periodParts[0]);
			$periodData                   = self::parseTimeDuration($periodParts[1]);
			$periodData['singleUnitName'] = Strings::upper($periodData['singleUnitName']);
			$dateFromCol                  = $item . '__date_from';
			$dateToCol                    = $item . '__date_to';
			$salesCol                     = $item . '__sales';

			if ($stepBackData['num'] > 0) {
				$datePastYmd     = $this->now->modifyClone('- ' . $stepBackData['num'] . ' ' . $stepBackData['unitName'])
											 ->format('Y-m-d');
				$dateFromSqlPart = "DATE(DATE_SUB('$datePastYmd', INTERVAL {$periodData['num']} {$periodData['singleUnitName']})) $dateFromCol";
				$dateToSqlPart   = "DATE('$datePastYmd') $dateToCol";
				$salesSqlPart    = "COALESCE(
					SUM(
						CASE
							WHEN os.created BETWEEN
								DATE_SUB('$datePastYmd', INTERVAL {$periodData['num']} {$periodData['singleUnitName']})
								AND
								'$datePastYmd'
								THEN oi.quantity
								ELSE 0
								END
						   ), 0
				   ) $salesCol";
			} else {
				$dateNowYmd      = $this->now->format('Y-m-d');
				$dateFromSqlPart = "DATE(IF(p.quantity > 0 OR ls.last_sale IS NULL, DATE_SUB('$dateNowYmd', INTERVAL {$periodData['num']} {$periodData['singleUnitName']}), 
					DATE_SUB(ls.last_sale, INTERVAL {$periodData['num']} {$periodData['singleUnitName']}))) $dateFromCol";
				$dateToSqlPart   = "DATE(IF(p.quantity > 0 OR ls.last_sale IS NULL, '$dateNowYmd', ls.last_sale)) $dateToCol";
				$salesSqlPart    = "COALESCE(
					SUM(
						CASE
					   		WHEN os.created BETWEEN
								IF(p.quantity > 0 OR ls.last_sale IS NULL, DATE_SUB('$dateNowYmd', INTERVAL {$periodData['num']} {$periodData['singleUnitName']}),
									DATE_SUB(ls.last_sale, INTERVAL {$periodData['num']} {$periodData['singleUnitName']}))
								AND
								IF(p.quantity > 0 OR ls.last_sale IS NULL, '$dateNowYmd', ls.last_sale)
							THEN oi.quantity ELSE 0
							END
							), 0
				   ) $salesCol";
			}

			$clauseSelect[] = $dateFromSqlPart;
			$clauseSelect[] = $dateToSqlPart;
			$clauseSelect[] = $salesSqlPart;
		}

		$clauseSelectStr = implode(', ', $clauseSelect);
		$clauseWith      = "WITH last_sales AS (SELECT p.id AS product_id, MAX(os.created) last_sale
                    FROM eshop_catalog__product p
                             JOIN eshop_orders__order_item oi ON p.id = oi.product_id
                             JOIN eshop_orders__order o ON (o.id = oi.order_id AND
                                                            o.is_corrective_tax_document = 0)
                             JOIN eshop_orders__order_status os ON (o.id = os.order_id AND os.status_id = 'created')
                    GROUP BY p.id)";
		$siteCondition   = '';
		if ($sites) {
			$siteCondition   = 'AND pis.site IN (:sites)';
			$params['sites'] = $sites;
			$types['sites']  = Connection::PARAM_STR_ARRAY;
		}

		$sql = "$clauseWith SELECT $clauseSelectStr
			FROM eshop_catalog__product p JOIN eshop_catalog__manufacturer m ON (p.id_manufacturer = m.id AND p.is_published = 1 AND p.skip_for_replenishment_planning = 0)
			JOIN eshop_catalog__product_in_site pis ON (pis.product_id = p.id AND pis.is_active = 1 $siteCondition)
			LEFT JOIN last_sales ls ON ls.product_id = p.id
			LEFT JOIN eshop_orders__order_item oi ON p.id = oi.product_id
			LEFT JOIN eshop_orders__order o ON o.id = oi.order_id
			LEFT JOIN eshop_orders__order_status os ON (o.id = os.order_id AND os.status_id = 'created')
			WHERE (o.is_corrective_tax_document IS NULL OR o.is_corrective_tax_document = 0)
			GROUP BY p.id
			LIMIT $limit OFFSET $offset
		";

		return $conn->executeQuery($sql, $params, $types)->fetchAllAssociative();
	}

	public function runCalc(): void
	{
		$conn = $this->em->getConnection();
		$conn->transactional(function() use ($conn) {
			$limit                = 500;
			$offset               = 0;
			$seasonCoefs          = $this->computeSeasonCoefByManufacturer();
			$statisticsList       = Config::load('productReplenishmentPlanner.statistics');
			$avgSalesPeriod       = Config::load('productReplenishmentPlanner.avgSalesPeriod');
			$avgSalesPeriodParsed = self::parseTimeDuration($avgSalesPeriod);
			$numDays              = self::timeDurationToDays($avgSalesPeriodParsed['num'], $avgSalesPeriodParsed['unit']);
			$lastUpdate           = (new DateTime)->format('Y-m-d H:i:s');

			$conn->executeQuery("DELETE FROM eshop_catalog__product_replenishment_plan");

			while (true) {
				$rows = $this->getSalesStatistics($limit, $offset);
				foreach ($rows as $row) {
					$productId       = (int) $row['id'];
					$productQuantity = max(((int) $row['quantity']), 0);
					$manufacturerId  = (int) $row['id_manufacturer'];

					if (!array_key_exists($manufacturerId, $seasonCoefs) || !$seasonCoefs[$manufacturerId]['leadTimeDays']) {
						continue;
					}

					$manufacturer    = $seasonCoefs[$manufacturerId];
					$manLeadTimeDays = $manufacturer['leadTimeDays'];
					$manSeasonCoef   = $manufacturer['seasonCoef'];

					$statistics = [];
					foreach ($statisticsList as $sItem) {
						$statistics[$sItem] = [
							'dateFrom' => $row[$sItem . '__date_from'],
							'dateTo'   => $row[$sItem . '__date_to'],
							'pieces'   => (int) $row[$sItem . '__sales'],
						];
					}

					if ($manSeasonCoef === null) {
						$recommendedQuantity = 1;
					} else {
						$avgSales            = ((int) $row['0y_' . $avgSalesPeriod . '__sales']) / $numDays;
						$recommendedQuantity = $avgSales * $manLeadTimeDays;
						$recommendedQuantity = (int) round(max(($recommendedQuantity * $manSeasonCoef) - $productQuantity, 0));

						if ($productQuantity === 0 && $recommendedQuantity === 0) {
							$recommendedQuantity = 1;
						}
					}

					$conn->executeQuery("INSERT INTO eshop_catalog__product_replenishment_plan
						(`product_id`, `last_update`, `recommended_quantity`, `sales_statistics`)
						VALUES (:productId, :lastUpdate, :recommendedQuantity, :salesStatistics)",
						[
							'productId'           => $productId,
							'lastUpdate'          => $lastUpdate,
							'recommendedQuantity' => $recommendedQuantity,
							'salesStatistics'     => Json::encode($statistics),
						]);
				}

				if (count($rows) < $limit) {
					break;
				} else {
					$offset += $limit;
				}
			}
		});
	}

}