<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\Event;
use Core\Model\Sites;
use EshopCatalog\FrontModule\Model\AvailabilityService;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\FrontModule\Model\Subscribers\ContentSubscriber;
use Nette\Caching\Cache;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use voku\helper\HtmlDomParser;

class CheckMissingEditorDataSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator $em;
	protected ProductsFacade         $productsFacade;
	protected Sites                  $sites;
	protected CacheService           $cacheService;
	protected AvailabilityService    $availabilityService;

	public function __construct(
		EntityManagerDecorator $em,
		ProductsFacade         $productsFacade,
		Sites                  $sites,
		CacheService           $cacheService,
		AvailabilityService    $availabilityService
	)
	{
		$this->em                  = $em;
		$this->productsFacade      = $productsFacade;
		$this->sites               = $sites;
		$this->cacheService        = $cacheService;
		$this->availabilityService = $availabilityService;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'core.content.checkMissingEditorData' => 'checkMissingEditorData',
		];
	}

	public function checkMissingEditorData(Event $event): void
	{
		$this->cacheService->productCache->clean([Cache::Tags => ['editorData']]);

		foreach ($event->data['data'] as &$row) {
			if (!isset($row['content'])) {
				continue;
			}

			$htmlDom = HtmlDomParser::str_get_html($row['content']);

			foreach (ContentSubscriber::findEshopCatalogProduct($htmlDom) as $htmlRow) {
				$ids = ContentSubscriber::findEshopCatalogProductData($htmlRow);

				$available = [];

				foreach ($this->productsFacade->getProducts($ids) as $product) {
					if ($product->canAddToCart) {
						$available[] = $product->getId();
					}
				}

				$missingIds = array_diff($ids, $available);

				foreach ($missingIds as $id) {
					$row['missingData'][] = [
						'module'         => 'EshopCatalog',
						'id'             => $id,
						'type'           => 'eshopCatalogProduct',
						'translateAttrs' => [
							'ids' => implode(', ', $missingIds),
						],
					];
				}
			}

			foreach (ContentSubscriber::findEshopCatalogProductFromCat($htmlDom) as $htmlRow) {
				if (!ContentSubscriber::findEshopCatalogProductFromCatData($this->em, $htmlRow, $this->cacheService, $this->sites->getCurrentSite()->getIdent(), $this->availabilityService)) {
					$row['missingData'][] = [
						'module' => 'EshopCatalog',
						'type'   => 'eshopCatalogProductFromCat',
					];
				}
			}
		}
	}
}
