<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\UI\Form\BootstrapRenderer;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Entities\Group;
use EshopCatalog\Model\Config;
use Navigations\AdminModule\Components\NavigationForm;
use Navigations\AdminModule\Model\Navigations;
use Navigations\Model\Entities\Navigation;
use Nette\DI\Container;
use Nette\Http\Request;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class NavigationFormSubscriber implements EventSubscriberInterface
{
	protected Container              $container;
	protected EntityManagerDecorator $em;
	protected Navigations            $navigations;
	protected Request                $request;

	protected static ?Navigation $navigation = null;

	public function __construct(
		Container              $container,
		EntityManagerDecorator $em,
		Navigations            $navigations,
		Request                $request
	)
	{
		$this->container   = $container;
		$this->em          = $em;
		$this->navigations = $navigations;
		$this->request     = $request;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			NavigationForm::class . '::createForm'    => ['createForm', 100],
			NavigationForm::class . '::formSuccess'   => ['formSuccess', 100],
			NavigationForm::class . '::setNavigation' => ['setNavigation', 100],
		];
	}

	public function createForm(CreateFormEvent $event): void
	{
		$form = $event->form;
		/** @var ?Navigation $navigation */
		$navigation = $event->data['entity'] ?? null;

		$container = $form->addContainer('eshopCatalog');
		$container->addBool('hideInEshopSubMenu', 'eshopCatalog.navigation.hideInEshopSubMenu')
				  ->setDefaultValue(0);
		$container->addBool('hideInEshopCategoryDetail', 'eshopCatalog.navigation.hideInEshopCategoryDetail')
				  ->setDefaultValue(0);

		$navigations = [];
		if ($navigation) {
			$navigations = $this->navigations->getOptionsForSelectParentsInSite($navigation->getSite()->getIdent());
		}

		$container->addSortableCheckboxList('topNavigationsInCategory', 'eshopCatalog.navigation.topNavigationsInCategory', $navigations);
		$container->addSortableCheckboxList('bottomNavigationsInCategory', 'eshopCatalog.navigation.bottomNavigationsInCategory', $navigations);

		// Dalsi obsah z dynamickeho modulu
		if (class_exists('EshopCatalog\Model\Config')) {
			$dynamicModuleCategoryDetail = Config::load('category.dynamicModuleCategoryDetailLink');
			if ($dynamicModuleCategoryDetail && $dynamicModuleCategoryDetail['module'] && class_exists('DynamicModule\Model\DynamicModuleConfig')) {
				/** @var \DynamicModule\Model\Repository\Groups|null $dmGroups */
				$dmGroups = $this->container->hasService('dynamicmodule.admin.groups')
					? $this->container->getService('dynamicmodule.admin.groups')
					: null;

				if ($dmGroups) {
					$qb = $dmGroups->getQueryBuilderByModule($dynamicModuleCategoryDetail['module'])
						->addOrderBy('g.lft', 'ASC');

					if (!DynamicModuleConfig::load('multiLangPublication')) {
						$qb->andWhere('g.isPublished = 1');
					}

					$list = [];

					foreach ($qb->getQuery()->getResult() as $g) {
						/** @var Group $g */
						$prefix = [];
						$p      = $g->getParent();

						while ($p) {
							if ($p->lvl <= 0) {
								break;
							}

							$prefix[] = $p->title;
							$p        = $p->getParent();
						}

						$list[$g->getId()] = ($prefix ? implode(' > ', array_reverse($prefix)) . ' > ' : '') . $g->title;
					}

					$container->addSortableCheckboxList('topDynamicModuleInCategory', 'eshopCatalog.navigation.topDynamicModuleInCategory', $list);
					$container->addSortableCheckboxList('bottomDynamicModuleInCategory', 'eshopCatalog.navigation.bottomDynamicModuleInCategory', $list);
				}
			}
		}

		/** @var BootstrapRenderer $renderer */
		$renderer = $event->form->getRenderer();

		$renderer->extendedLayout['eshopCatalog.navigation.catalogSettings'] = __DIR__ . '/NavigationFormLayout.latte';
	}

	public function formSuccess(FormSuccessEvent $event): void
	{
		/** @var Navigation $navigation */
		$navigation = $event->custom['entity'];
		$values     = $event->values;
		$postData   = $this->request->getPost('eshopCatalog');

		$navigation->setParam('eshopCatalog.hideInEshopSubMenu', (int) $values->eshopCatalog->hideInEshopSubMenu);
		$navigation->setParam('eshopCatalog.hideInEshopCategoryDetail', (int) $values->eshopCatalog->hideInEshopCategoryDetail);
		$navigation->setParam('eshopCatalog.topNavigationsInCategory', $postData['topNavigationsInCategory'] ?? []);
		$navigation->setParam('eshopCatalog.topDynamicModuleInCategory', $postData['topDynamicModuleInCategory'] ?? []);
		$navigation->setParam('eshopCatalog.bottomNavigationsInCategory', $postData['bottomNavigationsInCategory'] ?? []);
		$navigation->setParam('eshopCatalog.bottomDynamicModuleInCategory', $postData['bottomDynamicModuleInCategory'] ?? []);

		$this->em->persist($navigation);
	}

	public function setNavigation(SetFormDataEvent $event): void
	{
		/** @var Navigation|null $navigation */
		$navigation = $event->entity;

		if ($navigation) {
			$event->form->setDefaults([
				'eshopCatalog' => [
					'hideInEshopSubMenu'            => (int) $navigation->getParam('eshopCatalog.hideInEshopSubMenu'),
					'hideInEshopCategoryDetail'     => (int) $navigation->getParam('eshopCatalog.hideInEshopCategoryDetail'),
					'topNavigationsInCategory'      => $navigation->getParam('eshopCatalog.topNavigationsInCategory') ?: [],
					'topDynamicModuleInCategory'    => $navigation->getParam('eshopCatalog.topDynamicModuleInCategory') ?: [],
					'bottomNavigationsInCategory'   => $navigation->getParam('eshopCatalog.bottomNavigationsInCategory') ?: [],
					'bottomDynamicModuleInCategory' => $navigation->getParam('eshopCatalog.bottomDynamicModuleInCategory') ?: [],
				],
			]);
		}
	}
}
