<?php declare(strict_types = 1);

namespace EshopCatalog\Console;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Sites;
use Doctrine;
use Doctrine\ORM\Tools\Console\Command\SchemaTool\AbstractCommand;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Categories as FrontCategories;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use Exception;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Tracy\Debugger;

class CheckCategoryProducts extends AbstractCommand
{
	/** @var Translator|null @inject */
	public ?Translator $translator = null;

	/** @var FrontCategories|null @inject */
	public ?FrontCategories $frontCategories = null;

	/** @var Categories|null @inject */
	public ?Categories $categories = null;

	/** @var ProductsFacade|null @inject */
	public ?ProductsFacade $productsFacade = null;

	/** @var Sites|null @inject */
	public ?Sites $sites = null;

	protected function configure(): void
	{
		$this->setName('eshop-catalog:check-category-products')
			->setDescription('Check category products count');
	}

	protected function executeSchemaCommand(InputInterface $input, OutputInterface $output, Doctrine\ORM\Tools\SchemaTool $schemaTool, array $metadatas, SymfonyStyle $ui): int
	{
		ini_set('memory_limit', '5G');
		$this->categories->em->getConnection()->getConfiguration()->setSQLLogger(null);
		$output->writeln('Check category products count');

		try {
			Debugger::dispatch();

			foreach ($this->sites->getSites() as $site) {
				$siteIdent = $site->getIdent();
				$lang      = $site->currentLang;

				$output->writeLn('<info>Checking - ' . $siteIdent . '</info>');

				$this->translator->setLocale($lang);
				Sites::$currentIdentOverride = $siteIdent;
				Sites::$currentLangOverride  = $lang;
				ProductsFacade::$forceLocale = $lang;

				$this->categories->checkVirtualGroup($site);

				$rootId = $this->frontCategories->getRootIdForSite($siteIdent);
				if ($rootId) {
					$categories = array_keys($this->frontCategories->getCategories($rootId));

					if (!empty($categories)) {
						$this->categories->checkProductsCountInCategories($categories);
					}
				}

				$this->productsFacade->clearTemp();
				$this->categories->em->clear();
				gc_collect_cycles();
			}

			$output->writeLn('<info>[OK] - Checked</info>');

			return 0;
		} catch (Exception $e) {
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
		}

		return 1;
	}

}
