<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Components;

use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Navigation\Alias;
use Core\Model\UI\BaseControl;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Dao\Category;
use Navigations\Model\Navigations;
use Nette\Http\Request;
use Throwable;

class Navigation extends BaseControl
{
	/** @var DaoNavigationItem[]|null */
	protected ?array       $navs            = null;
	protected string       $navigationType;
	protected Navigations  $navigationsService;
	protected Categories   $categoriesService;
	protected CacheService $cacheService;
	protected bool         $cacheCleaned    = false;
	protected array        $cCategories     = [];
	public array           $otherCategories = [];
	public array           $editedLinks     = [];
	protected Category     $currentCategory;
	protected Request      $httpRequest;

	/** @var string|int|null */
	public $currentCategoryId = null;

	public function __construct(
		Navigations  $navigations,
		Categories   $categories,
		CacheService $cacheService,
		Request      $request
	)
	{
		$this->navigationsService = $navigations;
		$this->categoriesService  = $categories;
		$this->cacheService       = $cacheService;
		$this->httpRequest        = $request;
	}

	public function render(string $template = 'Default'): void
	{
		$navs = $this->getNavs();
		$this->checkIsCurrent($navs);

		$this->template->navs              = $navs;
		$this->template->getCategories     = function() { return $this->getCategories() + $this->otherCategories; };
		$this->template->currentCategoryId = $this->currentCategoryId ?: $this->getPresenter()->getParameter('id');
		$this->template->editedLinks       = $this->editedLinks;

		$navCategory = null;
		foreach ($navs as $nav) {
			if ($nav->componentType === 'eshopCatalog.navigation.category' && (int) $nav->componentParams['category'] === (int) $this->template->currentCategoryId) {
				$navCategory = $nav;
				break;
			}
		}
		$this->template->navCategory = $navCategory;

		$this->template->render($this->getTemplateFile($template));
	}

	/*******************************************************************************************************************
	 * ============================== Get / Set
	 */

	public function getCategories(): ?array
	{
		$rootId = $this->categoriesService->getRootIdForSite();
		if (!array_key_exists($rootId, $this->cCategories)) {
			$this->cCategories[$rootId] = $this->categoriesService->getCategories($rootId) ?: [];
		}

		return $this->cCategories[$rootId];
	}

	/**
	 * @return DaoNavigationItem[]
	 * @throws Throwable
	 */
	public function getNavs(): array
	{
		if (!$this->navs) {
			$allNavs = $this->navigationsService->getPublishedToDaoNavigationItem();
			foreach ($this->navigationsService->getPublishedDaoNavigationStructure()[$this->navigationType] ?? [] as $nav) {
				/** @var $nav DaoNavigationItem */
				if ($nav->componentType === Alias::TYPE) {
					$tmp = $allNavs[$nav->componentParams['navigation'][$this->translator->getLocale()]] ?? null;
					if ($tmp) {
						$this->navs[$tmp->id] = $tmp;
						continue;
					}
				}

				$this->navs[$nav->id] = $nav;
			}
		}

		return $this->navs ?: [];
	}

	public function setNavigationType(string $type): self
	{
		$this->navigationType = $type;

		return $this;
	}

	private function checkIsCurrent(array &$data): void
	{
		$fn = function(&$data) use (&$fn) {
			foreach ($data as $nav) {
				/** @var DaoNavigationItem $nav */
				if ($nav->isHomepage) {
					continue;
				}

				$path      = explode('/', $this->httpRequest->getUrl()->path);
				$isCurrent = false;

				do {
					if (implode('/', $path) == $nav->link) {
						$isCurrent = true;
						break;
					}

					array_pop($path);
				} while (!empty($path));

				if ($isCurrent) {
					$nav->isCurrent = true;
				}

				if ($nav->childs) {
					$fn($nav->childs);

					foreach ($nav->childs as $child) {
						if ($child->isCurrent) {
							$nav->isCurrent = true;
							break;
						}
					}
				}
			}
		};

		$fn($data);
	}

}
