<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Components\Personalization;

use Core\Model\Application\AppState;
use Core\Model\UI\Form\BaseForm;
use Currency\Model\Entities\Currency;
use EshopCatalog\Model\Personalization\Personalization;
use EshopCatalog\Model\Personalization\PersonalizationAccessControl;
use EshopOrders\Model\Entities\Customer;
use Nette\Http\Url;
use Nette\Utils\ArrayHash;

class AutocompleteForm extends PersonalizationControl
{
	protected string $currency;
	protected string $currencyText;
	protected Personalization $personalization;
	protected PersonalizationAccessControl $personalizationAccessControl;

	public function __construct(
		Personalization $personalization,
		PersonalizationAccessControl $personalizationAccessControl
	)
	{
		/** @var Currency|null $currency */
		$currency = AppState::getState('currency');
		$this->currency = $currency ? $currency->getCode() : 'czk';
		$this->currencyText = $currency ? $currency->symbol : 'Kč';
		$this->personalization = $personalization;
		$this->personalizationAccessControl = $personalizationAccessControl;
	}

	public function render(): void
	{
		$searchUrl = new Url($this->getPresenter()->link(':EshopCatalog:Front:Default:search'));
		$searchUrl->setQueryParameter('q', '__VAL__');
		$searchUrl->setQueryParameter('productListingFilter-filter[fc][0]', '__CAT_VAL__');
		$searchUrl->setQueryParameter('do', 'productListingFilter-set');

		$this->template->currencyText = $this->currencyText;
		$this->template->apiUrl = $this->apiUrl('__VAL__');
		$this->template->searchUrl = '/' . $searchUrl->relativeUrl;
		$this->template->requestAutocancelAllowed = $this->personalizationAccessControl->hasAutocompleteRequestAutocancelAllowed();
		$this->template->render($this->getTemplateFile());
	}

	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAction($this->getPresenter()->link(':EshopCatalog:Front:Default:search'));
		$form->setMethod('post');
		$form->getElementPrototype();
		$form->setShowLangSwitcher(false)->setHtmlAttribute('id', 'autocomplete-form');

		$form->addText('q');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		$this->getPresenter()->redirect(':EshopCatalog:Front:Default:search', ['q' => $values->q]);
	}

	public function isAllowed(): bool
	{
		return $this->personalizationAccessControl->isAutocompleteAllowed();
	}

	protected function apiUrl(string $query): Url
	{
		/** @var Customer|null $customer */
		$customer = $this->template->customer;

		return $this->personalization->getAutocompleteUrl($query, $customer);
	}

}