<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Components;

use Core\Model\Application\AppState;
use Core\Model\UI\BaseControl;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\FrontModule\Model\Tags;
use EshopCatalog\Model\Config;
use Nette\Application\UI\Multiplier;
use Nette\Utils\Paginator;
use Throwable;

class ProductsList extends BaseControl
{
	protected Config   $config;
	protected Products $productsService;
	protected array    $categories = [];

	/** @var Product[]|null */
	protected ?array  $products    = null;
	protected int     $allProductsCount;
	protected ?string $searchQuery = null;
	public int        $page        = 1;

	/** @var int|null @persistent */
	public ?int $limit = null;

	public ?int $overrideLimit    = null;
	public int  $start            = 0;
	public bool $paginatorEnabled = true;

	/** @var ProductsFilter|null */
	public $filterComponent;

	protected Tags                   $tagsService;
	protected Paginator              $paginator;
	protected IProductPreviewFactory $productPreviewFactory;
	protected ProductsFacade         $productsFacade;

	public function __construct(Config $config, Products $products, Tags $tags, IProductPreviewFactory $productPreviewFactory, ProductsFacade $productsFacade)
	{
		$this->config                = $config;
		$this->productsService       = $products;
		$this->tagsService           = $tags;
		$this->productPreviewFactory = $productPreviewFactory;
		$this->productsFacade        = $productsFacade;
	}

	public function render(): void
	{
		$this->template->paginator       = $this->getPaginator();
		$this->template->limit           = $this->getLimit();
		$this->template->start           = $this->start;
		$this->template->products        = $this->getProducts();
		$this->template->count           = $this->getCount();
		$this->template->filterComponent = $this->filterComponent;
		$this->template->sortLink        = function($a) {
			return $this->filterComponent
				? $this->filterComponent->createSortLink($a)
				: '#';
		};

		if (Config::load('tagsFilterUnderSort')) {
			$this->template->tagLink = function($a) {
				return $this->filterComponent
					? $this->filterComponent->createLink('tag', '0', (string) $a)
					: '#';
			};

			if ($this->filterComponent) {
				$this->template->tagFilters = $this->filterComponent->applyFilters()['tag_0'] ?? [];
			} else {
				$this->template->tagFilters = [];
			}
		}

		$currentSort                 = $this->filterComponent ? $this->filterComponent->sort ?: 'recommended' : null;
		$this->template->currentSort = $currentSort;
		AppState::setState('productsList.currentSort', $currentSort);

		$this->template->sortValues = $this->filterComponent ? $this->filterComponent->getSortValues() : [];

		$list = [];
		foreach (Config::load('productsList.itemsPerPageList') as $v) {
			$list[$v] = $this->link('limit!', $v);
		}
		$this->template->listLimits = $list;

		$this->template->render($this->getTemplateFile(null, 'ProductsList'));
		AppState::setState('productsList.currentSort', null);
	}

	public function handlePaginator(int $page): void
	{
		$this->page  = $page;
		$this->start = $page > 1 ? $this->getLimit() * ($page - 1) : 0;

		if ($this->getPresenter()->isAjax()) {
			$this->redrawControl('list');
			$this->redrawControl('paginator');
		}
	}

	public function handleLimit(int $limit): void
	{
		if ($limit == (int) Config::load('productsList.itemsPerPage') || $this->overrideLimit && $limit == $this->overrideLimit) {
			$this->limit = null;
		} else {
			$this->limit = $limit;
		}
	}

	public function getPageLink(int $page): string
	{
		return $this->presenter->link('this', [
			$this->getName() . '-page' => $page,
			'do'                       => $this->getName() . '-paginator',
		]);
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentProductPreview(): Multiplier
	{
		return new Multiplier(function(string $id): ProductPreview {
			$control = $this->productPreviewFactory->create();
			$id      = (int) $id;

			$control->showInStockVariant = true;

			$product = $this->getProducts()[$id] ?? null;
			if ($product) {
				$control->setProduct($product);
			} else {
				$control->setProductById($id);
			}

			return $control;
		});
	}

	/*******************************************************************************************************************
	 * ============================== Get / Set
	 */

	public function getPaginator(): Paginator
	{
		$itemsPerPage = $this->getLimit();

		if ($this->page === 1) {
			$pathParts      = explode('/', $_SERVER['REQUEST_URI']);
			$paginatorIndex = array_search($this->translator->translate('default.urlPart.page'), $pathParts);

			if ($paginatorIndex !== false) {
				$this->page  = (int) $pathParts[$paginatorIndex + 1];
				$this->start = $this->getLimit() * ($this->page - 1);
			}
		}

		$paginator = new Paginator();
		$paginator->setItemCount($this->getCount());
		$paginator->setItemsPerPage($itemsPerPage);
		$paginator->setPage($this->page);

		return $paginator;
	}

	/**
	 * @return Product[]|null
	 * @throws Throwable
	 */
	public function getProducts(): ?array
	{
		if ($this->products === null) {
			$this->getPaginator();

			$products = $this->filterComponent ? $this->filterComponent->getProductIdsAfterFilter() : [];
			$products = array_slice($products, $this->start, $this->getLimit(), true);

			$products       = $this->productsFacade->getProducts($products);
			$this->products = $products;

			if (empty($this->products)) {
				if (!empty($this->categories)) {
					/** @phpstan-ignore-next-line */
					if (empty($filter)) {
						$this->template->notFoundText = $this->t('eshopCatalogFront.productsList.productsNotFoundCategory');
					} else {
						$this->template->notFoundText = $this->t('eshopCatalogFront.productsList.productsNotFoundFilter');
					}
				} else if ($this->searchQuery) {
					$this->template->notFoundText = $this->t('eshopCatalogFront.productsList.productsNotFoundSearch');
				}
			}

			if ($this->products) {
				$this->productsService->loadQuantity($this->products);
			}
		}

		return $this->products;
	}

	protected function getCount(): int
	{
		return $this->filterComponent ? count($this->filterComponent->getProductIdsAfterFilter()) : 0;
	}

	public function setCategories(array $categories): self
	{
		$this->categories = $categories;

		return $this;
	}

	/**
	 * @param ProductsFilter $component
	 */
	public function setFilterComponent($component): self
	{
		$this->filterComponent = $component;
		if (!empty($this->categories)) {
			$this->filterComponent->setCategories($this->categories);
		} else if ($this->searchQuery) {
			$this->filterComponent->setSearchQuery($this->searchQuery);
		}

		return $this;
	}

	public function setProductsBySearch(string $q): self
	{
		$this->searchQuery = $q;

		return $this;
	}

	protected function getLimit(): int
	{
		if ($this->overrideLimit) {
			return $this->overrideLimit;
		}

		return $this->limit ?: (int) Config::load('productsList.itemsPerPage');
	}

}
