<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use EshopCatalog\FrontModule\Model\Dao\FeatureProduct;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\DynamicFeatureProduct;
use EshopCatalog\Model\Entities\Feature;
use Nette\Caching\Cache;

class DynamicFeaturesProducts extends BaseFrontEntityService
{
	public const CACHE_NAMESPACE = 'dynamicFeaturesProducts';

	protected              $entityClass = DynamicFeatureProduct::class;
	protected CacheService $cacheService;
	protected Features     $features;

	protected array $cFeatures = [];

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS   => ['dynamicFeaturesProducts'],
		Cache::EXPIRE => '1 week',
	];

	public function __construct(
		CacheService $cacheService,
		Features     $features
	)
	{
		$this->cacheService = $cacheService;
		$this->features     = $features;
	}

	public function getCache(): Cache
	{
		if ($this->cache === null) {
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);
		}

		return $this->cache;
	}

	public function getDaoByProducts(array $ids): array
	{
		if (empty($ids)) {
			return [];
		}
		$result   = [];
		$whereIds = [];
		$keys     = [];
		$lang = $this->translator->getLocale();

		foreach ($ids as $id) {
			if (isset($this->cFeatures[$id])) {
				$result[$id] = $this->cFeatures[$id];
			} else {
				$keys[] = 'dynamicFeature/' . $id . '/' . $lang;
			}
		}

		if (!empty($keys)) {
			foreach ($this->cacheService->productCache->bulkLoad($keys) as $key => $data) {
				$tmp = explode('/', $key);
				$id  = (int) $tmp[1];

				if ($data || is_array($data)) {
					$result[$id]          = $data;
					$this->cFeatures[$id] = $data;
				} else {
					$whereIds[] = $id;
				}
			}
		}

		if (!empty($whereIds)) {
			$toSave = [];

			foreach ($this->em->getConnection()->fetchAllAssociative("SELECT dfp.id, dfp.id_product, dfp.id_feature, dfp.value
					FROM eshop_catalog__dynamic_feature_product dfp
					INNER JOIN eshop_catalog__feature f ON f.id = dfp.id_feature
					WHERE dfp.id_product IN (" . implode(',', $ids) . ")") as $row) {
				$feature = $this->features->getFeatureById((int) $row['id_feature']);
				if (!$feature) {
					continue;
				}

				$unit = $feature['unit'];
				if ($unit === 'month') {
					$unit = $this->translator->translate('eshopCatalogFront.unit.month');
				}

				$dao                          = new FeatureProduct;
				$dao->idFeature               = $feature['id'];
				$dao->name                    = $feature['name'];
				$dao->position                = $feature['position'];
				$dao->showInProduct           = (int) $feature['showInProduct'];
				$dao->showInExport            = (int) $feature['showInExport'];
				$dao->showFilterLinkInProduct = (int) $feature['showFilterLinkInProduct'];
				$dao->value                   = $row['value'] . ($unit ? ' ' . $unit : '');
				$dao->rawValue                = $row['value'];
				$dao->unit                    = $row['unit'];
				$dao->useForVariantDiff       = (int) $feature['useForVariantDiff'];
				$dao->type                    = Feature::TYPE_RANGE;

				if (Config::load('features.allowFeatureImage')) {
					$dao->featureImage = $feature['image'] ?: null;
				}

				$toSave[$row['id_product']][$row['id_feature']][$row['id']] = $dao;
				$result[$row['id_product']][$row['id_feature']][$row['id']] = $dao;
			}

			foreach ($whereIds as $id) {
				$this->cFeatures[$id] = $toSave[$id] ?? [];
				$this->cacheService->productCache->save('dynamicFeature/' . $id . '/' . $lang, $toSave[$id] ?? [], [
					Cache::EXPIRE => '1 week',
				]);
			}
		}

		return $result;
	}

}
