<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureCategory;
use EshopCatalog\Model\Entities\FeatureProduct;
use Navigations\Model\Navigations;
use Nette\Application\LinkGenerator;
use Nette\Caching\Cache;
use Throwable;

class FeatureProducts extends BaseFrontEntityService
{
	protected $entityClass = FeatureProduct::class;

	protected Features     $features;
	public ?LinkGenerator  $linkGenerator = null;
	public ?Navigations    $navigations   = null;
	protected CacheService $cacheService;

	protected array $cProductFeatures      = [];
	protected array $cFeatureValuesIds     = [];
	protected array $cFeatureValuesIdsCats = [];
	protected array $cacheDep              = [
		Cache::TAGS   => ['featureProduct'],
		Cache::EXPIRE => '1 week',
	];

	public function __construct(
		Features     $features,
		CacheService $cacheService
	)
	{
		$this->features     = $features;
		$this->cacheService = $cacheService;
	}

	public function getFeatureValuesIdsForProducts(array $ids): array
	{
		$whereIds = [];
		$result   = [];

		foreach ($ids as $id) {
			if (isset($this->cFeatureValuesIds[$id])) {
				$result[$id] = $this->cFeatureValuesIds[$id];
			} else {
				$whereIds[]                   = $id;
				$this->cFeatureValuesIds[$id] = [];
				$result[$id]                  = [];
			}
		}

		if (!empty($whereIds)) {
			foreach (array_chunk($whereIds, 900) as $chunk) {
				foreach ($this->getEr()->createQueryBuilder('fp')
					         ->select('IDENTITY(fp.product) as product, IDENTITY(fp.featureValue) as value')
					         ->andWhere('fp.product IN (' . implode(',', $chunk) . ')')
					         ->getQuery()->getScalarResult() as $row) {
					$this->cFeatureValuesIds[$row['product']][] = $row['value'];
					$result[$row['product']][]                  = $row['value'];
				}
			}
		}

		return $result;
	}

	public function getFeatureValuesIdsForCategories(array $ids): array
	{
		if (!$ids) {
			return [];
		}

		$whereIds = [];
		$result   = [];

		$keys = [];
		foreach ($ids as $id) {
			$keys[] = 'catFeatures/' . $id;
		}

		foreach ($this->cacheService->categoryCache->bulkLoad($keys) as $key => $value) {
			$tmp = explode('/', $key);
			$id  = (int) $tmp[1];

			if ($value !== null) {
				$result[$id] = $value;
			} else {
				$whereIds[]  = $id;
				$result[$id] = [];
			}
		}

		if (!empty($whereIds)) {
			foreach (array_chunk($whereIds, 900) as $chunk) {
				foreach ($this->em->getRepository(FeatureCategory::class)->createQueryBuilder('fc')
					         ->select('IDENTITY(fc.category) as category, IDENTITY(fc.featureValue) as value')
					         ->andWhere('fc.category IN (' . implode(',', $chunk) . ')')
					         ->getQuery()->getScalarResult() as $row) {
					$this->cFeatureValuesIdsCats[$row['category']][] = $row['value'];
					$result[$row['category']][]                      = $row['value'];
				}
			}

			foreach ($whereIds as $id) {
				$this->cFeatureValuesIdsCats[$id] = $result[$id] ?? [];
				$this->cacheService->categoryCache->save('catFeatures/' . $id, $result[$id] ?? [], $this->cacheDep);
			}
		}

		return $result;
	}

	/**
	 * @param Dao\Product[] $products
	 *
	 * @return Dao\FeatureProduct[][]
	 * @throws Throwable
	 */
	public function getFeaturesForProduct(array $products): array
	{
		/** @var Dao\Product[] $whereIds */
		$whereIds   = [];
		$result     = [];
		$categories = [];

		foreach ($products as $product) {
			if ($product instanceof Dao\Product === false) {
				continue;
			}

			$id = $product->getId();

			if (isset($this->cProductFeatures[$id])) {
				$result[$id] = $this->cProductFeatures[$id];
			} else {
				$whereIds[] = $product;
			}
		}

		if (!empty($whereIds)) {
			$data = [];

			foreach ($whereIds as $product) {
				if ($product->defaultCategoryId) {
					$categories[$product->defaultCategoryId] = $product->defaultCategoryId;
				}
			}

			if (!empty($categories)) {
				$categories = $this->getFeatureValuesIdsForCategories($categories);
			}

			foreach ($whereIds as $product) {
				if ($product->defaultCategoryId && isset($categories[$product->defaultCategoryId])) {
					foreach ($categories[$product->defaultCategoryId] as $valueId) {
						$value = $this->features->getFeatureValueById($valueId);
						if (!$value) {
							continue;
						}
						$feature = $this->features->getFeatureById($value['featureId']);
						if (!$feature) {
							continue;
						}

						$data[$product->getId()][$valueId] = $this->fillDao($product->getId(), $feature, $value);
					}
				}

				foreach ($product->featureValuesIds as $valueId) {
					$value = $this->features->getFeatureValueById($valueId);
					if (!$value) {
						continue;
					}
					$feature = $this->features->getFeatureById($value['featureId']);
					if (!$feature) {
						continue;
					}

					$data[$product->getId()][$valueId] = $this->fillDao($product->getId(), $feature, $value);
				}
			}

			foreach ($data as $productId => $features) {
				uasort($features, static fn($a, $b) => $a->position <=> $b->position);
				$result[$productId]                 = $features;
				$this->cProductFeatures[$productId] = $features;
			}
		}

		return $result;
	}

	protected function fillDao(int $productId, array $feature, array $value): Dao\FeatureProduct
	{
		$fp                          = new Dao\FeatureProduct;
		$fp->idProduct               = $productId;
		$fp->idFeature               = $feature['id'];
		$fp->position                = $feature['position'];
		$fp->name                    = $feature['name'];
		$fp->idFeatureValue          = $value['id'];
		$fp->valuePosition           = $value['position'];
		$fp->value                   = $value['name'] . ($feature['unit'] ? ' ' . $feature['unit'] : '');
		$fp->rawValue                = (string) $value['name'];
		$fp->unit                    = $feature['unit'];
		$fp->useForVariantDiff       = (int) $feature['useForVariantDiff'];
		$fp->showInProduct           = (int) $feature['showInProduct'];
		$fp->showInExport            = (int) $feature['showInExport'];
		$fp->showFilterLinkInProduct = (int) $feature['showFilterLinkInProduct'] || (int) $value['showFilterLinkInProduct'] ? 1 : 0;
		$fp->valueType               = $feature['valueType'] ?: Feature::VALUE_TYPE_TEXT;
		$fp->seoNoFollow             = $value['seoNoFollow'] !== null ? (int) $value['seoNoFollow'] : (int) $feature['seoNoFollow'];

		if (isset($value['extraFields'])) {
			$fp->setExtraFields($value['extraFields']);
		}

		if (Config::load('features.allowImage')) {
			$fp->image = $value['image'];
		}

		if (Config::load('features.allowFeatureImage')) {
			$fp->featureImage = $feature['image'] ?: null;
		}

		if (Config::load('features.allowDescription')) {
			$fp->productTabTitle  = $feature['productTabTitle'];
			$fp->showAsTag        = (bool) $value['showAsTag'];
			$fp->tagText          = $value['tagText'];
			$fp->shortDescription = $value['shortDescription'];
			$fp->longDescription  = $value['longDescription'];
			$fp->tagTextColor     = $value['tagTextColor'];
			$fp->tagBgColor       = $value['tagBgColor'];

			if ($value['moreLink']) {
				$v = explode('|', $value['moreLink']);
				if ($v[0] === 'text') {
					$fp->moreLink = $v[1];
				} else if ($v[0] === 'navigation') {
					if ($nav = $this->navigations->getNavigation((int) $v[1])) {
						if (!is_array($nav->link)) {
							$fp->moreLink = (string) $nav->link;
						}
					}
				} else if ($v[0] === 'article') {
					$fp->moreLink = $this->linkGenerator->link('Blog:Front:Articles:detail', ['id' => $v[1]]);
				}
			}
		}

		if ($fp->valueType === Feature::VALUE_TYPE_COLOR && !empty($value['value1'])) {
			$fp->extraFields += $this->features->featuresHelper->parseValueColors((string) $value['value1']);
		}

		return $fp;
	}

}
