<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Closure;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\Strings;
use Doctrine\ORM\EntityRepository;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\QueryBuilder;
use EshopCatalog\Model\Config;
use Nette\Utils\DateTime;

class ProductQuery
{
	/** @var Closure[] */
	private array $filter = [];

	/** @var Closure[] */
	private array $select = [];

	protected string $lang;
	public ?string   $siteIdent         = null;
	public bool      $disableAutoIdSort = false;
	public bool      $disableIdGroup    = false;

	public static bool $ignoreProductPublish = false;

	public function __construct(string $lang)
	{
		$this->lang = $lang;
	}

	public function skipId(int $id): self
	{
		$this->filter[] = static function(QueryBuilder $qb) use ($id) {
			$qb->andWhere('p.id != :notId')->setParameter('notId', $id);
		};

		return $this;
	}

	public function withTexts(bool $addSelect = true): self
	{
		$this->filter[] = function(QueryBuilder $qb) {
			if (Arrays::contains($qb->getAllAliases(), 'pt')) {
				$qb->andWhere('pt.lang = :lang' . (Arrays::contains(Config::load('product.hideIfTitleMissing', []), $this->lang) ? ' AND pt.name != \'\'' : ''))
					->setParameter('lang', $this->lang);
			} else {
				$qb->innerJoin('p.productTexts', 'pt', 'WITH',
					'pt.lang = :lang' . (Arrays::contains(Config::load('product.hideIfTitleMissing', []), $this->lang) ? ' AND pt.name != \'\'' : ''))
					->setParameter('lang', $this->lang);
			}
		};
		if ($addSelect) {
			$this->select[] = static function(QueryBuilder $qb) {
				$qb->addSelect('pt');
			};
		}

		return $this;
	}

	public function withCategories(): self
	{
		$this->select[] = static function(QueryBuilder $qb) {
			$qb->addSelect('cp')
				->leftJoin('p.categoryProducts', 'cp');
		};

		return $this;
	}

	public function withVatRate(): self
	{
		$this->select[] = static function(QueryBuilder $qb) {
			$qb->addSelect('vr.rate as vatRate')
				->leftJoin('p.vatRate', 'vr');
		};

		return $this;
	}

	public function withVariantParent(): self
	{
		$this->select[] = function(QueryBuilder $qb) {
			$qb->leftJoin('p.isVariant', 'variants')
				->addSelect('variants.variantId as variantId')
				->addSelect('variants.defaultImage as variantImage')
				->leftJoin('variants.texts', 'variantsText', Join::WITH, 'variantsText.lang = :lang')
				->setParameter('lang', $this->lang)
				->addSelect('variantsText.name as variantName');
		};

		return $this;
	}

	public function withTags(): self
	{
		$this->select[] = static function(QueryBuilder $qb) {
			$qb->leftJoin('p.productTags', 'ptags');
		};

		return $this;
	}

	public function withTag(int $tag): self
	{
		$this->filter[] = static function(QueryBuilder $qb) use ($tag) {
			$now = new DateTime();
			$now->setTime((int) $now->format("H"), (int) $now->format("i"), 0);
			$tagKey = (string) $tag;
			$qb->innerJoin('p.productTags', 'pTags', 'WITH', 'pTags.tag = :pTagId' . $tagKey . ' AND (pTags.validFrom <= :pTagFrom' . $tagKey . ' OR pTags.validFrom IS NULL) AND (pTags.validTo >= :pTagTo' . $tagKey . ' OR pTags.validTo IS NULL)')
				->setParameter('pTagFrom' . $tagKey, $now)
				->setParameter('pTagTo' . $tagKey, $now)
				->setParameter('pTagId' . $tagKey, $tag)
				->groupBy('p.id');
		};

		return $this;
	}

	public function withoutTag(string $tag): self
	{
		$this->filter[] = static function(QueryBuilder $qb) use ($tag) {
			$qb->leftJoin('p.productTags', 'pTags')
				->leftJoin('pTags.tag', 'tags')
				->andWhere('pTags IS NULL OR tags.type != :withoutTag')->setParameter('withoutTag', $tag)
				->groupBy('p.id');
		};

		return $this;
	}

	public function hasPrice(): self
	{
		$this->filter[] = static function(QueryBuilder $qb) {
			$qb->andWhere('p.price IS NOT NULL');
		};

		return $this;
	}

	public function loadSites(): self
	{
		$this->select[] = static function(QueryBuilder $qb) {
			$qb->addSelect('GROUP_CONCAT(IDENTITY(sites.site)) as inSites')
				->leftJoin('p.sites', 'sites', Join::WITH, 'sites.isActive = 1');
		};

		return $this;
	}

	public function inSite(string $siteIdent): self
	{
		$this->filter[] = static function(QueryBuilder $qb) use ($siteIdent) {
			if (in_array('sites', $qb->getAllAliases())) {
				$qb->andWhere('sites.isActive = 1')
					->andWhere('sites.site = :siteIdent')
					->setParameter('siteIdent', $siteIdent);
			} else {
				$qb->innerJoin('p.sites', 'sites', 'WITH', 'sites.site = :site AND sites.isActive = 1')
					->setParameter('site', $siteIdent);
			}
		};

		return $this;
	}

	public function disableListing(?int $disableListing = 0): self
	{
		$this->filter[] = static function(QueryBuilder $qb) use ($disableListing) {
			$qb->andWhere('p.disableListing = :disableListing')
				->setParameter('disableListing', $disableListing);
		};

		return $this;
	}

	/**
	 * @param int|string|int[]|string[] $id
	 */
	public function inCategory($id): self
	{
		$this->filter[] = static function(QueryBuilder $qb) use ($id) {
			// Pokud chceme hledat pro eshop, je potreba nejdriv zavolat inSite()
			if (!in_array('sites', $qb->getAllAliases())) {
				$qb->leftJoin('p.sites', 'sites', Join::WITH, 'sites.isActive = 1');
			}

			if (is_array($id) && count($id) === 1) {
				$id = $id[0];
			}

			if (is_array($id)) {
				$qb->andWhere('cp.category IN (:categoryId) OR sites.category IN (:categoryId)');
			} else {
				$qb->andWhere('cp.category = :categoryId OR sites.category = :categoryId');
			}
			$qb->setParameter('categoryId', $id)
				->leftJoin('p.categoryProducts', 'cp');
		};

		return $this;
	}

	/**
	 * @param int|string|int[]|string[] $id
	 */
	public function inSiteCategory($id): self
	{
		$this->filter[] = function(QueryBuilder $qb) use ($id) {
			// Pokud chceme hledat pro eshop, je potreba nejdriv zavolat inSite()
			if (!in_array('sites', $qb->getAllAliases())) {
				$qb->leftJoin('p.sites', 'sites', Join::WITH, 'sites.isActive = 1');
			}

			if (is_array($id) && count($id) === 1) {
				$id = $id[0];
			}

			if (is_array($id)) {
				$qb->andWhere('sites.category IN (:categoryId)');
			} else {
				$qb->andWhere('sites.category = :categoryId');
			}
			$qb->setParameter('categoryId', $id);

			if (Config::load('category.publishedByLang')) {
				if (!in_array('sitesCat', $qb->getAllAliases())) {
					$qb->innerJoin('sites.category', 'sitesCat');
				}

				$qb->innerJoin('sitesCat.categoryTexts', 'sitesCatTexts', Join::WITH, 'sitesCatTexts.lang = :lang AND sitesCatTexts.isPublished = 1')
					->setParameter('lang', $this->lang);
			} else {
				if (in_array('sitesCat', $qb->getAllAliases())) {
					$qb->andWhere('sitesCat.isPublished = 1');
				} else {
					$qb->innerJoin('sites.category', 'sitesCat', Join::WITH, 'sitesCat.isPublished = 1');
				}
			}
		};

		return $this;
	}

	/**
	 * @param int|string|int[]|string[] $id
	 */
	public function inOtherCategories($id): self
	{
		$this->filter[] = function(QueryBuilder $qb) use ($id) {
			// Pokud chceme hledat pro eshop, je potreba nejdriv zavolat inSite()

			if (is_array($id) && count($id) === 1) {
				$id = $id[0];
			}

			$qb->innerJoin('p.categoryProducts', 'cp', Join::WITH, is_array($id) ? 'cp.category IN (:categoryId)' : 'cp.category = :categoryId')
				->setParameter('categoryId', $id);

			if (Config::load('category.publishedByLang')) {
				$qb->innerJoin('cp.category', 'cpCat')
					->innerJoin('cpCat.categoryTexts', 'cpCatTexts', Join::WITH, 'cpCatTexts.lang = :lang AND cpCatTexts.isPublished = 1')
					->setParameter('lang', $this->lang);
			} else {
				$qb->innerJoin('cp.category', 'cpCat', Join::WITH, 'cpCat.isPublished = 1');
			}
		};

		return $this;
	}

	public function search(string $query): self
	{
		$this->withTexts(false);

		$this->filter[] = function(QueryBuilder $qb) use ($query) {
			$qb->leftJoin('p.manufacturer', 'manu');

			$searchIn   = Config::load('searchIn');
			$searchType = Config::load('searchType');

			if ($searchIn['explodeQuery'] !== null) {
				$originQuery = $query;
				$query       = explode($searchIn['explodeQuery'], $query);

				array_unshift($query, $originQuery);
			} else {
				$query = [$query];
			}

			if ($searchType === 'orAnd') {
				$criteria = [
					'name'             => [],
					'code1'            => [],
					'ean'              => [],
					'code2'            => [],
					'name2'            => [],
					'shortDescription' => [],
					'description'      => [],
				];

				foreach ($query as $k => $word) {
					$word = trim((string) $word, ',. ');

					if (!is_numeric($word) && strlen($word) < 2) {
						continue;
					}

					$criteria['name'][] = "CONCAT(
					CASE WHEN manu IS NOT NULL THEN manu.name ELSE 0 END,
					' ', pt.name)
					LIKE :search_" . $k;

					$criteria['code1'][] = 'p.code1 LIKE :search_' . $k;
					$criteria['ean'][]   = 'p.ean LIKE :search_' . $k;

					if ($searchIn['code2']) {
						$criteria['code2'][] = 'p.code2 LIKE :search_' . $k;
					}
					if ($searchIn['name2']) {
						$criteria['name2'][] = 'pt.name2 LIKE :search_' . $k;
					}
					if ($searchIn['shortDescription']) {
						$criteria['shortDescription'][] = 'pt.shortDescription LIKE :search_' . $k;
					}
					if ($searchIn['description']) {
						$criteria['description'][] = 'pt.description LIKE :search_' . $k;
					}

					$qb->setParameter('search_' . $k, Strings::lower("%$word%"));
				}

				$where = [];
				foreach ($criteria as $v) {
					if (empty($v)) {
						continue;
					}

					$where[] = '(' . implode(' AND ', $v) . ')';
				}

				if (!empty($where)) {
					$qb->andWhere(implode(' OR ', $where));
				}
			} else {
				foreach ($query as $k => $word) {
					$word = trim((string) $word, ',. ');

					if (!is_numeric($word) && strlen($word) < 2) {
						continue;
					}

					$qb->andWhere("CONCAT(
									CASE WHEN manu IS NOT NULL THEN manu.name ELSE 0 END,
									' ', pt.name)
									LIKE :search_" . $k);
					$qb->orWhere('p.code1 LIKE :search_' . $k)
						->orWhere('p.ean LIKE :search_' . $k);

					if ($searchIn['code2']) {
						$qb->orWhere('p.code2 LIKE :search_' . $k);
					}
					if ($searchIn['name2']) {
						$qb->orWhere('pt.name2 LIKE :search_' . $k);
					}
					if ($searchIn['shortDescription']) {
						$qb->orWhere('pt.shortDescription LIKE :search_' . $k);
					}
					if ($searchIn['description']) {
						$qb->orWhere('pt.description LIKE :search_' . $k);
					}

					$qb->setParameter('search_' . $k, Strings::lower("%$word%"));
				}
			}

			if ($this->siteIdent) {
				if (!in_array('sites', $qb->getAllAliases())) {
					$qb->innerJoin('p.sites', 'sites', Join::WITH, 'sites.site = :site')
						->setParameter('site', $this->siteIdent);
				}

				if (Config::load('category.publishedByLang')) {
					if (!in_array('siteCat', $qb->getAllAliases())) {
						$qb->innerJoin('sites.category', 'siteCat', Join::WITH, 'siteCat.isPublished = 1');
						$qb->innerJoin('siteCat.categoryTexts', 'siteCatTexts', Join::WITH, 'siteCatTexts.lang = :lang AND siteCatTexts.isPublished = 1')
							->setParameter('lang', $this->lang);
					} else {
						$qb->andWhere('siteCat.isPublished = 1');
					}
				} else {
					if (!in_array('siteCat', $qb->getAllAliases())) {
						$qb->innerJoin('sites.category', 'siteCat', Join::WITH, 'siteCat.isPublished = 1');
					} else {
						$qb->andWhere('siteCat.isPublished = 1');
					}
				}
			}
		};

		return $this;
	}

	public function onlyInStockOrSupplier(?array $pvsIds = null): self
	{
		$this->filter[] = static function(QueryBuilder $qb) use ($pvsIds) {
			$qb->innerJoin('p.availability', 'av', Join::WITH, 'av.canShowOnList = 1');
			$arr = [];
			if ($pvsIds) {
				$arr[] = 'p.id IN (' . implode(',', $pvsIds) . ')';
			}

			if (!empty($arr)) {
				$qb->andWhere(implode(' OR ', $arr));
			}
		};

		return $this;
	}

	public function availableInSearch(): self
	{
		$this->select[] = static function(QueryBuilder $qb) {
			$qb->innerJoin('p.availability', 'av', Join::WITH, 'av.canShowOnSearch = 1');
		};

		return $this;
	}

	public function selectInStockBool(): self
	{
		$this->select[] = static function(QueryBuilder $qb) {
			if (!in_array('ps', $qb->getAllAliases())) {
				$qb->leftJoin('p.suppliers', 'ps');
			}
			$qb->addSelect('CASE WHEN p.quantity > 0 THEN 1 WHEN ps.quantity > 0 THEN 1 ELSE 0 END as hidden inStockBool');
		};

		return $this;
	}

	public function useFilter(array $filter): self
	{
		if (isset($filter['features'])) {
			$this->filterFeatures($filter['features']);
		}
		if (isset($filter['priceRange']['min'])) {
			$this->filterPriceMin($filter['priceRange']['min']);
		}
		if (isset($filter['priceRange']['max'])) {
			$this->filterPriceMax($filter['priceRange']['max']);
		}
		if (isset($filter['manu'])) {
			$this->filterManu($filter['manu']);
		}

		return $this;
	}

	public function countryPriceFilled(string $country): self
	{
		$this->filter[] = static function(QueryBuilder $qb) use ($country) {
			if (!Arrays::contains($qb->getAllAliases(), 'countryPrice')) {
				$qb->innerJoin('p.prices', 'countryPrice', Join::WITH, 'countryPrice.country = :cpCountry AND countryPrice.price IS NOT NULL');
			} else {
				$qb->andWhere('countryPrice.country = :cpCountry AND countryPrice.price IS NOT NULL');
			}
			$qb->setParameter('cpCountry', $country);
		};

		return $this;
	}

	/**
	 * @param int|string $min
	 */
	public function filterPriceMin($min): self
	{
		$this->filter[] = static function(QueryBuilder $qb) use ($min) {
			$qb->andWhere('p.price >= :priceMin')->setParameter('priceMin', $min);
		};

		return $this;
	}

	/**
	 * @param int|string $max
	 */
	public function filterPriceMax($max): self
	{
		$this->filter[] = static function(QueryBuilder $qb) use ($max) {
			$qb->andWhere('p.price <= :priceMax')->setParameter('priceMax', $max);
		};

		return $this;
	}

	public function filterFeatures(array $ids): self
	{
		if ($ids) {
			$this->filter[] = static function(QueryBuilder $qb) use ($ids) {
				$qb->andWhere('p.id IN (:featureP)')->setParameter('featureP', $ids);
			};
		}

		return $this;
	}

	/**
	 * @param int|string|int[]|string[] $ids
	 */
	public function filterManu($ids): self
	{
		if ($ids) {
			$this->filter[] = static function(QueryBuilder $qb) use ($ids) {
				$qb->andWhere('p.manufacturer ' . (is_array($ids) ? 'IN (:ids)' : '= :ids'))->setParameter('ids', $ids);
			};
		}

		return $this;
	}

	public function selectIds(): self
	{
		$this->select[] = static function(QueryBuilder $qb) {
			$qb->select('p.id');
		};

		if (!$this->disableIdGroup) {
			$this->select[] = static function(QueryBuilder $qb) {
				$qb->groupBy('p.id');
			};
		}

		return $this;
	}

	public function selectQuantity(): self
	{
		$this->filter[] = static function(QueryBuilder $qb) {
			$qb->leftJoin('p.suppliers', 'ps', Join::WITH, 'ps.isActive = 1');
		};
		$this->select[] = static function(QueryBuilder $qb) {
			$qb->select('p.id, p.quantity as productQ, GROUP_CONCAT(ps.quantity) as suppliersQ, GROUP_CONCAT(IDENTITY(ps.availabilityAfterSoldOut)) as suppliersA');
		};

		return $this;
	}

	/**
	 * @param int|int[] $id
	 */
	public function byId($id): self
	{
		$this->select[] = static function(QueryBuilder $qb) use ($id) {
			if (is_array($id)) {
				$qb->andWhere('p.id IN (:id)');
			} else {
				$qb->andWhere('p.id = :id');
			}
			$qb->setParameter('id', $id);
		};

		return $this;
	}

	public function orderByPosition(): self
	{
		$this->select[] = static function(QueryBuilder $qb) {
			$qb->addSelect('CASE WHEN p.position IS NULL THEN 999999999 ELSE p.position END as cPosition')
				->addOrderBy('cPosition');
		};

		return $this;
	}

	public function sortBy(?string $sort = null, ?string $direction = null): self
	{
		if ($sort) {
			$this->select[] = function(QueryBuilder $qb) use ($sort, $direction) {
				if (is_string($sort) && strpos($sort, 'pt') === 0 && !in_array('pt', $qb->getAllAliases())) {
					$qb->innerJoin('p.productTexts', 'pt', 'WITH', 'pt.lang = :lang')
						->setParameter('lang', $this->lang);
				}
				$qb->orderBy($sort, $direction);
			};
		}

		return $this;
	}

	public function addSortBy(?string $sort = null, ?string $direction = null): self
	{
		if ($sort) {
			$this->select[] = static function(QueryBuilder $qb) use ($sort, $direction) {
				$qb->addOrderBy($sort, $direction);
			};
		}

		return $this;
	}

	public function disableAutoIdSort(bool $val = true): self
	{
		$this->disableAutoIdSort = $val;

		return $this;
	}

	public function disableIdGroup(bool $val = true): self
	{
		$this->disableIdGroup = $val;

		return $this;
	}

	protected function doCreateQuery(EntityRepository $repository): QueryBuilder
	{
		$qb = $this->createBasicDql($repository);

		foreach ($this->select as $modifier) {
			$modifier($qb);
		}

		if (!$this->disableAutoIdSort) {
			$qb->addOrderBy('p.id', 'DESC');
		}

		return $qb;
	}

	/**
	 * @param EntityRepository $repository
	 */
	protected function doCreateCountQuery($repository): QueryBuilder
	{
		return $this->createBasicDql($repository)->select('count(p.id)');
	}

	/**
	 * @param EntityRepository $repository
	 */
	private function createBasicDql($repository): QueryBuilder
	{
		/** @var QueryBuilder $qb */
		$qb = $repository->createQueryBuilder('p', 'p.id')
			->addSelect('IDENTITY(p.gallery) as gallery')
			->addSelect('IDENTITY(p.manufacturer) as manufacturer')
			->addSelect('IDENTITY(p.availabilityAfterSoldOut) as availabilityAfterSoldOut');

		if (!self::$ignoreProductPublish) {
			if (Config::load('product.publishedByLang')) {
				if (Arrays::contains($qb->getAllAliases(), 'pt')) {
					$qb->andWhere('pt.isPublished = 1');
				} else {
					$qb->innerJoin('p.productTexts', 'pt', Join::WITH, 'pt.lang = :ptLang AND pt.isPublished = 1')
						->setParameter('ptLang', $this->lang);
				}
			} else {
				$qb = $qb->andWhere('p.isPublished = 1');
			}
		}

		$qb->andWhere('p.isDeleted = 0');

		foreach ($this->filter as $modifier) {
			$modifier($qb);
		}

		return $qb;
	}

	public function getQueryBuilder(EntityRepository $repository): QueryBuilder
	{
		return $this->doCreateQuery($repository);
	}

}
