<?php declare(strict_types = 1);

namespace EshopCatalog\Model;

use Core\Model\Event\Event;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\BaseFrontEntityService;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\PackageItem;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductSupplier;

class AvailabilityService extends BaseFrontEntityService
{
	protected              $entityClass   = Availability::class;
	protected ?array       $cAvs          = null;
	protected static array $updated       = [];
	protected ?array       $cAllSuppliers = null;

	public static bool $allowCheckCountInCategories = true;

	protected EventDispatcher $eventDispatcher;

	public function __construct(
		EventDispatcher $eventDispatcher
	)
	{
		$this->eventDispatcher = $eventDispatcher;
	}

	public function getReference(int $id): Availability
	{
		/** @var Availability $av */
		$av = $this->em->getReference(Availability::class, $id);

		return $av;
	}

	/**
	 * Kontrola stavu produktu pokud je vyprodán nebo skladem
	 */
	public function updateAvailabilityByQuantity(Product $product): void
	{
		if (!Config::wareHouseExist() && !$product->getSuppliers()->toArray()) {
			return;
		}

		// Pokud je produkt rezervovan, neresime prepnuti podle poctu ks
		// TODO mozna pridat informaci do nastaveni dostupnosti?
		if ($product->getAvailability() && $product->getAvailability()->getIdent() === Availability::RESERVED) {
			return;
		}

		$allSuppliers  = $this->getAllSuppliers();
		$avs           = $this->getAvailability();
		$allowExternal = (int) $this->settings->get('eshopCatalogDisableExternalStorage', 0) === 0;

		if (Config::load('product.allowPackage', false) && $product->package) {
			$packageQuantity     = null;
			$packageAvailability = $avs[Availability::SOLD_OUT];

			foreach ($product->package->items->toArray() as $item) {
				$itemProduct = $item->getProduct();

				if ($packageQuantity === null || $packageQuantity > $itemProduct->quantity) {
					if ($itemProduct->unlimitedQuantity) {
						$packageQuantity     = 9999;
						$packageAvailability = $avs[Availability::IN_STOCK];
					} else if ($item->quantity <= $itemProduct->quantity && $itemProduct->getAvailability()->getIdent() !== Availability::SOLD_OUT) {
						$packageQuantity     = $itemProduct->quantity;
						$packageAvailability = $itemProduct->getAvailability()->getId();
					} else {
						$packageQuantity     = 0;
						$packageAvailability = $avs[Availability::SOLD_OUT];
					}
				}
			}

			$product->quantity = (int) $packageQuantity;

			if (!$product->getAvailability() || $product->getAvailability()->getId() !== $packageAvailability) {
				$product->setAvailability($this->getReference($packageAvailability));

				self::$updated[$product->getId()] = $product->getId();
			}
		} else if ($product->unlimitedQuantity) {
			if (
				$product->getAvailability()->getIdent() !== Availability::PREORDER
				&& $product->getAvailability()->getIdent() !== Availability::IN_STOCK
			) {
				$product->setAvailability($this->getReference($avs[Availability::IN_STOCK]));
				self::$updated[$product->getId()] = $product->getId();
			}
		} else if ($product->quantity <= 0) {
			$supplierPos = null;
			$supplierAv  = null;

			if ($allowExternal) {
				foreach ($product->getSuppliers()->toArray() as $supp) {
					/** @var ProductSupplier $supp */
					$supplier = $allSuppliers[$supp->getSupplier()->getId()] ?? null;
					if (!$supp->isActive || !$supplier || !$supplier['allowSale']) {
						continue;
					}

					if ($supp->availabilityAfterSoldOut && $supp->quantity > 0 && ($supplierPos === null || $supplierPos > $supp->availabilityAfterSoldOut->getPosition())) {
						$supplierAv  = $supp->availabilityAfterSoldOut;
						$supplierPos = $supp->availabilityAfterSoldOut->getPosition();
					}
				}
			}

			if (!$supplierAv && $product->availabilityAfterSoldOut) {
				$supplierAv = $product->availabilityAfterSoldOut;
			}

			if ($supplierAv && (!$product->getAvailability() || $product->getAvailability()->getId() !== $supplierAv->getId())) {
				$product->setAvailability($supplierAv);
				self::$updated[$product->getId()] = $product->getId();
			} else if (!$supplierAv && (!$product->getAvailability() || $product->getAvailability()->getId() !== $avs[Availability::SOLD_OUT])) {
				$product->setAvailability($this->getReference($avs[Availability::SOLD_OUT]));
				self::$updated[$product->getId()] = $product->getId();
			}
		} else if (!in_array($product->getAvailability()->getIdent(), [
			Availability::IN_STOCK,
			Availability::PREORDER,
		], true)) {
			if (isset($avs[Availability::PREORDER]) && $product->getMoreDataValue('lastStatusPreorder', null)) {
				$product->setAvailability($this->getReference($avs[Availability::PREORDER]));
				$product->removeMoreData('lastStatusPreorder');
			} else {
				$product->setAvailability($this->getReference($avs[Availability::IN_STOCK]));
			}

			self::$updated[$product->getId()] = $product->getId();
		}

		// Aktualizace balíčku pokud ho produkt obsahuje
		if (Config::load('product.allowPackage', false)) {
			$packages = [];
			foreach ($this->em->getRepository(PackageItem::class)->createQueryBuilder('pi')
				         ->select('IDENTITY(pi.package) as package')
				         ->where('pi.product = :id')
				         ->setParameter('id', $product->getId())
				         ->getQuery()->getArrayResult() as $row) {
				$packages[] = $row['package'];
			}

			foreach (array_chunk($packages, 200) as $chunk) {
				foreach ($this->em->getRepository(Product::class)->createQueryBuilder('p')
					         ->where('p.package IN (' . implode(',', $chunk) . ')')
					         ->getQuery()->getResult() as $row) {
					/** @var Product $row */
					$this->updateAvailabilityByQuantity($row);
				}
			}
		}

		if (self::$allowCheckCountInCategories) {
			$this->eventDispatcher->dispatch(new Event(['products' => array_values(self::$updated)]), 'eshopCatalog.category.checkProductsCount');
		}
	}

	public function updateProductsWithoutSupplier(): void
	{
		if (!Config::wareHouseExist()) {
			return;
		}

		$avs     = $this->getAvailability();
		$inStock = $avs[Availability::IN_STOCK];
		$soldOut = $avs[Availability::SOLD_OUT];

		$isIn   = [];
		$isOut  = [];
		$others = [];

		foreach ($this->em->getRepository(Product::class)->createQueryBuilder('p')
			         ->select('p.id, p.quantity, IDENTITY(p.availability) as av, IDENTITY(p.availabilityAfterSoldOut) as avAfterSoldOut, p.unlimitedQuantity')
			         ->andWhere('p.isDeleted = 0')
			         ->leftJoin('p.suppliers', 'ps')
			         ->andWhere('ps.product IS NULL')
			         ->getQuery()->getResult() as $row) {
			if (isset(self::$updated[$row['id']])) {
				continue;
			}

			if ($row['unlimitedQuantity']) {
				if ($row['av'] == $soldOut) {
					$others[$row['avAfterSoldOut']][] = $row['id'];
				}
			}
			if ($row['avAfterSoldOut'] && $row['quantity'] <= 0) {
				$others[$row['avAfterSoldOut']][] = $row['id'];
			} else if ($row['quantity'] > 0 && $row['av'] == $soldOut) {
				$isIn[] = $row['id'];
			} else if ($row['quantity'] <= 0) {
				$isOut[] = $row['id'];
			}
		}

		foreach (array_chunk($isIn, 100) as $chunk) {
			$this->em->getConnection()
				->executeStatement("UPDATE eshop_catalog__product SET id_availability = {$inStock} WHERE id IN (" . implode(',', $chunk) . ")");

			if (self::$allowCheckCountInCategories) {
				$this->eventDispatcher->dispatch(new Event(['products' => array_values($chunk)]), 'eshopCatalog.category.checkProductsCount');
			}
		}

		foreach (array_chunk($isOut, 100) as $chunk) {
			$this->em->getConnection()
				->executeStatement("UPDATE eshop_catalog__product SET id_availability = {$soldOut} WHERE id IN (" . implode(',', $chunk) . ")");

			if (self::$allowCheckCountInCategories) {
				$this->eventDispatcher->dispatch(new Event(['products' => array_values($chunk)]), 'eshopCatalog.category.checkProductsCount');
			}
		}

		foreach ($others as $avId => $ids) {
			foreach (array_chunk($ids, 100) as $chunk) {
				$this->em->getConnection()
					->executeStatement("UPDATE eshop_catalog__product SET id_availability = {$avId} WHERE id IN (" . implode(',', $chunk) . ")");

				if (self::$allowCheckCountInCategories) {
					$this->eventDispatcher->dispatch(new Event(['products' => array_values($chunk)]), 'eshopCatalog.category.checkProductsCount');
				}
			}
		}
	}

	/**
	 * @return array<string, int>
	 */
	protected function getAvailability(): array
	{
		if ($this->cAvs === null) {
			$this->cAvs = [];

			foreach ($this->getEr()->createQueryBuilder('a')->select('a.id, a.ident')
				         ->getQuery()->getArrayResult() as $row) {
				$this->cAvs[$row['ident']] = (int) $row['id'];
			}
		}

		return $this->cAvs;
	}

	/**
	 * @return array<int, array{
	 *     name: string,
	 *     allowSale: int,
	 * }>
	 */
	protected function getAllSuppliers(): array
	{
		if ($this->cAllSuppliers === null) {
			$this->cAllSuppliers = [];

			foreach ($this->em->getConnection()->fetchAllAssociative("SELECT s.id, s.allow_sale, s.name 
					FROM eshop_catalog__supplier s ") as $row) {
				$this->cAllSuppliers[(int) $row['id']] = [
					'name'      => $row['name'],
					'allowSale' => (int) $row['allow_sale'],
				];
			}
		}

		return $this->cAllSuppliers;
	}
}
