<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Contributte\Translation\Translator;
use Core\Model\Event\EntityRemoveEvent;
use Core\Model\Event\EventDispatcher;
use Doctrine;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Mapping as ORM;
use EshopCatalog\FrontModule\Model\CacheService;
use Nette\SmartObject;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class FeatureValueListener implements EventSubscriberInterface
{
	use SmartObject;

	protected CacheService    $cacheService;
	protected Translator      $translator;
	protected EventDispatcher $eventDispatcher;

	protected ?int $objectId = null;

	public function __construct(
		CacheService    $cacheService,
		Translator      $translator,
		EventDispatcher $eventDispatcher
	)
	{
		$this->cacheService    = $cacheService;
		$this->translator      = $translator;
		$this->eventDispatcher = $eventDispatcher;
	}

	public static function getSubscribedEvents(): array
	{
		return [];
	}

	/**
	 * @ORM\PostUpdate
	 *
	 * @param FeatureValue|FeatureValueTexts $entity
	 */
	public function onPostUpdate($entity, LifecycleEventArgs $event): void
	{
		$featureIds = [];

		if ($entity instanceof FeatureValue && $entity->getFeature() && $entity->getFeature()->getId() !== null) {
			$featureIds[] = $entity->getFeature()->getId();
		}

		$this->cacheService->clearFeatures($featureIds);

		$this->cacheService->clearFeaturesValues();
	}

	/**
	 * @ORM\PreRemove
	 *
	 * @param FeatureValue|FeatureValueTexts $entity
	 */
	public function preRemoveHandler($entity, LifecycleEventArgs $event): void
	{
		if ($entity instanceof FeatureValue) {
			$this->objectId = $entity->getId();
			$this->eventDispatcher->dispatch(new EntityRemoveEvent((string) $this->objectId, $entity), FeatureValue::class . '::preRemove');
		}
	}

	/**
	 * @ORM\PostRemove
	 *
	 * @param FeatureValue|FeatureValueTexts $entity
	 */
	public function postRemoveHandler($entity, LifecycleEventArgs $event): void
	{
		if ($entity instanceof FeatureValue) {
			$this->eventDispatcher->dispatch(new EntityRemoveEvent((string) $this->objectId, $entity), FeatureValue::class . '::remove');
		}
	}
}
