<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Contributte\Translation\Translator;
use Core\Model\Event\EntityRemoveEvent;
use Core\Model\Event\EventDispatcher;
use Doctrine;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Mapping as ORM;
use EshopCatalog\FrontModule\Model\CacheService;
use Nette\SmartObject;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class ManufacturerListener implements EventSubscriberInterface
{
	use SmartObject;

	protected CacheService    $cacheService;
	protected Translator      $translator;
	protected EventDispatcher $eventDispatcher;

	protected ?int $objectId = null;

	public function __construct(
		CacheService    $cacheService,
		Translator      $translator,
		EventDispatcher $eventDispatcher
	)
	{
		$this->cacheService    = $cacheService;
		$this->translator      = $translator;
		$this->eventDispatcher = $eventDispatcher;
	}

	public static function getSubscribedEvents(): array
	{
		return [];
	}

	/**
	 * @ORM\PreFlush
	 *
	 * @param Manufacturer|ManufacturerTexts $entity
	 */
	public function onPreFlush($entity, Doctrine\ORM\Event\PreFlushEventArgs $args): void
	{
		$this->cacheService->clearManufacturersCache();
	}

	/**
	 * @ORM\PostUpdate
	 *
	 * @param Manufacturer|ManufacturerTexts $entity
	 */
	public function preUpdate($entity, LifecycleEventArgs $args): void
	{
		bdump('post update');
		bdump($entity);
		$this->cacheService->clearManufacturersCache();
	}

	/**
	 * @ORM\PreRemove
	 *
	 * @param Manufacturer|ManufacturerTexts $manufacturer
	 */
	public function preRemoveHandler($manufacturer, LifecycleEventArgs $event): void
	{
		if ($manufacturer instanceof Manufacturer) {
			$this->objectId = $manufacturer->getId();
			$this->eventDispatcher->dispatch(new EntityRemoveEvent((string) $this->objectId, $manufacturer), Manufacturer::class . '::preRemove');
		}
	}

	/**
	 * @ORM\PostRemove
	 *
	 * @param Manufacturer|ManufacturerTexts $manufacturer
	 */
	public function postRemoveHandler($manufacturer, LifecycleEventArgs $event): void
	{
		$this->cacheService->clearManufacturersCache();

		if ($manufacturer instanceof Manufacturer) {
			$this->eventDispatcher->dispatch(new EntityRemoveEvent((string) $this->objectId, $manufacturer), Manufacturer::class . '::remove');
		}
	}
}
