<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Personalization;

use Core\Model\Sites;
use EshopCatalog\Model\Config;
use Nette\Caching\Cache;
use Nette\Http\IRequest;
use Nette\Http\IResponse;

class PersonalizationAccessControl
{
	public const SERVICE_ADVICIO = 'advicio';
	public const SERVICE_PERSOO = 'persoo';

	protected Sites $sites;
	protected Cache $cache;
	protected IResponse $response;
	protected IRequest $request;

	public function __construct(Sites $sites, IResponse $response, IRequest $request)
	{
		$this->sites = $sites;
		$this->response = $response;
		$this->request = $request;
	}

	protected function isIpAllowed(string $siteIdent): bool
	{
		$currentIp = $_SERVER['REMOTE_ADDR'];
		$ipRestrictBySites = Config::load('personalization.ipRestrictBySites');
		$ipRestrict = Config::load('personalization.ipRestrict');

		if ($ipRestrict) {
			return in_array($currentIp, $ipRestrict, true);
		}

		if ($ipRestrictBySites && array_key_exists($siteIdent, $ipRestrictBySites)) {
			return in_array($currentIp, $ipRestrictBySites[$siteIdent], true);
		}

		return true;
	}

	protected function isAllowed(bool $isEnable): bool
	{
		$site = $this->sites->getCurrentSite();

		$siteIdent = $site->getIdent();
		$ipAllowed = $this->isIpAllowed($siteIdent);

		return $ipAllowed && $isEnable;
	}

	public function isAutocompleteAllowed(): bool
	{
		return $this->isOn() && $this->isAllowed((bool) Config::load('personalization.autocompleteForm.enable'));
	}

	public function isAlternativeProductsAllowed(): bool
	{
		return $this->isOn() && $this->isAllowed((bool) Config::load('personalization.alternativeProducts.enable'));
	}

	public function isCrossSellProductsAllowed(): bool
	{
		return $this->isOn() && $this->isAllowed((bool) Config::load('personalization.crossSellProducts.enable'));
	}

	public function isBestSellingProductsAllowed(): bool
	{
		return $this->isOn() && $this->isAllowed((bool) Config::load('personalization.bestSellingProducts.enable'));
	}

	public function isRecentlyViewedProductsAllowed(): bool
	{
		return $this->isOn() && $this->isAllowed((bool) Config::load('personalization.recentlyViewedProducts.enable'));
	}

	public function isProductListingFilterAllowed(): bool
	{
		return $this->isOn() && $this->isAllowed((bool) Config::load('personalization.productListingFilter.enable'));
	}

	public function isSwitcherAllowed(): bool
	{
		if (!Config::load('personalization.persooAdvicioSwitcher.enable')) {
			return false;
		}

		$site = $this->sites->getCurrentSite();

		$currentIp = $_SERVER['REMOTE_ADDR'];
		$ipRestrict = Config::load('personalization.persooAdvicioSwitcher.ipRestrict');

		$siteIdent = $site->getIdent();

		$sites = Config::load('personalization.persooAdvicioSwitcher.sites');
		$sites = is_array($sites) ? $sites : [];

		return ($sites === [] || in_array($siteIdent, $sites, true)) && ($ipRestrict === [] || in_array($currentIp, $ipRestrict, true)) && $this->isIpAllowed($siteIdent);
	}

	/**
	 * Bez ohlednu na ip restrict
	 */
	public function isPutOrderAllowed(): bool
	{
		$site = $this->sites->getCurrentSite();

		$isEnableCrossSellProducts = (bool) Config::load('personalization.crossSellProducts.enable');
		$sitesWithCrossSellProducts = Config::load('personalization.crossSellProducts.sites');
		$sitesWithCrossSellProducts = is_array($sitesWithCrossSellProducts) ? $sitesWithCrossSellProducts : [];
		$isCrossSellProductsAllowed = $isEnableCrossSellProducts && ($sitesWithCrossSellProducts === [] || in_array($site->getIdent(), $sitesWithCrossSellProducts, true));

		$isEnableBestSellingProducts = (bool) Config::load('personalization.bestSellingProducts.enable');
		$sitesWithBestSellingProducts = Config::load('personalization.bestSellingProducts.sites');
		$sitesWithBestSellingProducts = is_array($sitesWithBestSellingProducts) ? $sitesWithBestSellingProducts : [];
		$isBestSellingProductsAllowed = $isEnableBestSellingProducts && ($sitesWithBestSellingProducts === [] || in_array($site->getIdent(), $sitesWithBestSellingProducts, true));

		return $isCrossSellProductsAllowed || $isBestSellingProductsAllowed;
	}

	public function isOn(): bool
	{
		$isForcePersoo = $this->request->getCookie('personalization-persoo-advicio-switcher') === self::SERVICE_PERSOO;
		return !$isForcePersoo && $this->isAllowed(true);
	}

	public function switchService(): void
	{
		$this->response->setCookie('personalization-persoo-advicio-switcher', $this->isOn() ? self::SERVICE_PERSOO : self::SERVICE_ADVICIO, '1 week');
	}

	public function hasAutocompleteRequestAutocancelAllowed(): bool
	{
		$site = $this->sites->getCurrentSite();

		return Config::load('personalization.autocompleteForm.autocancelRequest') || in_array($site->getIdent(), Config::load('personalization.autocompleteForm.autocancelRequestForSites'), true);
	}

}
