<?php declare(strict_types = 1);

namespace EshopCatalog\Model\TemplateTextType;

use Contributte\Translation\Translator;
use Core\Model\Helpers\Strings;
use Core\Model\TemplateReader\Providers\ITemplateTextType;
use Core\Model\TemplateReader\Providers\TemplateTextType;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopOrders\FrontModule\Model\Customers;
use Nette\Utils\Validators;
use Users\Model\Security\User;

class CategoryWithProducts extends TemplateTextType implements ITemplateTextType
{
	/** @var string */
	protected $type = 'eshopCatalogCategoryWithProducts';

	protected Products       $productsService;
	protected ProductsFacade $productsFacade;
	protected Categories     $categoriesService;
	protected Translator     $translator;
	protected Customers      $customers;
	protected User           $user;

	public function __construct(
		Products       $products,
		ProductsFacade $productsFacade,
		Categories     $categories,
		Translator     $translator,
		Customers      $customers,
		User           $user
	)
	{
		$this->productsService   = $products;
		$this->productsFacade    = $productsFacade;
		$this->categoriesService = $categories;
		$this->translator        = $translator;
		$this->customers         = $customers;
		$this->user              = $user;
	}

	/**
	 * @inheritDoc
	 */
	public function loadContainer($formContainer, $params = [], $texts = [])
	{
		if (!isset($params['categoryId']) || !isset($params['limit'])) {
			$container = $formContainer->addContainer($params['name']);
		} else {
			return;
		}

		if (!isset($params['categoryId'])) {
			$opts = ['' => ''];
			foreach ($this->categoriesService->getCategories() as $cat) {
				$prefix = '';
				for ($i = 1; $i < $cat->lvl; $i++) {
					$prefix .= '---';
				}
				$opts[$cat->id] = trim($prefix . ' ' . $cat->name);
			}
			$container->addSelect('categoryId', 'eshopCatalog.templateTextFields.categoryId', $opts);
		}

		if (!isset($params['limit'])) {
			$container->addText('limit', 'eshopCatalog.templateTextFields.limit')
				->setHtmlType('number')
				->setValue(6);
		}
	}

	public function render(array $params): array
	{
		$default              = $this->getDefault();
		$params['limit']      = $params['limit'] ?: $default['limit'] ?? null;

		if (Validators::isNone($params['categoryId'])) {
			$params['categoryId'] = !Validators::isNone($default['categoryId']) ? ((int) $default['categoryId']) : [];
		} elseif (Strings::contains($params['categoryId'], ',')) {
			$params['categoryId'] = array_map('intval', explode(',', $params['categoryId']));
		} else {
			$params['categoryId'] = [(int) $params['categoryId']];
		}

		if (!isset($params['toVar'])) {
			bdump('Need variable for output');

			return [];
		}
		if (!$params['categoryId']) {
			bdump('Need categoryId for output');

			return [];
		}

		$customer = $this->user->isLoggedIn()
			/** @phpstan-ignore-next-line */
			? $this->customers->getByUser($this->user->getIdentity())
			: null;

		$categories = [];
		$cats = $this->categoriesService->getCategories();
		foreach (is_array($params['categoryId']) ? $params['categoryId'] : [] as $catId) {
			if (!array_key_exists($catId, $cats)) {
				continue;
			}

			$category = $cats[$catId];

			if (!$this->categoriesService->checkCategoryRestrictionAccess($category->allowedCustomerGroups, $customer)) {
				continue;
			}

			$categories[$category->getId()] = $category;
		}

		if (!$categories) {
			return [];
		}

		$productIds = $this->productsService->getProductsIdInCategory(array_keys($categories), 0, isset($params['limit']) ? ((int) $params['limit']) : null);
		if (count($categories) === 1) {
			return [
				'category' => $categories[array_key_first($categories)],
				'products' => $productIds ? $this->productsFacade->getProducts($productIds) : [],
			];
		}

		$data = [];
		foreach ($this->productsFacade->getProducts($productIds) as $product) {
			if ($product->defaultCategoryId && array_key_exists($product->defaultCategoryId, $categories)) {
				$data[$product->defaultCategoryId]['category'] = $categories[$product->defaultCategoryId];
				$data[$product->defaultCategoryId]['products'][$product->getId()] = $product;
			}
		}

		$return = [];
		foreach ($categories as $category) {
			if (isset($data[$category->id])) {
				$return[] = $data[$category->id];
			}
		}

		return $return;
	}

}
