<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\FrontModule\Model\Dao;
use Nette\Caching\Cache;
use Nette\Caching\Storage;

class AvailabilityService extends BaseFrontEntityService
{
	/** @var string */
	protected $entityClass = Availability::class;

	const CACHE_NAMESPACE = 'eshopCatalogAvailability';

	/** @var Cache */
	protected $cache;

	protected ?array $cAll = null;

	protected array $cacheDep = [
		Cache::EXPIRE => '1 week',
	];

	public function __construct(Storage $storage)
	{
		$this->cache = new Cache($storage, self::CACHE_NAMESPACE);
	}

	/**
	 * @return Dao\Availability[]
	 * @throws \Throwable
	 */
	public function getAll(): array
	{
		if ($this->cAll === null) {
			$this->cAll = $this->cache->load('availabilities/' . $this->translator->getLocale(), function(&$dep) {
				$dep = $this->cacheDep;

				$arr = [];
				$qb  = $this->getEr()->createQueryBuilder('a')
					->select('a.id, a.ident, at.name, at.description, a.delay, a.color as textColor, a.bgColor, a.canShowOnList, a.canShowOnSearch, a.canAddToCart, a.openGraphText')
					->innerJoin('a.texts', 'at', 'WITH', 'at.lang = :lang')
					->setParameter('lang', $this->translator->getLocale());

				if (Config::load('allowWarehouseOverdraft')) {
					$qb->addSelect('a.warehouseOverdraft');
				}

				foreach ($qb->getQuery()->getArrayResult() as $row) {
					$arr[$row['id']] = $this->fillDao($row);
				}

				return $arr;
			}) ?: [];
		}

		return $this->cAll;
	}

	public function getByIdent(string $ident): ?Dao\Availability
	{
		foreach ($this->getAll() as $item)
			if ($item->getIdent() === $ident)
				return $item;

		return null;
	}

	public function get(int $id): ?Dao\Availability
	{
		return $this->getAll()[$id] ?? null;
	}

	public function fillDao(array $data): Dao\Availability
	{
		$dao = new Dao\Availability();
		$dao->setData($data);

		return $dao;
	}
}
