<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Doctrine;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Mapping as ORM;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Config;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Navigations\Model\Navigations;
use Nette\Localization\ITranslator;
use Nette\Caching\Cache;
use Nette\SmartObject;
use Tracy\Debugger;
use Users\Model\Security\User;

class CategoryTextListener implements EventSubscriberInterface
{
	use SmartObject;

	/** @var array */
	protected static $cleared = [];

	/** @var CacheService */
	protected $cacheService;

	/** @var ITranslator */
	protected $translator;

	protected ?User $user = null;

	public function __construct(ITranslator $translator, CacheService $cacheService, User $user)
	{
		$this->cacheService = $cacheService;
		$this->translator   = $translator;
		if ($user->isLoggedIn()) {
			$this->user = $user;
		}
	}

	/**
	 * Zkontrolované cache, takto OK
	 * 
	 * @ORM\PostUpdate
	 *
	 * @param CategoryTexts      $categoryText
	 * @param LifecycleEventArgs $event
	 *
	 * @throws Doctrine\ORM\ORMException
	 */
	public function postUpdateHandler(CategoryTexts $categoryText, LifecycleEventArgs $event)
	{
		if (!self::$cleared[$categoryText->getCategory()->getId()]) {
			$em        = $event->getEntityManager();
			$changeSet = $em->getUnitOfWork()->getEntityChangeSet($categoryText);

			if (isset($changeSet['alias'])) {
				$this->cacheService->clean('navigation', [
					Cache::TAGS => [Navigations::CACHE_NAMESPACE],
				]);
			}
		}

		if (Config::load('allowLogCategoryShortDescChanges', false)) {
			$changeSet = $event->getEntityManager()->getUnitOfWork()->getEntityChangeSet($categoryText);

			if (isset($changeSet['shortDescription'])) {
				try {
					throw new \Exception();
				} catch (\Exception $e) {
					Debugger::log($e, 'categoryShortDescChanges');
					Debugger::log([
						'entityId' => $categoryText->id->getId(),
						'changeSet' => $changeSet['shortDescription'],
					], 'categoryShortDescChanges-moreinfo');
				}
			}
		}

		self::$cleared[$categoryText->getCategory()->getId()] = true;
	}

	public static function getSubscribedEvents(): array
	{
		return [];
	}

	/**
	 * @ORM\PreFlush
	 *
	 * @param CategoryTexts      $categoryText
	 * @param LifecycleEventArgs $args
	 */
	public function onPreFlush($categoryText, $args)
	{
	}
}
