<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Helpers\Strings;
use Doctrine\ORM\Mapping as ORM;
use Core\Model\Entities\TId;
use Nette\Utils\DateTime;

/**
 * @ORM\Table(name="eshop_catalog__product_price_history", indexes={
 *     @ORM\Index(name="byCountry", columns={"direction", "product"}),
 * })
 * @ORM\Entity
 * @ORM\EntityListeners({"EshopCatalog\Model\Listeners\ProductPriceHistoryListener"})
 */
class ProductPriceHistory
{
	use TId;

	/**
	 * @ORM\ManyToOne(targetEntity="Product")
	 * @ORM\JoinColumn(name="product", referencedColumnName="id", onDelete="CASCADE")
	 */
	public Product $product;

	/**
	 * @var double
	 * @ORM\Column(name="price", type="decimal", precision=10, scale=2, nullable=false)
	 */
	public $price;

	/**
	 * @ORM\Column(name="created", type="datetime", nullable=false, options={"default": "CURRENT_TIMESTAMP"})
	 */
	protected \DateTimeInterface $created;

	/**
	 * @ORM\Column(name="direction", type="smallint", nullable=false, options={"default": 0})
	 */
	public int $direction = 0;

	/**
	 * @ORM\Column(name="reason", type="string", length=255, nullable=true)
	 */
	public ?string $reason = null;

	public function __construct(
		Product $product,
		?float  $price,
		float   $newPrice,
		?string $reason = null
	)
	{
		if (!$price) {
			$price = 0.0;
		}

		$this->product   = $product;
		$this->price     = Strings::formatEntityDecimal($price);
		$this->created   = new DateTime();
		$this->direction = $newPrice > $price ? 1 : 0;
		$this->reason    = $reason;
	}

	public function getCreated(): \DateTimeInterface { return $this->created; }
}
