<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Categories;

use Core\Components\Flashes\Flashes;
use Core\Model\Helpers\Arrays;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Category;
use EshopCatalog\Model\Entities\CategoryProduct;
use EshopCatalog\Model\Entities\CategoryTexts;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductInSite;
use Nette\Application\UI\Multiplier;
use Nette\Caching\Cache;
use Nette\Utils\Html;

class CategoriesGrid extends BaseControl
{
	protected ?array $cData                         = null;
	protected ?array $cDefaultCategoryProductsCount = null;
	protected ?array $cOtherCategoryProductsCount   = null;

	public function __construct(
		protected string               $siteIdent,
		protected Categories           $categoriesService,
		protected ICategoryFormFactory $categoryFormFactory,
		protected IRelatedFormFactory  $relatedFormFactory,
		protected CacheService         $cacheService,
	)
	{
	}

	public function render(): void
	{
		$this->template->categories = $this->getData()['tree'];
		$this->template->siteIdent  = $this->siteIdent;
		$this->template->render($this->getTemplateFile());
	}

	protected function canUpdateGrid(): bool
	{
		return $this->categoriesService->getSessionSection()->get('categoriesGridUpdateEnabled') ?? true;
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete(int $id): void
	{
		$presenter = $this->presenter;
		if ($this->categoriesService->removeCategory($id)) {
			$presenter->flashMessage('eshopCatalog.defaultGrid.removed', Flashes::FLASH_SUCCESS);
		} else {
			$presenter->flashMessage('eshopCatalog.defaultGrid.removeFailed', Flashes::FLASH_DANGER);
		}

		$this->cacheStorage->clean([Cache::Tags => ['eshopNavigation']]);
		$this->cacheService->categoryCache->clean(
			[Cache::Tags => [\EshopCatalog\FrontModule\Model\Categories::CACHE_NAMESPACE]],
		);

		if ($presenter->isAjax()) {
			$this->redrawControl('list');
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	public function handleEdit(int $id): void
	{
		$category = $this->categoriesService->get($id);

		if ($category) {
			$this['categoryForm']->categoryId = $id;
			$this->template->modalTitle       = $this->t(
					'eshopCatalog.title.editCategory',
				) . ' ' . $category->getCategoryText()
					->getName();
			$this->template->modal            = 'categoryForm';
			$this->redrawControl('modal');
		}
	}

	public function handleRelated(int $id): void
	{
		$category = $this->categoriesService->get($id);

		if ($category) {
			$this->template->modalTitle = $this->t(
					'eshopCatalog.title.editCategory',
				) . ' ' . $category->getCategoryText()
					->getName();
			$this->template->modal      = 'relatedForm';
			$this->template->id         = $id;
			$this->redrawControl('modal');
		}
	}

	public function handleChangePosition(): void
	{
		$presenter = $this->presenter;
		$request   = $presenter->getHttpRequest();

		$id   = (int) $request->getPost('id');
		$move = (int) $request->getPost('move');

		if ($id) {
			$er  = $this->categoriesService->getEr();
			$cat = $er->find($id);

			if ($cat) {
				if ($move < 0) {
					$er->moveUp($cat, abs($move));
				} else if ($move > 0) {
					$er->moveDown($cat, $move);
				}

				$presenter->flashMessageSuccess('default.positionChanged');
			}
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		$this->em->flush();
		if ($id) {
			$this->categoriesService->cleanCacheDeep($id);
		}

		$this->cacheStorage->clean([Cache::Tags => ['eshopNavigation']]);
		$this->cacheService->categoryCache->clean(
			[Cache::Tags => [\EshopCatalog\FrontModule\Model\Categories::CACHE_NAMESPACE]],
		);

		$presenter->redrawControl('flashes');
	}

	public function handleClearProductsNotDefaultNested(string $siteIdent, int $id): void
	{
		if (Config::load(
				'category.allowClearProductsNotDefault',
			) && $this->categoriesService->clearProductsNotDefaultCategoryNested($siteIdent, [$id])) {
			$this->presenter->flashMessageSuccess('default.saved');
		} else {
			$this->presenter->flashMessageDanger('default.error');
		}

		$this->presenter->redrawControl('flashes');
	}

	public function handleInvertPublish(int $id, ?string $lang = null): void
	{
		$presenter = $this->presenter;
		if (!$presenter) {
			return;
		}

		if ($this->categoriesService->invertPublish($id, $lang)) {
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.publishChanged');
		} else {
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.publishChangeFailed');
		}

		if ($this->canUpdateGrid()) {
			//			$this->redrawControl('publish' . $id);
			$this->redrawControl('list');
		}
		$presenter->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentCategoryForm(): CategoryForm
	{
		$control = $this->categoryFormFactory->create();
		if ($this->getParameter('id')) {
			$control->setCategory((int) $this->getParameter('id'));
		}

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			if ($this->canUpdateGrid()) {
				$this->redrawControl('list');
			}

			$this->presenter->redrawControl('flashes');
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			if ($this->canUpdateGrid()) {
				$this->redrawControl('list');
			}

			$this->presenter->payload->hideModal = true;
			$this->presenter->redrawControl('flashes');
		};
		$control['form']->getComponent('saveControl')->closeModalOnCancel();

		return $control;
	}

	protected function createComponentRelatedForm(): Multiplier
	{
		return new Multiplier(fn($id): RelatedForm => $this->relatedFormFactory->create((int) $id));
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function getData(): array
	{
		if ($this->cData !== null) {
			return $this->cData;
		}

		$rootId = $this->categoriesService->getEr()->createQueryBuilder('c')
			->select('c.id')
			->innerJoin(CategoryTexts::class, 'ct', Join::WITH, 'c.id = ct.id')
			->where('c.lvl = :lvl')
			->andWhere('ct.alias = :alias')
			->setParameters([
				'lvl'   => 0,
				'alias' => $this->siteIdent,
			])->getQuery()->setMaxResults(1)->getArrayResult()[0]['id'] ?? null;

		if (!$rootId) {
			$newCategory     = new Category;
			$newCategoryText = new CategoryTexts($newCategory, $this->translator->getLocale());
			$newCategoryText->setName($this->siteIdent);

			$this->em->persist($newCategory)
				->persist($newCategoryText)
				->flush();

			$rootId = $newCategory->getId();
		}

		// Kategorie
		$qb   = $this->categoriesService->getEr()->createQueryBuilder('c', 'c.id')
			->select(
				"c.id, c.isPublished, IDENTITY(c.parent) as parent, GROUP_CONCAT(ct.lang, ';', ct.name, ';', ct.isPublished SEPARATOR '|') as texts",
			)
			->leftJoin('c.categoryTexts', 'ct')
			->andWhere('c.root = :root')
			->setParameters([
				'root' => $rootId,
			])
			->groupBy('c.id')
			->addOrderBy('c.lft');
		$data = $qb->getQuery()->getArrayResult();

		foreach ($data as $k => &$row) {
			$texts = [];
			foreach (explode('|', (string) $row['texts']) as $v) {
				$tmp = explode(';', $v);

				$texts[$tmp[0]] = [
					'name'        => $tmp[1],
					'isPublished' => (int) $tmp[2],
				];
			}

			$row['texts'] = $texts;

			// Počet produktů podle výchozí kategorie
			$row['productsCount'] = count(
				$this->getProductsCountByDefaultCategory((int) $k) + $this->getProductsCountByOtherCategories((int) $k),
			);
		}

		$this->addPublishHtml($data);

		$this->cData = [
			'data' => $data,
			'tree' => Arrays::buildTree($data, 'parent', 'id', $rootId),
		];

		return $this->cData;
	}

	protected function getProductsCountByDefaultCategory(int $categoryId): array
	{
		if ($this->cDefaultCategoryProductsCount === null) {
			$this->cDefaultCategoryProductsCount = [];
			foreach ($this->em->getRepository(Product::class)->createQueryBuilder('p')
				         ->select('p.id, IDENTITY(p.idCategoryDefault) as cat')
				         ->andWhere('p.isDeleted = 0')
				         ->getQuery()->getScalarResult() as $row) {
				$this->cDefaultCategoryProductsCount[(int) $row['cat']][] = $row['id'];
			}

			foreach ($this->em->getRepository(ProductInSite::class)->createQueryBuilder('ps')
				         ->select('IDENTITY(ps.product) as id, IDENTITY(ps.category) as cat')
				         ->getQuery()->getScalarResult() as $row) {
				$this->cDefaultCategoryProductsCount[(int) $row['cat']][] = $row['id'];
			}
		}

		return $this->cDefaultCategoryProductsCount[$categoryId] ?: [];
	}

	protected function getProductsCountByOtherCategories(int $categoryId): array
	{
		if ($this->cOtherCategoryProductsCount === null) {
			$this->cOtherCategoryProductsCount = [];
			foreach ($this->em->getRepository(CategoryProduct::class)->createQueryBuilder('cp')
				         ->select('IDENTITY(cp.product) as id, IDENTITY(cp.category) as cat')
				         ->getQuery()->getScalarResult() as $row) {
				$this->cOtherCategoryProductsCount[(int) $row['cat']][] = $row['id'];
			}
		}

		return $this->cOtherCategoryProductsCount[$categoryId] ?: [];
	}

	protected function addPublishHtml(array &$data): void
	{
		foreach ($data as &$row) {
			$wrap = Html::el('div class=grid-langs');
			/** @var array<int, array{
			 *     class: string,
			 *     href: string,
			 *     text: ?string,
			 *     ico: ?Html,
			 * }> $links
			 */
			$links = [];

			if (Config::load('category.publishedByLang')) {
				foreach ($this->langsService->getLangs(false) as $lang) {
					$text = $row['texts'][$lang->getTag()];
					if ($text) {
						$links[] = [
							'class' => 'ajax btn btn-xs ' . ($text['isPublished'] ? 'btn-success' : 'btn-danger'),
							'href'  => $this->link('invertPublish!', [$row['id'], $lang->getTag()]),
							'text'  => $lang->getShortTitle(),
						];
					}
				}
			} else {
				$wrap->setAttribute('style', 'justify-content: center;');
				$text    = $row['texts'][$this->translator->getLocale()];
				$links[] = [
					'class' => 'ajax btn btn-xs ' . ($text['isPublished'] ? 'btn-success' : 'btn-danger'),
					'href'  => $this->link('invertPublish!', [$row['id']]),
					'ico'   => Html::el('i', [
						'class' => 'fa ' . ($text['isPublished'] ? 'fa-check' : 'fa-times'),
					]),
				];
			}

			foreach ($links as $link) {
				$el = Html::el('div', [
					'class' => 'grid-langs__lang',
					'style' => count($links) === 1 ? 'width: auto;' : '',
				]);

				$a = Html::el('a', [
					'class' => $link['class'],
					'href'  => $link['href'],
				]);

				if (isset($link['text'])) {
					$a->setText($link['text']);
				} else if (isset($link['ico'])) {
					$a->addHtml($link['ico']);
				}

				$el->addHtml($a);

				$wrap->addHtml($el);
			}

			$row['publishHtml'] = $wrap;
		}
	}

}
